package org.gcube.data.analysis.tabulardata.cube.tablemanagers;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang.RandomStringUtils;
import org.gcube.data.analysis.tabulardata.cube.data.DatabaseWrangler;
import org.gcube.data.analysis.tabulardata.cube.exceptions.NoSuchTableException;
import org.gcube.data.analysis.tabulardata.cube.exceptions.TableCreationException;
import org.gcube.data.analysis.tabulardata.cube.metadata.CubeMetadataWrangler;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.type.IdColumnType;
import org.gcube.data.analysis.tabulardata.model.datatype.IntegerType;
import org.gcube.data.analysis.tabulardata.model.idioms.ColumnHasName;
import org.gcube.data.analysis.tabulardata.model.metadata.table.TableMetadata;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.table.TableType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

public abstract class DefaultTableCreator implements TableCreator {

	private Logger log = LoggerFactory.getLogger(this.getClass());

	protected DatabaseWrangler dbWrangler;
	protected CubeMetadataWrangler mdWrangler;
	protected TableManager tableManager;

	protected List<Column> newColumns = Lists.newArrayList();

	// Following attributes are used for cloning
	protected Table tableToClone = null;
	protected Set<Column> columnsToRemove = Sets.newHashSet();
	protected boolean copyData = false;

	private TableType tableType;

	public DefaultTableCreator(DatabaseWrangler dbWrangler, CubeMetadataWrangler mdWrangler, TableManager tableManager,
			TableType tableType) {
		super();
		this.dbWrangler = dbWrangler;
		this.mdWrangler = mdWrangler;
		this.tableManager = tableManager;
		this.tableType = tableType;
	}

	@Override
	public TableCreator addColumn(Column column) {
		checkColumnToAdd(column);
		newColumns.add(column);
		return this;
	}

	@Override
	public TableCreator addColumns(Column... columns) {
		for (Column column : columns) {
			addColumn(column);
		}
		return this;
	}

	private void checkColumnToAdd(Column column) {
		if (!isAllowedColumn(column))
			throw new IllegalArgumentException("Invalid column type: " + column.getColumnType());
	}

	protected final boolean isAllowedColumn(Column column) {
		if (column.getColumnType().equals(new IdColumnType()))
			return false;
		return true;
	}

	@Override
	public TableCreator like(Table table, boolean copyData) {
		return like(table, copyData, new ArrayList<Column>(0));
	}

	@Override
	public TableCreator like(Table table, boolean copyData, List<Column> columnsToRemove) {
		tableToClone = table;
		this.copyData = copyData;
		removeColumns(columnsToRemove);
		return this;
	}

	private void removeColumns(List<Column> columnsToRemove) {
		if (tableToClone==null) return;
		this.columnsToRemove = Sets.newHashSet();
		for (Column columnToRemove : columnsToRemove) {
			removeColumn(columnToRemove);
		}
	}

	private void removeColumn(Column columnToRemove) {
		if (tableToClone.getColumns().contains(columnToRemove)) {
			this.columnsToRemove.add(columnToRemove);
		}
	}

	@Override
	public Table create() throws TableCreationException {

		checkConsistency();

		setColumnNames();

		String tableName = null;
		if (tableToClone != null) {
			// Handle cloning
			tableName = dbWrangler.cloneTable(tableToClone.getName(), copyData, false);

			for (Column column : columnsToRemove) {
				dbWrangler.removeColumn(tableName, column.getName());
			}

			for (Column column : newColumns) {
				dbWrangler.addColumn(tableName, column.getName(), column.getDataType());
			}

		} else {
			// Handle simple creation of empty table
			tableName = dbWrangler.createTable();
			for (Column column : newColumns) {
				dbWrangler.addColumn(tableName, column.getName(), column.getDataType());
			}
		}

		addIndexes(tableName, getAllColumnsExceptId());

		List<Column> columns = Lists.newArrayList(createIdColumn());
		columns.addAll(getAllColumnsExceptId());
		Table newTable = createBaseTable(tableName, columns);

		// Clone metadata
		if (tableToClone != null) {
			cloneMetadata(tableToClone, newTable);
		}

		// Register codelist on metadata DB
		return mdWrangler.save(newTable);
	}

	protected void checkConsistency() throws TableCreationException {
		try {
			checkColumnsRelationship();
		} catch (Exception e) {
			throw new TableCreationException(e.getMessage());
		}
	}

	private void checkColumnsRelationship() throws Exception {
		for (Column column : getAllColumnsExceptId()) {
			if (column.hasRelationship()) {
				try {
					Table targetTable = tableManager.get(column.getRelationship().getTargetTableId());
					targetTable.getColumnById(column.getRelationship().getTargetColumnId());
				} catch (NoSuchTableException e) {
					throw new Exception(String.format("Referenced Codelist with ID %1$s does not exists.", column
							.getRelationship().getTargetTableId()));
				}
			}
		}
	}

	protected void setColumnNames() {
		List<Column> columns = getAllColumnsExceptId();
		for (Column column : columns) {
			if (column.hasName()) {
				continue;
			}
			String eligibleName;
			do {
				eligibleName = ColumnNameGenerator.generateColumnName();
			} while (!Collections2.filter(columns, new ColumnHasName(eligibleName)).isEmpty());
			column.setName(eligibleName);
		}
	}

	protected Table createBaseTable(String tableName, List<Column> columns) {
		Table result = new Table(tableType);
		result.setColumns(columns);
		result.setName(tableName);
		return result;
	}

	protected void cloneMetadata(Table sourceTable, Table destTable) {
		for (TableMetadata m : sourceTable.getAllMetadata()) {
			if (m.isInheritable())
				destTable.setMetadata(m);
		}
	}

	protected List<Column> getAllColumnsExceptId() {
		List<Column> result = Lists.newArrayList();
		if (tableToClone != null) {
			Collection<Column> clonedColumns = Lists.newArrayList(tableToClone.getColumns());
			clonedColumns.removeAll(columnsToRemove);
			result.addAll(clonedColumns);
			result.removeAll(tableToClone.getColumnsByType(new IdColumnType()));
		}
		result.addAll(newColumns);
		return result;
	}

	protected abstract void addIndexes(String tableName, Collection<Column> columns);

	protected Column createIdColumn() {
		Column idColumn = new Column(new IntegerType(), new IdColumnType());
		idColumn.setName("id");
		return idColumn;
	}

	private static class ColumnNameGenerator {

		public static String generateColumnName() {
			return RandomStringUtils.random(6, true, false);
		}

	}

}
