package org.gcube.data.analysis.tabulardata.operation.csv.importer;

import java.util.List;

import javax.inject.Inject;
import javax.inject.Singleton;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.operation.factories.types.ImportWorkerFactory;
import org.gcube.data.analysis.tabulardata.operation.parameters.Cardinality;
import org.gcube.data.analysis.tabulardata.operation.parameters.Parameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.BooleanParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.RegexpStringParameter;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationDescriptor.OperationId;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.Worker;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.InvalidInvocationException;

import com.google.common.collect.Lists;

import static org.gcube.data.analysis.tabulardata.operation.csv.Constants.*;

@Singleton
public class CSVImportFactory extends ImportWorkerFactory {

	private static final OperationId OPERATION_ID = new OperationId(100);

	private static final List<Parameter> parameters = Lists.newArrayList();

	private static RegexpStringParameter urlParameter =  new RegexpStringParameter(URL, "Document URL",
			"URL that points to a location where the document can be downloaded.", Cardinality.ONE,
			".*");
	
	private static RegexpStringParameter separatorParameter= new RegexpStringParameter(SEPARATOR, "Separator", "Char separator as string", Cardinality.ONE, "^\\W$");
	
	private static RegexpStringParameter encodingParameter = new RegexpStringParameter(ENCODING, "Encoding", "Document Encoding", Cardinality.ONE, ".*");
	
	private static BooleanParameter booleanParameter = new BooleanParameter(HASHEADER, "Header", "Tells if the document has header or not",
			Cardinality.ONE);
	
	static {
		parameters.add(urlParameter);
		parameters.add(separatorParameter);
		parameters.add(encodingParameter);
		parameters.add(booleanParameter);
	}

	private CubeManager cubeManager;

	private DatabaseConnectionProvider connectionProvider;
	
	@Inject
	public CSVImportFactory(CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider) {
		if (cubeManager == null)
			throw new IllegalArgumentException("cubeManager cannot be null");
		if (connectionProvider == null)
			throw new IllegalArgumentException("connectionProvider cannot be null");
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
	}

	public Worker createWorker(OperationInvocation invocation) throws InvalidInvocationException {
		validateInvocation(invocation);
		return new CSVImport(invocation, cubeManager, connectionProvider);
	}

	@Override
	protected String getOperationName() {
		return "CSV Import";
	}

	@Override
	protected String getOperationDescription() {
		return "Create a new table with a CSV file";
	}

	@Override
	protected OperationId getOperationId() {
		return OPERATION_ID;
	}

	@Override
	protected List<Parameter> getParameters() {
		return parameters;
	}

	public void validateInvocation(OperationInvocation invocation) throws InvalidInvocationException{
		if (invocation==null) throw new InvalidInvocationException(invocation, "null invocation");
		if (!encodingParameter.validate((String)invocation.getParameterInstances().get(ENCODING))) throw new InvalidInvocationException(invocation, "encoding not valid");
		if (!separatorParameter.validate((String)invocation.getParameterInstances().get(SEPARATOR))) throw new InvalidInvocationException(invocation, "separator not valid");
		if (!urlParameter.validate((String)invocation.getParameterInstances().get(SEPARATOR))) throw new InvalidInvocationException(invocation, "url not valid");
	}
	
}
