package org.gcube.data.analysis.tabulardata.statistical;

import java.io.File;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.gcube.data.analysis.statisticalmanager.proxies.StatisticalManagerDataSpace;
import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.metadata.NoSuchMetadataException;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.type.IdColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.ValidationColumnType;
import org.gcube.data.analysis.tabulardata.model.metadata.table.ExportMetadata;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.csv.Constants;
import org.gcube.data.analysis.tabulardata.operation.csv.exporter.CSVExportFactory;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.ImmutableWorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.Worker;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerStatus;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerWrapper;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.InvalidInvocationException;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;
import org.gcube.dataanalysis.ecoengine.datatypes.enumtypes.TableTemplates;

public class ExportToStatisticalOperation extends Worker{

	
	private CSVExportFactory csvExportFactory;
	private CubeManager cubeManager;
	private StatisticalManagerDataSpace statisticalDataSpace;
	
	public ExportToStatisticalOperation(OperationInvocation sourceInvocation,
			CSVExportFactory csvExportFactory, CubeManager cubeManager,
			StatisticalManagerDataSpace statisticalDataSpace) {
		super(sourceInvocation);
		this.csvExportFactory = csvExportFactory;
		this.cubeManager = cubeManager;
		this.statisticalDataSpace = statisticalDataSpace;
	}

	
	private Table targetTable;
	private Table exportedTable;
	private String dataSpaceTableId;
	private String user;
	
	@Override
	protected WorkerResult execute() throws WorkerException {
		loadParameters();
		updateProgress(0.1f);
		exportCSV();
		updateProgress(0.5f);
		importIntoDataSpace();
		updateProgress(0.9f);
		return new ImmutableWorkerResult(exportedTable);
	}
	
	private void loadParameters(){
		Map<String, Object> params = getSourceInvocation()
				.getParameterInstances();
		user = (String) params.get(StatisticalOperationFactory.USER
				.getIdentifier());
		targetTable=cubeManager.getTable(getSourceInvocation().getTargetTableId());
	}
	
	
	private void exportCSV() throws WorkerException {
		// prepare parameters
		HashMap<String, Object> exportParams = new HashMap<String, Object>();
		List<String> columns = new ArrayList<String>();
		for (Column col : targetTable.getColumnsByType(IdColumnType.class,
				ValidationColumnType.class))
			columns.add(col.getName());
		exportParams.put(Constants.COLUMNS, columns);
		exportParams.put(Constants.ENCODING, Charset.defaultCharset().toString());
		exportParams.put(Constants.SEPARATOR, ",");
		WorkerWrapper wrapper = new WorkerWrapper(csvExportFactory);
		try {
			WorkerStatus status = wrapper.execute(targetTable.getId(), null,
					exportParams);
			if (status.equals(WorkerStatus.SUCCEDED)) {
				WorkerResult result = wrapper.getResult();
				exportedTable = result.getResultTable();
			} else
				throw new WorkerException(
						"Failed export to CSV, worker status was " + status);
		} catch (InvalidInvocationException e) {
			throw new WorkerException("Unable to export table as CSV", e);
		}
	}
	
	private void importIntoDataSpace() throws WorkerException {
		String template = "GENERIC";
		TableTemplates tableTemplate = null;
		for (TableTemplates t : TableTemplates.values())
			if (template.contentEquals(t.toString())) {
				tableTemplate = t;
				break;
			}
		boolean hasHeader = true;
		String delimiter = ",";
		try {
			File f = new File(exportedTable.getMetadata(ExportMetadata.class)
					.getUri());

			dataSpaceTableId = statisticalDataSpace.createTableFromCSV(f,
					hasHeader, delimiter, "",
					"TDM - " + exportedTable.getName(), tableTemplate,
					"Exported from TDM", user);
			exportedTable=cubeManager.modifyTableMeta(exportedTable.getId()).setTableMetadata(new ExportMetadata("Statistical dataspace",dataSpaceTableId,new Date(System.currentTimeMillis()))).create();
		} catch (NoSuchMetadataException e) {
			throw new WorkerException("Unable to locate exported CSV file", e);
		}
	}
}
