package org.gcube.data.analysis.tabulardata.statistical;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLStreamHandler;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.gcube.contentmanager.storageclient.model.protocol.smp.SMPURLConnection;
import org.gcube.data.analysis.statisticalmanager.proxies.StatisticalManagerDataSpace;
import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.metadata.NoSuchMetadataException;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.type.IdColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.ValidationColumnType;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.csv.exporter.CSVExportFactory;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerStatus;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerWrapper;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.InvalidInvocationException;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;
import org.gcube.data.analysis.tabulardata.operation.worker.results.resources.ImmutableURIResult;
import org.gcube.data.analysis.tabulardata.operation.worker.results.resources.ResourceDescriptorResult;
import org.gcube.data.analysis.tabulardata.operation.worker.types.ResourceCreatorWorker;
import org.gcube.dataanalysis.ecoengine.datatypes.enumtypes.TableTemplates;

public class ExportToStatisticalOperation extends ResourceCreatorWorker{

	
	private CSVExportFactory csvExportFactory;
	private CubeManager cubeManager;
	private StatisticalManagerDataSpace statisticalDataSpace;
	
	public ExportToStatisticalOperation(OperationInvocation sourceInvocation,
			CSVExportFactory csvExportFactory, CubeManager cubeManager,
			StatisticalManagerDataSpace statisticalDataSpace) {
		super(sourceInvocation);
		this.csvExportFactory = csvExportFactory;
		this.cubeManager = cubeManager;
		this.statisticalDataSpace = statisticalDataSpace;
	}

	
	private Table targetTable;
	private ResourceDescriptorResult exportedTable;
	private String dataSpaceTableId;
	private String user;
	
	@Override
	protected ResourceDescriptorResult execute() throws WorkerException {
		loadParameters();
		updateProgress(0.1f,"Creating csv");
		exportCSV();
		updateProgress(0.5f,"Saving to user's SM dataspace");
		importIntoDataSpace();
		updateProgress(0.9f,"Finalizing");
		try{
			return new ImmutableURIResult(new URI(Constants.STATISTICAL_URI_PREFIX+":"+dataSpaceTableId),"Statistical dataspace CSV version");
		}catch (URISyntaxException e) {
			throw new WorkerException("Statistical URI is invalid",e);
		}
	}
	
	private void loadParameters(){
		Map<String, Object> params = getSourceInvocation()
				.getParameterInstances();
		user = (String) params.get(StatisticalOperationFactory.USER
				.getIdentifier());
		targetTable=cubeManager.getTable(getSourceInvocation().getTargetTableId());
	}
	
	
	private void exportCSV() throws WorkerException {
		// prepare parameters
		HashMap<String, Object> exportParams = new HashMap<String, Object>();
		List<String> columns = new ArrayList<String>();
		for (Column col : targetTable.getColumnsExceptTypes(IdColumnType.class,
				ValidationColumnType.class))
			columns.add(col.getName());
		exportParams.put(org.gcube.data.analysis.tabulardata.operation.csv.Constants.COLUMNS, columns);
		exportParams.put(org.gcube.data.analysis.tabulardata.operation.csv.Constants.ENCODING, Charset.defaultCharset().toString());
		exportParams.put(org.gcube.data.analysis.tabulardata.operation.csv.Constants.SEPARATOR, ",");
		WorkerWrapper<ResourceCreatorWorker,ResourceDescriptorResult> wrapper = new WorkerWrapper<ResourceCreatorWorker,ResourceDescriptorResult>(csvExportFactory);
		try {
			WorkerStatus status = wrapper.execute(targetTable.getId(), null,
					exportParams);
			if (status.equals(WorkerStatus.SUCCEDED)) {
				exportedTable = wrapper.getResult();
			} else
				throw new WorkerException(
						"Failed export to CSV, worker status was " + status);
		} catch (InvalidInvocationException e) {
			throw new WorkerException("Unable to export table as CSV", e);
		}
	}
	
	private void importIntoDataSpace() throws WorkerException {
		String template = "GENERIC";
		TableTemplates tableTemplate = null;
		for (TableTemplates t : TableTemplates.values())
			if (template.contentEquals(t.toString())) {
				tableTemplate = t;
				break;
			}
		boolean hasHeader = true;
		String delimiter = ",";
		try {
			File f = getInputFile();

			dataSpaceTableId = statisticalDataSpace.createTableFromCSV(f,
					hasHeader, delimiter, "",
					"TDM - " + exportedTable.getDescription(), tableTemplate,
					"Exported from TDM", user);
		} catch (NoSuchMetadataException e) {
			throw new WorkerException("Unable to locate exported CSV file", e);
		} catch (Exception e) {
			throw new WorkerException("Unable to locate exported CSV file", e);
		} 
	}
	
	
	private File getInputFile() throws Exception{

		InputStream inputStream = new URL(null, exportedTable.getResource().getStringValue(), new URLStreamHandler() {

			@Override
			protected URLConnection openConnection(URL url) throws IOException {
				return new SMPURLConnection(url);
			}
		}).openStream();
		File tempFile = File.createTempFile("import", ".csv");

		OutputStream outputStream = new FileOutputStream(tempFile);

		int read = 0;
		byte[] bytes = new byte[1024];

		while ((read = inputStream.read(bytes)) != -1) 
			outputStream.write(bytes, 0, read);

		outputStream.close();
		inputStream.close();
		return tempFile;
	}
}
