package gr.cite.geoanalytics.dataaccess.entities.project.dao;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import javax.persistence.NoResultException;
import javax.persistence.Query;
import javax.persistence.TypedQuery;

import org.hibernate.Hibernate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Repository;

import gr.cite.geoanalytics.dataaccess.dao.JpaDao;
import gr.cite.geoanalytics.dataaccess.entities.principal.Principal;
import gr.cite.geoanalytics.dataaccess.entities.project.Project;
import gr.cite.geoanalytics.dataaccess.entities.project.ProjectDocument;
import gr.cite.geoanalytics.dataaccess.entities.project.Project.ProjectStatus;
import gr.cite.geoanalytics.dataaccess.entities.tenant.Tenant;
import gr.cite.geoanalytics.dataaccess.entities.workflow.Workflow;

@Repository
public class ProjectDaoImpl extends JpaDao<Project, UUID> implements ProjectDao {
	private static final Logger log = LoggerFactory.getLogger(ProjectDaoImpl.class);
	
	@Override
	public List<Project> getActiveProjects() {
		return entityManager.createQuery("from project p where p.status=" + 
				ProjectStatus.ACTIVE.statusCode(), Project.class).getResultList();
	}
	
	@Override
	public List<Project> getArchivedProjects() {
		return entityManager.createQuery("from project p where p.status=" + 
				ProjectStatus.ARCHIVE.statusCode(), Project.class).getResultList();
	}
	
	@Override
	public List<Project> getDeletedProjects() {
		return entityManager.createQuery("from project p where p.status=" + 
				ProjectStatus.DELETED.statusCode(), Project.class).getResultList();
	}
	
	@Override
	public List<Project> findByCreator(Principal principal) {
		TypedQuery<Project> query = entityManager.createQuery("from Project p where p.creator = :principal", Project.class);
		query.setParameter("principal", principal);
		
		return query.getResultList();
	}

	@Override
	public List<Project> findActiveByNameAndCreator(String name, Principal creator) {
		List<Project> projects = new ArrayList<Project>();
		
		try{
			TypedQuery<Project> query = entityManager.createQuery("from Project p where p.name = :name and p.creator = :creator and p.status="
					+ProjectStatus.ACTIVE.statusCode(), Project.class);
			query.setParameter("name", name);
			query.setParameter("creator", creator);
			
			projects = query.getResultList();
		} catch(Exception e) {
			e.printStackTrace();
		}
		
		return projects;
	}

	@Override
	public List<Project> findActiveByNameAndCreatorAndTenant(String name, Principal creator, String tenant) {
		TypedQuery<Project> query = entityManager.createQuery("FROM Project p WHERE p.name = :name AND p.tenant.name = :tenant AND p.creator = :creator AND p.status="
				+ProjectStatus.ACTIVE.statusCode(), Project.class);
		query.setParameter("name", name);
		query.setParameter("tenant", tenant);
		query.setParameter("creator", creator);
		
		return query.getResultList();
	}
	
	@Override
	public List<Project> findByNameAndCreator(String name, Principal creator) {
		TypedQuery<Project> query = entityManager.createQuery("from Project p where p.name = :name and p.creator = :creator", Project.class);
		query.setParameter("name", name);
		query.setParameter("creator", creator);
		
		return query.getResultList();
	}
	
	@Override
	public List<Project> findByNameAndCreatorAndTenant(String name, Principal creator, String tenant) {
		TypedQuery<Project> query = entityManager.createQuery("FROM Project p WHERE p.name = :name AND p.tenant.name = :tenant AND p.creator = :creator", Project.class);
		query.setParameter("name", name);
		query.setParameter("tenant", tenant);
		query.setParameter("creator", creator);
		
		return query.getResultList();
	}
	
	@Override
	public List<Project> findByName(String projectName){
		TypedQuery<Project> query = entityManager.createQuery("from Project p where p.name = :name", Project.class);
		query.setParameter("name", projectName);
		
		return query.getResultList();
	}
	
	@Override
	public List<Project> findByNameAndTenant(String projectName, String tenant){
		TypedQuery<Project> query = entityManager.createQuery("from Project p where p.name = :name AND p.tenant.name = :tenant", Project.class);
		query.setParameter("name", projectName);
		query.setParameter("tenant", tenant);
		
		return query.getResultList();
	}

	@Override
	public List<Project> findActiveByCreator(Principal creator) {
		TypedQuery<Project> query = entityManager.createQuery("from Project p where p.creator = :creator and p.status="
				+ProjectStatus.ACTIVE.statusCode(), Project.class);
		query.setParameter("creator", creator);
		
		return query.getResultList();
	}

	@Override
	public List<Project> findActiveByCreatorAndTenant(Principal creator, String tenantName) {
		List<Project> result = new ArrayList<Project>();
		
		try{
			TypedQuery<Project> query = entityManager.createQuery("from Project p where p.creator = :creator and p.tenant.name = :tenant and p.status="
					+ProjectStatus.ACTIVE.statusCode(), Project.class);
			query.setParameter("creator", creator);
			query.setParameter("tenant", tenantName);
			
			query.getResultList();
			
			for(Project project : result){
				Hibernate.initialize(project.getPrincipalProject());
			}
		} catch(Exception e){
			e.printStackTrace();
		}
		
		return result;
	}

	@Override
	public List<Project> findByTenant(Tenant tenant) {
		TypedQuery<Project> query = entityManager.createQuery("from Project p where p.tenant = :tenant", Project.class);
		query.setParameter("tenant", tenant);
		
		return query.getResultList();
	}

	@Override
	public List<Project> findByActiveTenant(Tenant tenant) {
		TypedQuery<Project> query = entityManager.createQuery("from Project p where p.tenant = :tenant and p.status="
				+ ProjectStatus.ACTIVE.statusCode(), Project.class);
		query.setParameter("tenant", tenant);
		
		return query.getResultList();
	}

	@Override
	public List<String> listProjects() {
		return entityManager.createQuery("select p.name from project p", String.class).getResultList();
	}
	
	@Override
	public List<String> listActiveProjects() {
		return entityManager.createQuery("select p.name from project p where p.status=" + 
				ProjectStatus.ACTIVE.statusCode(), String.class).getResultList();
	}
	
	@Override
	public List<String> listProjectsOfCreator(Principal creator) {
		TypedQuery<String> query = entityManager.createQuery("select p.name from Project p where p.creator = :creator", String.class);
		query.setParameter("creator", creator);
		
		return query.getResultList();
	}
	
	@Override
	public List<String> listActiveProjectsOfCreator(Principal creator) {
		TypedQuery<String> query = entityManager.createQuery("select p.name from Project p where p.creator = :creator and " +
				"p.status=" + ProjectStatus.ACTIVE.statusCode(), String.class);
		query.setParameter("principal", creator);
		
		return query.getResultList();
	}

	@Override
	public List<String> listProjectsOfTenant(Tenant tenant) {
		TypedQuery<String> query = entityManager.createQuery("select p.name from Project p where p.tenant = :tenant", String.class);
		query.setParameter("tenant", tenant);
		
		return query.getResultList();
	}

	@Override
	public List<String> listActiveProjectsOfTenant(Tenant tenant) {
		TypedQuery<String> query = entityManager.createQuery("select p.name from Project p where p.tenant = :tenant and " +
				"p.status=" + ProjectStatus.ACTIVE.statusCode(), String.class);
		query.setParameter("tenant", tenant);
		
		return query.getResultList();
	}
	
	//active projects
	@Override
	public List<Project> searchProjects(List<String> terms) {
		List<Project> result = null;
		
		StringBuilder queryB = new StringBuilder();
		queryB.append("from Project p");

		if(!terms.isEmpty()) queryB.append(" where p.status=" + ProjectStatus.ACTIVE.statusCode() + " and (");
		int j = 0;
		for(int i=0; i<terms.size(); i++)
		{
			queryB.append("lower(p.name) like :term" + j);
			j++;
			queryB.append(" or lower(p.description) like :term" + j);
			j++;
			if(i < terms.size()-1)
				queryB.append(" or ");
		}
		queryB.append(")");
		TypedQuery<Project> query = entityManager.createQuery(queryB.toString(), Project.class);
		j = 0;
		for(int i=0; i<terms.size(); i++)
		{
			String lower = terms.get(i).toLowerCase();
			query.setParameter("term"+(j++), "%"+lower+"%");
			query.setParameter("term"+(j++), "%"+lower+"%");
		}

		result = query.getResultList();
		
		log.debug("Active projects by name/description pattern matching:");
		log.debug((result != null ? result.size() : 0) + " results");
		if(log.isDebugEnabled() && result != null)
		{
			for (Project p : (List<Project>) result) 
				log.debug("Project (" + p.getId() + ")");
		}
		
		return result;
	}

	//active projects
	@Override
	public List<Project> searchProjectsOfCreator(List<String> terms, Principal creator) {
		List<Project> result = null;
		
		StringBuilder queryB = new StringBuilder();
		queryB.append("from Project p");

		if(!terms.isEmpty()) queryB.append(" where p.creator = :creator and p.status=" + ProjectStatus.ACTIVE.statusCode() + " and (");
		int j = 0;
		for(int i=0; i<terms.size(); i++)
		{
			queryB.append("lower(p.name) like :term" + j);
			j++;
			queryB.append(" or lower(p.description) like :term" + j);
			j++;
			if(i < terms.size()-1)
				queryB.append(" or ");
		}
		queryB.append(")");
		TypedQuery<Project> query = entityManager.createQuery(queryB.toString(), Project.class);
		query.setParameter("creator", creator);
		j = 0;
		for(int i=0; i<terms.size(); i++)
		{
			String lower = terms.get(i).toLowerCase();
			query.setParameter("term"+(j++), "%"+lower+"%");
			query.setParameter("term"+(j++), "%"+lower+"%");
		}

		result = query.getResultList();
		
		log.debug("Active projects of creator " + creator.getId() + " by name/description pattern matching:");
		log.debug((result != null ? result.size() : 0) + " results");
		if(log.isDebugEnabled() && result != null) {
			for (Project p : (List<Project>) result) 
				log.debug("Project (" + p.getId() + ")");
		}
		
		return result;
	}

	//active projects
	@Override
	public List<Project> searchProjectsOfTenant(List<String> terms, Tenant tenant) {
		List<Project> result = null;
		
		StringBuilder queryB = new StringBuilder();
		queryB.append("from Project p");

		if(!terms.isEmpty()) queryB.append(" where p.tenant = :tenant and p.status=" + ProjectStatus.ACTIVE.statusCode() + " and (");
		int j = 0;
		for(int i=0; i<terms.size(); i++)
		{
			queryB.append("lower(p.name) like :term" + j);
			j++;
			queryB.append(" or lower(p.description) like :term" + j);
			j++;
			if(i < terms.size()-1)
				queryB.append(" or ");
		}
		queryB.append(")");
		TypedQuery<Project> query = entityManager.createQuery(queryB.toString(), Project.class);
		query.setParameter("tenant", tenant);
		j = 0;
		for(int i=0; i<terms.size(); i++)
		{
			String lower = terms.get(i).toLowerCase();
			query.setParameter("term"+(j++), "%"+lower+"%");
			query.setParameter("term"+(j++), "%"+lower+"%");
		}

		result = query.getResultList();
		
		log.debug("Active projects of tenant " + tenant.getId() + " by name/description pattern matching:");
		log.debug((result != null ? result.size() : 0) + " results");
		if(log.isDebugEnabled() && result != null)
		{
			for (Project p : (List<Project>) result) 
				log.debug("Project (" + p.getId() + ")");
		}
		
		return result;
	}

	@Override
	public List<Workflow> getWorkflowsOfProject(Project p) {
		TypedQuery<Workflow> query = entityManager.createQuery("from Workflow w where w.project = :p", Workflow.class);
		query.setParameter("p", p);
	
		return query.getResultList();
	}
	
	@Override
	public Set<String> findClientOfPrincipal(Principal principal ) {
		TypedQuery<String> query = entityManager.createQuery("select distinct p.client from Project p where p.creator = :principal", 
				String.class);
		query.setParameter("principal", principal);
		
		List<String> result =  query.getResultList();
		return new HashSet<String>(result);
	}

	@Override
	public Project loadDetails(Project p) {
		p.getCreator().getName();
		p.getPrincipalProject().size();
		if(p.getTenant() != null)
			p.getTenant().getId();
		return p;
	}
}
