package gr.cite.geoanalytics.client;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.Serializable;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.Response.Status;

import org.apache.spark.api.java.JavaRDD;
import org.apache.spark.api.java.JavaSparkContext;
import org.apache.spark.api.java.function.Function;
import org.apache.spark.api.java.function.Function2;
import org.apache.spark.broadcast.Broadcast;
import org.geotools.geojson.feature.FeatureJSON;
import org.opengis.feature.simple.SimpleFeature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpEntity;
import org.springframework.http.RequestEntity;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.stereotype.Component;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import com.fasterxml.jackson.core.type.TypeReference;
import com.google.gson.Gson;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.core.util.MultivaluedMapImpl;
import com.vividsolutions.jts.geom.Geometry;

import gr.cite.clustermanager.exceptions.NoAvailableGos;
import gr.cite.clustermanager.layers.DataCreatorGeoanalytics;
import gr.cite.clustermanager.model.GosDefinition;
import gr.cite.clustermanager.model.ZNodeData.ZNodeStatus;
import gr.cite.clustermanager.trafficshaping.SimpleTrafficShaper;
import gr.cite.clustermanager.trafficshaping.TrafficShaper;
import gr.cite.gaap.datatransferobjects.ShapeMessenger;
import gr.cite.geoanalytics.dataaccess.entities.geocode.GeocodeSystem;
import gr.cite.geoanalytics.dataaccess.entities.layer.DataSource;
import gr.cite.geoanalytics.dataaccess.entities.layer.Layer;
import gr.cite.geoanalytics.dataaccess.entities.layer.LayerTenant;
import gr.cite.geoanalytics.dataaccess.entities.principal.Principal;
import gr.cite.geoanalytics.dataaccess.entities.shape.Shape;
import gr.cite.geoanalytics.dataaccess.entities.sysconfig.xml.layer.LayerBounds;
import gr.cite.geoanalytics.dataaccess.entities.tenant.Tenant;
import gr.cite.geoanalytics.dataaccess.geoserverbridge.elements.Bounds;
import gr.cite.geoanalytics.dataaccess.geoserverbridge.elements.FeatureType;
import gr.cite.geoanalytics.dataaccess.geoserverbridge.elements.GeoserverLayer;
import gr.cite.gos.client.GeoserverManagement;
import gr.cite.gos.client.ShapeManagement;

@Component
public class GeoanalyticsImportManagement extends GeoanalyticsManagement implements Serializable{

	private static final long serialVersionUID = -217872471686069616L;

	@Autowired private DataCreatorGeoanalytics dataCreatorGeoanalytics;
	
	private GeoanalyticsImportManagement(){}
	
	public GeoanalyticsImportManagement(String authenticationStr){
		super(authenticationStr);
	}
	
	
	
	public String importLayerFromRDD(String geoanalyticsEndpoint, GosDefinition gosDefinition, String layerName, String tenantID, String creatorID, JavaRDD<List<ShapeMessenger>> featuresRDD, String srid) throws Exception{
		

		JavaSparkContext jsc = new JavaSparkContext(featuresRDD.context());
		
		if(geoanalyticsEndpoint==null || geoanalyticsEndpoint.isEmpty() || gosDefinition==null || gosDefinition.getGosEndpoint()==null || gosDefinition.getGosEndpoint().isEmpty())
			throw new Exception("No gos or geoanalytics endpoint were correct: geoanalyticsEP="+geoanalyticsEndpoint+" gosEP="+gosDefinition.getGosEndpoint());
		
		//create Layer on geoanalytics
		Layer layer = new Layer();
		layer.setIsTemplate((short)0);
		layer.setName(layerName);
		layer.setDataSource(DataSource.PostGIS);
		layer.setDescription("This layer is generated by geoanalytics functions");
		layer.setReplicationFactor(1);
		
		Principal principal = new Principal();
		principal.setId(UUID.fromString(creatorID));
		layer.setCreator(principal);
		
		Set<LayerTenant> layerTenants = new HashSet<LayerTenant>();
		LayerTenant lt = new LayerTenant();
		Tenant t = new Tenant();
		t.setId(UUID.fromString(tenantID));
		lt.setTenant(t);
		layerTenants.add(lt);
		layer.setLayerTenants(layerTenants);
		
		
		String layerID = createLayer(geoanalyticsEndpoint, layer);
		
		if(layerID==null || layerID.isEmpty()) //no layer was created, so it can't proceed.
			throw new Exception("Error while creating the layer on Geoanalytics to host the results of the analytics function... Will not proceed with function execution!");
		
		layer.setId(UUID.fromString(layerID));
		
		
		Broadcast<String> sridBC = jsc.broadcast(srid);
		Broadcast<String> layerIdBC = jsc.broadcast(layerID);
		Broadcast<String> creatorIdBC = jsc.broadcast(creatorID);
		Broadcast<String> authStrBC = jsc.broadcast(authenticationStr);
//		Broadcast<String> geoanalyticsEndpointBC = jsc.broadcast(geoanalyticsEndpoint);
		Broadcast<GosDefinition> gosDefinitionBC = jsc.broadcast(gosDefinition);
		
		
		Set<Boolean> results = featuresRDD
				.map(new Mapper(layerIdBC.getValue(), creatorIdBC.getValue(), authStrBC.getValue(), gosDefinitionBC.getValue().getGosEndpoint()))
				.reduce(new Reducer());
		
		
		//if successful, create also the Layer on geoanalytics db, else rollback the shapes on GOS
		if(results.contains(new Boolean(false))){
			ShapeManagement sm = new ShapeManagement(authenticationStr);
			//means that there was at least one failed on executors, so rollback
			//1. delete from geoserver layer mappings (optional)
			GeoserverManagement gm = new GeoserverManagement(authenticationStr);
			gm.deleteGeoserverLayer(gosDefinition.getGosEndpoint(), layerID, layer.getDataSource());
			//2. delete layer view (optional)
			sm.applyOnView(gosDefinition.getGosEndpoint(), viewDeletion(layerID));
			//3. delete shapes
			sm.deleteShapesOfLayer(gosDefinition.getGosEndpoint(), layerID);
			//4. delete layer
			deleteLayer(geoanalyticsEndpoint, layer);
			// finally throw an exception notifying about the rollback
			throw new Exception("Error while doing the computations for the analytics... Performed a rollback!");
		}
		else {
			//means that all were successful, so go on with the remaining layer creation (create views, create geoserver entries, etc)
			//1. create layer view (optional)
			ShapeManagement sm = new ShapeManagement(authenticationStr);
			sm.applyOnView(gosDefinition.getGosEndpoint(), viewCreation(layerID, "numeric"));
			//2. create geoserver layer mapping (optional)
			addGeoserverLayer(gosDefinition, layerID, "point");
			//3. Notify zookeeper about the new layer
			dataCreatorGeoanalytics.addLayer(layerID, ZNodeStatus.ACTIVE, gosDefinition.getGosIdentifier());
		}
		
		return layerID;
		
	}
	
	
	public String createLayer(String geoanalyticsEndpoint, Layer layer){
		
		MultiValueMap<String, String> headers = new LinkedMultiValueMap<String, String>();
		headers.add(super.HEADER_AUTHENTICATION_PARAM_NAME, authenticationStr);
		headers.add("Content-Type", "application/json");

		RestTemplate restTemplate = new RestTemplate();
		restTemplate.getMessageConverters().add(new MappingJackson2HttpMessageConverter());
		HttpEntity<Layer> request = new HttpEntity<Layer>( layer, headers);
		return restTemplate.postForObject(geoanalyticsEndpoint+"/createLayerSpark", request, String.class);
		
	}
	
	public String deleteLayer(String geoanalyticsEndpoint, Layer layer) {
		
		MultiValueMap<String, String> headers = new LinkedMultiValueMap<String, String>();
		headers.add(super.HEADER_AUTHENTICATION_PARAM_NAME, authenticationStr);
		headers.add("Content-Type", "application/json");

		RestTemplate restTemplate = new RestTemplate();
		restTemplate.getMessageConverters().add(new MappingJackson2HttpMessageConverter());
		HttpEntity<Layer> request = new HttpEntity<Layer>( layer, headers);
		return restTemplate.postForObject(geoanalyticsEndpoint+"/deleteLayerSpark", request, String.class);
	}
	
	public String viewCreation(String layerID, String valueDatatype){
		return "CREATE MATERIALIZED VIEW \""+layerID+"\" AS SELECT s.\"SHP_Geography\"::geometry AS \"SHP_Geography\", s.\"SHP_ID\", (xpath('//extraData/function_result/text()'::text, s.\"SHP_ExtraData\"))[1]::text::"+valueDatatype+" AS value FROM \"Shape\" s  WHERE s.\"SHP_LayerID\" = '"+layerID+"'::uuid WITH DATA";
	}
	
	public String viewDeletion(String layerID){
		return "DROP MATERIALIZED VIEW IF EXISTS \""+layerID+"\"";
	}
	
	
	private void addGeoserverLayer(GosDefinition gosDefinition, String layerID, String style) throws IOException{
		
		Bounds boundingBox = new Bounds();
		boundingBox.setMinx(0);
		boundingBox.setMiny(0);
		boundingBox.setMaxx(180);
		boundingBox.setMaxy(90);
		boundingBox.setCrs("EPSG:4326");

		FeatureType featureType = new FeatureType();
		featureType.setDatastore(gosDefinition.getDatastoreName());
		featureType.setWorkspace(gosDefinition.getGeoserverWorkspace());
		featureType.setEnabled(true);
		featureType.setName(layerID);
		featureType.setTitle(layerID);
		featureType.setSrs("EPSG:4326");
		featureType.setNativeCRS("EPSG:4326");
		featureType.setNativeBoundingBox(boundingBox);
		featureType.setLatLonBoundingBox(boundingBox);

		GeoserverLayer geoserverLayer = new GeoserverLayer();
		geoserverLayer.setDatastore(gosDefinition.getDatastoreName());
		geoserverLayer.setWorkspace(gosDefinition.getGeoserverWorkspace());
		geoserverLayer.setEnabled(true);
		geoserverLayer.setDefaultStyle(style);
		geoserverLayer.setId(layerID);
		geoserverLayer.setTitle(layerID);
		geoserverLayer.setType("VECTOR");
		
		Integer minScale = null;//layerConfig.getMinScale();
		Integer maxScale = null;//layerConfig.getMaxScale();
		
		Map<String, String> layerStyles = new HashMap<String, String>();//configurationManager.getLayerStyles();
		layerStyles.put("point", style);
		
		GeoserverManagement gm = new GeoserverManagement(authenticationStr);
		gm.addGeoserverLayer(gosDefinition.getGosEndpoint(), geoserverLayer, featureType, layerStyles, minScale, maxScale);
	}
	
	
}
