package gr.cite.regional.data.collection.dataaccess.services;

import gr.cite.regional.data.collection.dataaccess.daos.DataModelDao;
import gr.cite.regional.data.collection.dataaccess.entities.DataModel;
import gr.cite.regional.data.collection.dataaccess.entities.Domain;
import gr.cite.regional.data.collection.dataaccess.entities.UserReference;
import gr.cite.regional.data.collection.dataaccess.exceptions.ServiceException;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.stream.Collectors;

@Service
public class DataModelServiceImpl implements DataModelService {
	private static final Logger logger = LogManager.getLogger(DataModelServiceImpl.class);
	
	private DataModelDao dataModelDao;
	private DomainService domainService;
	private UserReferenceService userReferenceService;
	
	@Autowired
	public DataModelServiceImpl(DataModelDao dataModelDao, DomainService domainService, UserReferenceService userReferenceService) {
		this.dataModelDao = dataModelDao;
		this.domainService = domainService;
		this.userReferenceService = userReferenceService;
	}
	
	@Override
	@Transactional(rollbackOn = ServiceException.class)
	public DataModel addDataModel(DataModel dataModel) throws ServiceException {
		try {
			setManagedReferences(dataModel);
			return this.dataModelDao.create(dataModel);
		} catch (Exception e) {
			throw new ServiceException("Error on DataModel insertion", e);
		}
	}
	
	private void setManagedReferences(DataModel dataModel) throws ServiceException {
		Domain domain = this.domainService.getDomainByLabel(dataModel.getDomain().getLabel());
		UserReference creator = this.userReferenceService.getUserReferenceByLabel(dataModel.getCreateUser().getLabel());
		dataModel.setDomain(domain);
		dataModel.setCreateUser(creator);
	}
	
	@Override
	@Transactional
	public DataModel updateDataModel(DataModel dataModel) throws ServiceException {
		logger.debug("Service that updates data model");
		
		DataModel current = this.dataModelDao.read(dataModel.getId());
		if (current  == null) throw new NoSuchElementException("DataModel [" + dataModel.getId() + "] does not exist");

		if ( dataModel.getUpdateUser() != null)
			dataModel.setUpdateUser( this.userReferenceService.getUserReferenceByLabel( dataModel.getUpdateUser().getLabel() ) );

		replaceModifiedFields(dataModel, current);
		
		return this.dataModelDao.update(current);
	}
	
	private void replaceModifiedFields(DataModel dataModel, DataModel currentDataModel) {
		if (dataModel.getLabel() != null) {
			currentDataModel.setLabel(dataModel.getLabel());
		}
		if (dataModel.getDefinition() != null) {
			currentDataModel.setDefinition(dataModel.getDefinition());
		}
		if (dataModel.getPrevious() != null) {
			currentDataModel.setPrevious(dataModel.getPrevious());
		}
		if (dataModel.getUri() != null) {
			currentDataModel.setUri(dataModel.getUri());
		}
		if (dataModel.getVersion() != null) {
			currentDataModel.setVersion(dataModel.getVersion());
		}
		if (dataModel.getProperties() != null) {
			currentDataModel.setProperties(dataModel.getProperties());
		}
		if ( dataModel.getUpdateUser() != null )
			currentDataModel.setUpdateUser( dataModel.getUpdateUser() );
	}
	
	@Override
	@Transactional
	public DataModel getDataModel(Integer dataModelId) {
		logger.debug("Service that gets data model by id");
		
		DataModel dataModel = this.dataModelDao.read(dataModelId);
		if (dataModel == null) throw new NoSuchElementException("DataModel [" + dataModelId + "] does not exist");
		return dataModel;
	}



	@Override
	@Transactional
	public List<DataModel> getDataModelsByDomainLabel(String domainLabel) throws ServiceException {
		logger.debug("Service that gets data models by domain label");
		Domain domain = this.domainService.getDomainByLabel( domainLabel );

		return new ArrayList<>(domain.getDataModels());//.forEach( dm -> this.dataModelDao.loadDetails( dm ) );
	}
	
	@Override
	@Transactional
	public List<DataModel> getAllDataModels() throws ServiceException {
		logger.debug("Service that gets all data models");
		try {
			return this.dataModelDao.getAll();
		} catch (Exception e) {
			throw new ServiceException("An error occurred when retrieving data models", e);
		}
	}
	
	@Override
	@Transactional
	public void deleteDataModel(Integer dataModelId) throws ServiceException {
		logger.debug("Service that deletes data model");
		
		DataModel dataModel = this.dataModelDao.read(dataModelId);
		
		if (dataModel == null) throw new NoSuchElementException("DataModel [" + dataModelId + "] does not exist");
		if (dataModel.getDataCollections().size() > 0) throw new ServiceException("A data model cannot be deleted when the data collection period is active");
		
		try {
			this.dataModelDao.delete(dataModel);
		} catch (Exception e) {
			throw new ServiceException("An error occurred while deleting the data model " + dataModel.getLabel(), e);
		}
	}

	@Override
	@Transactional(rollbackOn = ServiceException.class)
	public boolean isDataCollectionPeriodActiveByDataModelId(Integer id) throws ServiceException {
		return !this.dataModelDao.read(id).getDataCollections().isEmpty();
	}
}