package gr.cite.regional.data.collection.application.services;

import gr.cite.regional.data.collection.application.dtos.AttributesDto;
import gr.cite.regional.data.collection.application.endpoint.ServiceDiscoveryException;
import gr.cite.regional.data.collection.application.endpoint.social.SocialNetworkingService;
import gr.cite.regional.data.collection.dataaccess.entities.DataCollection;
import gr.cite.regional.data.collection.dataaccess.entities.DataSubmission;
import gr.cite.regional.data.collection.dataaccess.entities.SubmissionStatus;
import gr.cite.regional.data.collection.dataaccess.entities.UserReference;
import gr.cite.regional.data.collection.dataaccess.exceptions.ServiceException;
import gr.cite.regional.data.collection.dataaccess.services.DataCollectionService;
import gr.cite.regional.data.collection.dataaccess.services.DataSubmissionService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

@Component
public class DataSubmissionHandlerImpl implements DataSubmissionHandler {
	private static  final Logger logger = LoggerFactory.getLogger(DataSubmissionHandlerImpl.class);
	private SocialNetworkingService socialNetworkingService;
	private DataSubmissionService dataSubmissionService;
	private DataCollectionService dataCollectionService;

	@Autowired
	public DataSubmissionHandlerImpl(SocialNetworkingService socialNetworkingService, DataSubmissionService dataSubmissionService, DataCollectionService dataCollectionService) {
		this.socialNetworkingService = socialNetworkingService;
		this.dataSubmissionService = dataSubmissionService;
		this.dataCollectionService = dataCollectionService;
	}

	@Override
	public DataSubmission addDataSubmissionAndNotify(String gcubeToken, AttributesDto attributesDto, DataSubmission dataSubmission) throws ServiceException {
		DataSubmission newDataSubmission = dataSubmissionService.createDataSubmission(dataSubmission);
		DataCollection dataCollection = this.dataCollectionService.getDataCollection(newDataSubmission.getDataCollection().getId());

		ExecutorService executorService = Executors.newSingleThreadExecutor();
		executorService.submit(() -> {
			boolean shouldDeleteDataSubmission = false;
			try {
				dataSubmissionService.addDataSubmissionCdts(newDataSubmission);
				notifyDataCollectionUsersForDataSubmission(dataCollection, newDataSubmission, gcubeToken, attributesDto, true);
			} catch (ServiceException e) {
				shouldDeleteDataSubmission = true;
				notifyDataCollectionUsersForDataSubmission(dataCollection, newDataSubmission, gcubeToken, attributesDto, false);
			}
			if (shouldDeleteDataSubmission){
				try {
					this.dataSubmissionService.deleteDataSubmission(newDataSubmission.getId());
				} catch (Exception e) {
					logger.error("Failed to create data submission", e);
				}
			}
		});
		return newDataSubmission;
	}

	@Override
	public void updateDataSubmission(String gcubeToken, AttributesDto attributesDto, DataSubmission dataSubmissionEntity) throws ServiceException {
		ExecutorService executorService = Executors.newSingleThreadExecutor();
		if (!this.dataSubmissionService.canModifyDataSubmission(dataSubmissionEntity.getId())) throw new IllegalStateException("Can not modify locked data submission");

		DataCollection dataCollection = this.dataCollectionService.getDataCollection(dataSubmissionEntity.getDataCollection().getId());
		this.dataSubmissionService.setDataSubmissionStatus(dataSubmissionEntity, SubmissionStatus.PENDING);
		executorService.submit(() -> {
			try {
				DataSubmission dataSubmissionEntityUpdated = this.dataSubmissionService.updateDataSubmissionAndData(dataSubmissionEntity);
				notifyDataCollectionUsersForDataSubmission(dataCollection, dataSubmissionEntityUpdated, gcubeToken, attributesDto, true);
			} catch (Exception e) {
				notifyDataCollectionUsersForDataSubmission(dataCollection, dataSubmissionEntity, gcubeToken, attributesDto, false);
				this.dataSubmissionService.setDataSubmissionStatus(dataSubmissionEntity, SubmissionStatus.COMPLETED);
				logger.error("Failed to create data submission", e);
			}
		});
	}



	private void notifyDataCollectionUsersForDataSubmission(DataCollection dataCollection, DataSubmission dataSubmission, String token, AttributesDto attributesDto, boolean success) {
		UserReference dataCollectionCreator = dataCollection.getCreateUser();
		UserReference dataCollectionUpdateUser = dataCollection.getUpdateUser();

		try {
			notifyUser(dataCollection, dataSubmission, dataCollectionCreator, token, attributesDto, success);
			if (dataCollectionUpdateUser != null)
				notifyUser(dataCollection, dataSubmission, dataCollectionUpdateUser, token, attributesDto, success);

		} catch (ServiceDiscoveryException e) {
			logger.error("Failed to invoke social-networking-service for data submission [" + dataSubmission.getId() + "]", e);
		}
	}
	private void notifyUser(DataCollection dataCollection, DataSubmission dataSubmission, UserReference user, String token, AttributesDto attributesDto, boolean success) throws ServiceDiscoveryException {
		this.socialNetworkingService.notifyUser(dataSubmission.getDomain().getLabel(), dataSubmission.getId(), token, user.getLabel(), dataCollection, attributesDto, success);
	}
}
