/*
 * Copyright (C) 2012 Engineering Ingegneria Informatica S.p.A.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gcube.accounting.datamodel;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;

import org.gcube.accounting.exception.InvalidValueException;

@XmlRootElement(name = "usageRecord")
@XmlAccessorType(XmlAccessType.PROPERTY)
public class NetworkUsageRecord extends AbstractDelegateUsageRecord {

	public static final String TYPE_NETWORK = "network";

	private static final String OVERALL_NETWORK_IN = "overallNetworkIn";
	private static final String PERIOD_NETWORK_IN = "periodNetworkIn";
	private static final String OVERALL_NETWORK_OUT = "overallNetworkOut";
	private static final String PERIOD_NETWORK_OUT = "periodNetworkOut";
	private static final String REF_VM = "refVM";

	public NetworkUsageRecord(UsageRecord ur) {
		super(ur);
		this.setResourceType(TYPE_NETWORK);
	}

	public NetworkUsageRecord() {
		this(new RawUsageRecord());
	}

	/**
	 * Return the amount of bytes received in the period covered by this record
	 * 
	 * @return
	 */
	@XmlElement(name = "periodNetworkIn")
	public String getPeriodNetworkIn() {
		return this.getResourceSpecificProperty(PERIOD_NETWORK_IN);
	}

	/**
	 * Set the amount of bytes received in the period covered by this record
	 * 
	 * @return
	 */
	public void setPeriodNetworkIn(Long netIn) throws InvalidValueException {
		if(netIn<0)
			throw new InvalidValueException("network usage cannot be less than zero");
		this.setResourceSpecificProperty(PERIOD_NETWORK_IN, netIn.toString());
	}

	/**
	 * Set the amount of bytes sent in the period covered by this record
	 * 
	 * @return
	 */
	public void setPeriodNetworkOut(Long netOut) throws InvalidValueException {
		if(netOut<0)
			throw new InvalidValueException("network usage cannot be less than zero");
		this.setResourceSpecificProperty(PERIOD_NETWORK_OUT, netOut.toString());
	}

	/**
	 * Return the amount of bytes sent in the period covered by this record
	 * 
	 * @return
	 */
	@XmlElement(name = "periodNetworkOut")
	public String getPeriodNetworkOut() {
		return this.getResourceSpecificProperty(PERIOD_NETWORK_OUT);
	}

	/**
	 * Set the overall amount of bytes sent in the VM lifetime
	 * 
	 * @return
	 */
	public void setOverallNetworkIn(Long netIn) throws InvalidValueException {
		if(netIn<0)
			throw new InvalidValueException("network usage cannot be less than zero");
		this.setResourceSpecificProperty(OVERALL_NETWORK_IN, netIn.toString());
	}

	/**
	 * Return the overall amount of bytes sent in the VM lifetime
	 * 
	 * @return
	 */
	@XmlElement(name = "overallNetworkIn")
	public String getOverallNetworkIn() {
		return this.getResourceSpecificProperty(OVERALL_NETWORK_IN);
	}

	/**
	 * Set the overall amount of bytes sent in the VM lifetime
	 * 
	 * @return
	 */
	public void setOverallNetworkOut(Long netOut) throws InvalidValueException {
		if(netOut<0)
			throw new InvalidValueException("network usage cannot be less than zero");
		this.setResourceSpecificProperty(OVERALL_NETWORK_OUT, netOut.toString());
	}

	/**
	 * Return the overall amount of bytes sent in the VM lifetime
	 * 
	 * @return
	 */
	@XmlElement(name = "overallNetworkOut")
	public String getOverallNetworkOut() {
		return this.getResourceSpecificProperty(OVERALL_NETWORK_OUT);
	}

	/**
	 * Return a reference to the VM generating network traffic
	 * 
	 * @return
	 */
	@XmlElement(name = "refVM")
	public String getRefVM() {
		return this.getResourceSpecificProperty(REF_VM);
	}

	/**
	 * Set a reference to the VM generating network traffic
	 * 
	 * @return
	 */
	public void setRefVM(String refVM) {
		this.setResourceSpecificProperty(REF_VM, refVM);
	}

	public void validate() throws InvalidValueException {
		super.validate();
		if(!TYPE_NETWORK.toString().equals(this.getResourceType()))
			throw new InvalidValueException("field 'resourceType' must be '"+TYPE_NETWORK.toString()+"'");
	}	
	
	
}
