/*
 * Copyright (C) 2012 Engineering Ingegneria Informatica S.p.A.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gcube.accounting.datamodel;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;

import org.gcube.accounting.exception.InvalidValueException;

@XmlRootElement(name = "usageRecord")
@XmlAccessorType(XmlAccessType.PROPERTY)
public class StorageRecord extends AbstractDelegateUsageRecord {

	public static final String STORAGE = "storage";

	private static final String STORAGE_VOLUME = "storageVolume";
	private static final String STORAGE_TRANSACTIONS = "storageTransactions";
	private static final String STORAGE_ITEMS = "itemCount";
	private static final String STORAGE_CONTEXT = "context";

	public StorageRecord() {
		this(new RawUsageRecord());
	}

	public StorageRecord(UsageRecord ur) {
		super(ur);
		this.setResourceType(STORAGE);
	}

	/**
	 * Set the overall amount of storage in use at the end of the record period
	 * 
	 * @param volume
	 */
	public void setStorageVolume(Long volume) throws InvalidValueException {
		if (volume < 0)
			throw new InvalidValueException("volume cannot be less than zero");
		this.setResourceSpecificProperty(STORAGE_VOLUME, volume.toString());
	}

	/**
	 * Return the overall amount of storage in use at the end of the record
	 * period
	 * 
	 * @return
	 */
	@XmlElement(name="storageVolume")
	public long getStorageVolume() {
		try {
			return Long.parseLong(this.ur
					.getResourceSpecificProperty(STORAGE_VOLUME));
		} catch (NumberFormatException e) {
			return 0l;
		}
	}

	/**
	 * Set the number of basic storage operations performed in the period
	 * 
	 * @param transactions
	 */
	public void setStorageTransactions(Long transactions)
			throws InvalidValueException {
		if (transactions < 0) {
			throw new InvalidValueException(
					"transaction number cannot be less than zero");
		}
		this.setResourceSpecificProperty(STORAGE_TRANSACTIONS, transactions
				.toString());
	}

	/**
	 * Return the number of basic storage operations performed in the period
	 * 
	 * @return
	 */
	@XmlElement(name="storageTransactions")
	public long getStorageTransactions() {
		try {
			return Long.parseLong(this.ur
					.getResourceSpecificProperty(STORAGE_TRANSACTIONS));
		} catch (NumberFormatException e) {
			return 0l;
		}
	}

	/**
	 * Set the number of storage items in use at the end of the period
	 * 
	 * @param items
	 */
	public void setItemCount(Long items) throws InvalidValueException {
		if (items < 0) {
			throw new InvalidValueException(
					"items number cannot be less than zero");
		}
		this.setResourceSpecificProperty(STORAGE_ITEMS, items.toString());
	}

	/**
	 * Return the number of storage items in use at the end of the period
	 * 
	 * @param items
	 */
	@XmlElement(name="itemCount")
	public long getItemCount() {
		try {
			return Long.parseLong(this.ur
					.getResourceSpecificProperty(STORAGE_ITEMS));
		} catch (NumberFormatException e) {
			return 0l;
		}
	}

	/**
	 * Set the context we're referring to
	 * 
	 * @param context
	 */
	public void setStorageContext(String context) {
		// to avoid setting a null value in the map
		if (context != null)
			this.setResourceSpecificProperty(STORAGE_CONTEXT, context);
	}

	/**
	 * Return the context for this record
	 * 
	 * @return
	 */
	public String getStorageContext() {
		return this.getResourceSpecificProperty(STORAGE_CONTEXT);
	}
	
	public void validate() throws InvalidValueException {
		super.validate();
		if(!STORAGE.toString().equals(this.getResourceType()))
			throw new InvalidValueException("field 'resourceType' must be '"+STORAGE.toString()+"'");
	}	

}
