/*
 * Copyright (C) 2012 Engineering Ingegneria Informatica S.p.A.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gcube.accounting.ut;

import java.net.URI;
import java.util.Collection;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriBuilder;

import org.apache.log4j.Logger;
import org.gcube.accounting.datamodel.AbstractDelegateUsageRecord;
import org.gcube.accounting.datamodel.Consumer;
import org.gcube.accounting.datamodel.JobUsageRecord;
import org.gcube.accounting.datamodel.NetworkUsageRecord;
import org.gcube.accounting.datamodel.RawUsageRecord;
import org.gcube.accounting.datamodel.StorageRecord;
import org.gcube.accounting.datamodel.UsageRecord;
import org.gcube.accounting.datamodel.User;
import org.gcube.accounting.datamodel.VMUsageRecord;
import org.gcube.accounting.security.authn.HTTPBasicAuthentication;

import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.api.client.config.ClientConfig;
import com.sun.jersey.api.client.config.DefaultClientConfig;

/**
 * A client-side class for accessing the usage tracker
 */

public class UsageTrackerRestAPI {

	/**
	 * A logger for this class
	 */
	private static Logger logger = Logger.getLogger(UsageTrackerRestAPI.class);

	private WebResource service;
//	private static Client client;
	private Client client;
	
	private String uri;

	/**
	 * 
	 * @param uri
	 *            the uri of the Usage Tracker
	 */
	public UsageTrackerRestAPI(String uri) { 
		ClientConfig config = new DefaultClientConfig();
		client = Client.create(config);
		this.uri=uri;
		this.service = client.resource(getBaseURI(this.uri));

	}

	private static URI getBaseURI(String uri) {
		return UriBuilder.fromUri(uri).build();
	}

	/**
	 * Insert a set of usage records
	 * 
	 * @param records
	 */
	public void insertUsageRecords(Collection<UsageRecord> records) {

		service=client.resource(this.uri);
		logger.debug("bulk upload of " + records.size() + " usage records...");
		for (UsageRecord ur : records) {
			this.insertUsageRecord(ur);
		}
		logger.debug("bulk upload completed");
	}

	/**
	 * Insert the new given usage record
	 * 
	 * @param ur
	 */
	public void insertUsageRecord(UsageRecord ur) {

		service=client.resource(this.uri);
		logger.debug("uploading usage record " + ur.getId() + "...");
		service.path("usagerecords").accept(MediaType.TEXT_PLAIN).type(
				MediaType.APPLICATION_XML).post(ur);
		logger.debug("uploaded");
	}

	/**
	 * Insert the new given usage record
	 * 
	 * @param ur
	 */
	public void insertUsageRecord(AbstractDelegateUsageRecord ur) {
		this.insertUsageRecord(ur.getDelegateUR());
	}

	/**
	 * Insert the given VM-specific usage record
	 * @param ur
	 */
	public void insertUsageRecord(VMUsageRecord ur) {
		logger.debug("uploading vm usage record " + ur.getId() + "...");
		service=client.resource(this.uri);
		service.path("usagerecords").path("vm").accept(MediaType.TEXT_PLAIN)
		.type(MediaType.APPLICATION_XML).post(ur);
		logger.debug("uploaded");
	}

	/**
	 * Insert the given network-specific usage record
	 * @param ur
	 */
	public void insertUsageRecord(NetworkUsageRecord ur) {

		service=client.resource(this.uri);
		logger.debug("uploading network usage record " + ur.getId() + "...");
		service.path("usagerecords").path("network").accept(
				MediaType.TEXT_PLAIN).type(MediaType.APPLICATION_XML).post(ur);
		logger.debug("uploaded");
	}

	/**
	 * Insert the given storage-specific usage record
	 * @param ur
	 */
	public void insertUsageRecord(StorageRecord ur) {

		service=client.resource(this.uri);
		logger.debug("uploading storage usage record " + ur.getId() + "...");
		service.path("usagerecords").path("storage").accept(
				MediaType.TEXT_PLAIN).type(MediaType.APPLICATION_XML).post(ur);
		logger.debug("uploaded");
	}

	/**
	 * Insert the given job-specific usage record
	 * @param ur
	 */
	public void insertUsageRecord(JobUsageRecord ur) {

		service=client.resource(this.uri);
		logger.debug("uploading job usage record " + ur.getId() + "...");
		service.path("usagerecords").path("job").accept(
				MediaType.TEXT_PLAIN).type(MediaType.APPLICATION_XML).post(ur);
		logger.debug("uploaded");
	}


	/**
	 * Update an existing usage record. The match with the esisting record in
	 * the usage tracker is based on the record id
	 * 
	 * @param ur
	 */
	public void updateUsageRecord(UsageRecord ur) {

		service=client.resource(this.uri);
		logger.debug("updating usage record " + ur.getId() + "...");
		service.path("usagerecords").path(ur.getId()).accept(
				MediaType.TEXT_PLAIN).type(MediaType.APPLICATION_XML).put(ur);
		logger.debug("updated");
	}

	/**
	 * Update an existing usage record. The match with the esisting record in
	 * the usage tracker is based on the record id
	 * 
	 * @param ur
	 */
	public void updateUsageRecord(AbstractDelegateUsageRecord ur) {
		this.updateUsageRecord(ur.getDelegateUR());
	}

	/**
	 * Return the usage records matching the given query.
	 * 
	 * @param query must have the form: name=value&name>value&kname=value&...
	 * @return
	 */
	public Collection<UsageRecord> getUsageRecords(String query) {
		service=client.resource(this.uri);
		RawUsageRecord[] records = service.path("usagerecords").queryParam(
				"query", query).get(RawUsageRecord[].class);
		logger.debug("retrieved " + records.length + " usage records");
		Collection<UsageRecord> out = new Vector<UsageRecord>();
		for (UsageRecord r : records) {
			out.add(r);
		}
		return out;
	}

	/**
	 * Return the usage records matching the given query.
	 * 
	 * @deprecated use getUsageRecords(query) instead
	 * 
	 * @param query must have the form: name=value&name>value&kname=value&...
	 * @return
	 */
	public Collection<UsageRecord> getUsageRecordsByQuery(String query) {
		return this.getUsageRecords(query);
	}

	/**
	 * Return the VM usage records matching the given query.
	 * 
	 * @param query must have the form: name=value&name>value&kname=value&...
	 * @return
	 */
	public Collection<VMUsageRecord> getVMRecords(String query) {

		service=client.resource(this.uri);
		VMUsageRecord[] records = service.path("usagerecords").path("vm")
				.queryParam("query", query).get(VMUsageRecord[].class);
		logger.debug("retrieved " + records.length + " vm records");
		Collection<VMUsageRecord> out = new Vector<VMUsageRecord>();
		for (VMUsageRecord r : records) {
			out.add(r);
		}
		return out;
	}	

	/**
	 * Return the VM usage records matching the given query.
	 * 
	 * @deprecated use getUsageRecords(query) instead
	 * 
	 * @param query must have the form: name=value&name>value&kname=value&...
	 * @return
	 */
	public Collection<VMUsageRecord> getVMRecordsByQuery(String query) {
		return this.getVMRecords(query);
	}

	/**
	 * Return the network usage records matching the given query.
	 * 
	 * @param query must be the form: name=value&name>value&kname=value&...
	 * @return
	 */
	public Collection<NetworkUsageRecord> getNetworkRecords(String query) {

		service=client.resource(this.uri);
		NetworkUsageRecord[] records = service.path("usagerecords").path(
				"network").queryParam("query", query).get(
						NetworkUsageRecord[].class);
		logger.debug("retrieved " + records.length + " network records");
		Collection<NetworkUsageRecord> out = new Vector<NetworkUsageRecord>();
		for (NetworkUsageRecord r : records) {
			out.add(r);
		}
		return out;
	}

	/**
	 * Return the network records matching the given query.
	 * 
	 * @deprecated use getUsageRecords(query) instead
	 * 
	 * @param query must have the form: name=value&name>value&kname=value&...
	 * @return
	 */
	public Collection<NetworkUsageRecord> getNetworkRecordsByQuery(String query) {
		return this.getNetworkRecords(query);
	}

	/**
	 * Return the storage usage records matching the given query.
	 * 
	 * @param query must be the form: name=value&name>value&kname=value&...
	 * @return
	 */
	public Collection<StorageRecord> getStorageRecords(String query) {

		service=client.resource(this.uri);
		StorageRecord[] records = service.path("usagerecords").path("storage")
				.queryParam("query", query).get(StorageRecord[].class);
		logger.debug("retrieved " + records.length + " storage records");
		Collection<StorageRecord> out = new Vector<StorageRecord>();
		for (StorageRecord r : records) {
			out.add(r);
		}
		return out;
	}	

	/**
	 * Return the storage records matching the given query.
	 * 
	 * @deprecated use getUsageRecords(query) instead
	 * 
	 * @param query must have the form: name=value&name>value&kname=value&...
	 * @return
	 */
	public Collection<StorageRecord> getStorageRecordsByQuery(String query) {
		return this.getStorageRecords(query);
	}

	/**
	 * Return the job usage records matching the given query.
	 * 
	 * @param query must be the form: name=value&name>value&kname=value&...
	 * @return
	 */
	public Collection<JobUsageRecord> getJobrecords(String query) {

		service=client.resource(this.uri);
		JobUsageRecord[] records = service.path("usagerecords").path("job")
				.queryParam("query", query).get(JobUsageRecord[].class);
		logger.debug("retrieved " + records.length + " job records");
		Collection<JobUsageRecord> out = new Vector<JobUsageRecord>();
		for (JobUsageRecord r : records) {
			out.add(r);
		}
		return out;
	}		

	/**
	 * Query the Usage Tracker for update information from each infrastructure (resource owner)
	 * 
	 * @return
	 */
	public Collection<UpdateInfo> getUpdateInfo() {
		
		logger.debug("querying the Usage Tracker for update information...");
		service=client.resource(this.uri);
		UpdateInfo[] infos = service.path("meta").path("update_info").get(
				UpdateInfo[].class);
		logger.debug("done");
		Collection<UpdateInfo> out = new Vector<UpdateInfo>();
		for (UpdateInfo info : infos) {
			out.add(info);
		}
		return out;
	}

	/**
	 * Return the list of users with an associated usage record
	 * 
	 * @return
	 * @deprecated this operation is for development/testing purposes only. It
	 *             will be removed in future releases
	 */
	public Collection<User> getUsers() {
		logger.debug("retrieving tracked users...");
		service=client.resource(this.uri);
		User[] users = service.path("meta").path("users").get(User[].class);
		logger.debug("retrieved " + users.length + " users");
		Collection<User> out = new Vector<User>();
		for (User user : users) {
			out.add(user);
		}
		return out;
	}

	public Collection<Consumer> getTrackedConsumers() {
		service=client.resource(this.uri);
		Consumer[] consumers = service.path("meta").path("consumers").get(Consumer[].class);
		Collection<Consumer> out = new Vector<Consumer>();
		for(Consumer c:consumers)
			out.add(c);
		return out;
		/*
		String tc = service.path("meta").path("consumers").get(String.class);
		logger.info(tc);
		StringTokenizer st = new StringTokenizer(tc, "|");
		Collection<String> out = new Vector<String>();
		while(st.hasMoreTokens()) {
			out.add(st.nextToken());
		}
		return out;
		*/
	}

	/**
	 * Remove all elements in the Usage Tracker
	 * 
	 * @deprecated this operation is for development/testing purposes only. It
	 *             will be removed in future releases
	 */
	public void clear() {
		logger.debug("clearing the usage tracker...");
		service=client.resource(this.uri);
		service.path("usagerecords").delete();
		logger.debug("done");
	}

	/**
	 * Query the Usage Tracker for records with the given <key, value> pair
	 * 
	 * @param key
	 * @param value
	 * @return
	 * @deprecated use getUsageRecordsByQuery instead
	 */
	public Collection<UsageRecord> getUsageRecordByProperty(String key,
			String value) {
		logger.debug("retrieving usage records where '" + key + "' is '"
				+ value + "'...");
		return this.getUsageRecordsByQuery(key + "=" + value);
	}

	public void deleteUsageRecord(UsageRecord record) {
		service=client.resource(this.uri);
		logger.info("deleting usage record " + record.getId() + "...");
		service.path("usagerecords/"+record.getId()).delete();
		logger.info("deleted");
	}

	public void setUserToken(String username, String password) {
		client.addFilter(new HTTPBasicAuthentication(username,password));
	}
	
	public void setUserToken(String username, String password, String delegator) {
		client.addFilter(new HTTPBasicAuthentication(username,password,delegator));
	}
	
}
