/*
 * Copyright (C) 2012 Engineering Ingegneria Informatica S.p.A.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gcube.accounting.common.helper;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

import org.apache.log4j.Logger;

public class TimeHelper {

	private static Logger logger = Logger.getLogger(TimeHelper.class);
	
	/**
	 * Compute the calendar object corresponding to the start of the week of the
	 * given year
	 * 
	 * @param year
	 * @param week
	 * @return
	 */
	public static Calendar getStart(int year, int week) {
		Calendar c = Calendar.getInstance();
		c.set(Calendar.HOUR_OF_DAY, 0);
		c.set(Calendar.MINUTE, 0);
		c.set(Calendar.SECOND, 0);
		c.set(Calendar.DAY_OF_WEEK, Calendar.SUNDAY);
		c.set(Calendar.YEAR, year);
		c.set(Calendar.WEEK_OF_YEAR, week);
		return c;
	}

	public static Calendar getMonthStart(int year, int month) {
		Calendar c = Calendar.getInstance();
		c.set(Calendar.HOUR_OF_DAY, 0);
		c.set(Calendar.MINUTE, 0);
		c.set(Calendar.SECOND, 0);
		c.set(Calendar.DAY_OF_MONTH, 1);
		c.set(Calendar.MONTH, month);
		c.set(Calendar.YEAR, year);
		return c;
	}

	/**
	 * Compute the calendar object corresponding to the end of the week of the
	 * given year
	 * 
	 * @param year
	 * @param week
	 * @return
	 */
	public static Calendar getEnd(int year, int week) {
		Calendar c = getStart(year, week);
		c.add(Calendar.WEEK_OF_YEAR, 1);
		return c;
	}

	public static Calendar getMonthEnd(int year, int month) {
		Calendar c = getMonthStart(year, month);
		c.add(Calendar.MONTH, 1);
		return c;
	}

	/**
	 * Return a human-readable representation of the given calendar.
	 * 
	 * @param c
	 *            the calendar object to format
	 * @param time
	 *            whether you want HH:MM:SS information as well
	 * @return the string representation for the calendar object
	 */
	public static String format(Calendar c, boolean time) {
		if (c == null) {
			return "undefined";
		}
		if (time) {
			SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
			return sdf.format(c.getTime());
		} else {
			SimpleDateFormat sdf = new SimpleDateFormat("EEE dd MMM yyyy");
			return sdf.format(c.getTime());
		}
	}

	/**
	 * Computes a Calendar for the given timestamp
	 * 
	 * @param timestamp
	 * @return
	 */
	public static Calendar getCalendar(long timestamp) {
		Date d = new Date(timestamp);
		Calendar cal = Calendar.getInstance();
		cal.setTime(d);
		return cal;
	}

	public static Calendar getCalendar(String timestamp)
			throws NumberFormatException {
		try {
			Long millis = Long.parseLong(timestamp);
			return getCalendar(millis);
		} catch (NumberFormatException e) {
			try {
				Double millis = Double.parseDouble(timestamp);
				return getCalendar(Math.round(millis));
			} catch (NumberFormatException e2) {
				throw e2;
			}
		}
	}

	/**
	 * Produce a human-readable representation of the given timestamp. Output is
	 * of the form: "5 days, 4 hours, 8 minutes, 12 seconds"
	 * 
	 * @param seconds
	 * @return
	 */
	public static String getHumanReadableTime(long seconds) {
		String s = "";
		long days = (long) Math.floor(seconds / 60 / 60 / 24);
		long hours = (long) Math.floor((seconds / 60 / 60) - (days * 24));
		long minutes = (long) Math.floor((seconds / 60)
				- (days * 24 * 60 + hours * 60));
		long rSeconds = (long) Math.floor((seconds)
				- (days * 24 * 60 * 60 + hours * 60 * 60 + minutes * 60));

		if (days > 0) {
			s += days + (days > 1 ? " days, " : " day, ");
		}
		if (days > 0 || hours > 0) {
			s += hours + (hours > 1 ? " hours, " : " hour, ");
		}
		if (days > 0 || hours > 0 || minutes > 0) {
			s += minutes + (minutes > 1 ? " minutes, " : " minute, ");
		}
		s += rSeconds + (rSeconds > 1 ? " seconds" : " second");

		return s;
	}

	/**
	 * Compute a ISO9601 representation of the given timestamp
	 * 
	 * @param seconds
	 * @return
	 */
	public static String getISO8601(long seconds) {
		String s = "P";

		long days = (long) Math.floor(seconds / 60 / 60 / 24);
		if (days > 0) {
			s += days + "D";
		}

		s += "T";

		long hours = (long) Math.floor((seconds / 60 / 60) - (days * 24));
		if (days > 0 || hours > 0) {
			s += hours + "H";
		}

		long minutes = (long) Math.floor((seconds / 60)
				- (days * 24 * 60 + hours * 60));
		if (days > 0 || hours > 0 || minutes > 0) {
			s += minutes + "M";
		}

		long rSeconds = (long) Math.floor((seconds)
				- (days * 24 * 60 * 60 + hours * 60 * 60 + minutes * 60));
		s += rSeconds + "S";
		return s;
	}

}
