/*
 * Copyright (C) 2012 Engineering Ingegneria Informatica S.p.A.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gcube.accounting;

import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Properties;

import org.apache.log4j.Logger;
import org.gcube.accounting.common.helper.TimeHelper;

public class VersionHelper {

	private static Logger logger = Logger.getLogger(VersionHelper.class);
	
	private String resource;
	private Properties properties;
	
	private VersionHelper() {
		properties = new Properties();
	}
	
	public VersionHelper(String resource) {
		this();
		this.resource = resource;
		this.loadProperties();
	}
	
	public VersionInfo getVersionInfo() {
		VersionInfo vi = new VersionInfo();
		vi.setBuildDate(this.getFormattedBuildDate());
		vi.setBuildId(this.getBuildId());
		vi.setBuildNumber(this.getBuildNumber());
		return vi;
	}
	
	private void loadProperties() {
        InputStream in = this.getClass().getResourceAsStream(this.resource);
        try {
        	properties.load(in);
        } catch (IOException e) {
        	logger.error(e.getMessage(), e);
        }
	}
	
	public Calendar getBuildDate() {
		// extract date from config file
		String value = this.properties.getProperty("build.date");
		if(value==null || value.isEmpty())
			return null;
		// parse date
		DateFormat formatter = new SimpleDateFormat("yyyy-MM-dd_HH-mm-ss");
		try {
			Date date = (Date)formatter.parse(value);
			Calendar out = Calendar.getInstance();
			out.setTime(date);
			return out;
		}
		catch(ParseException e) {
			logger.warn("unparseable version string: " + value);
			return null;
		}
	}
	
	public String getFormattedBuildDate() {
		Calendar c = this.getBuildDate();
		if(c!=null)
			return TimeHelper.format(c, true);
		else
			return "unknown";
	}

	public String getBuildNumber() {
		return this.properties.getProperty("build.number", "unknown");
	}

	public String getBuildId() {
		return this.properties.getProperty("build.id", "unknown");
	}

	public static void main(String[] args) {
		VersionHelper vh = new VersionHelper("/org/venusc/mab/usagetracker/version.properties");
		System.out.println(vh.getFormattedBuildDate());
		System.out.println(vh.getBuildNumber());
		System.out.println(vh.getBuildId());
	}

}