/*
 * Copyright (C) 2012 Engineering Ingegneria Informatica S.p.A.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gcube.accounting.datamodel;

import java.util.Calendar;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;

import org.apache.log4j.Logger;
import org.gcube.accounting.common.helper.TimeHelper;
import org.gcube.accounting.exception.InvalidValueException;

@XmlRootElement(name = "usageRecord")
@XmlAccessorType(XmlAccessType.PROPERTY)
public class JobUsageRecord extends AbstractDelegateUsageRecord {
	
	private static Logger logger = Logger.getLogger(JobUsageRecord.class);

	public static final String TYPE_JOB = "job";

	private static final String JOB_ID = "jobId";
	private static final String JOB_STATUS = "jobStatus";
	private static final String WALL_DURATION = "wallDuration";
	private static final String CPU_DURATION = "cpuDuration";
	private static final String REF_VM = "refVM";
	private static final String JOB_START = "jobStart";
	private static final String JOB_END = "jobEnd";
	private static final String JOB_NAME = "jobName";
	private static final String OVERALL_NETWORK_IN = "overallNetworkIn";
	private static final String OVERALL_NETWORK_OUT = "overallNetworkOut";
	private static final String MEMORY = "memory";
	private static final String DISK = "disk";
	private static final String PROCESSORS = "processors";
	private static final String CORES = "cores";
	private static final String INPUT_FILES_NUMBER = "inputFilesNumber";
	private static final String INPUT_FILES_SIZE = "inputFilesSize";
	private static final String OUTPUT_FILES_NUMBER = "outputFilesNumber";
	private static final String OUTPUT_FILES_SIZE = "outputFilesSize";

	public JobUsageRecord(UsageRecord ur) {
		super(ur);
		this.setResourceType(TYPE_JOB);
	}

	public JobUsageRecord() {
		this(new RawUsageRecord());
	}

	public String getJobId() {
		return this.getResourceSpecificProperty(JOB_ID);
	}

	public void setJobId(String jobId) {
		this.setResourceSpecificProperty(JOB_ID, jobId);
	}

	public String getJobStatus() {
		return this.getResourceSpecificProperty(JOB_STATUS);
	}

	public void setJobStatus(String jobStatus) {
		this.setResourceSpecificProperty(JOB_STATUS, jobStatus);
	}

	/**
	 * in millis
	 * 
	 * @return
	 */
	public Long getWallDuration() {
		try {
			return Long.parseLong(this
					.getResourceSpecificProperty(WALL_DURATION));
		} catch (NumberFormatException e) {
			return 0L;
		}
	}

	/**
	 * in millis
	 * 
	 * @param duration
	 */
	public void setWallDuration(Long duration) throws InvalidValueException {
		if(duration==null)
			return;
		if(duration<0)
			throw new InvalidValueException("duration cannot be less than zero");
		this.setResourceSpecificProperty(WALL_DURATION, duration.toString());
	}

	/**
	 * in millis
	 * 
	 * @return
	 */
	public Long getCpuDuration() {
		try {
			return Long.parseLong(this
					.getResourceSpecificProperty(CPU_DURATION));
		} catch (NumberFormatException e) {
			return 0L;
		}
	}

	/**
	 * in millis
	 * 
	 * @param duration
	 */
	public void setCpuDuration(Long duration) throws InvalidValueException {
		if(duration==null)
			return;
		if(duration<0)
			throw new InvalidValueException("duration cannot be less than zero");
		this.setResourceSpecificProperty(CPU_DURATION, duration.toString());
	}

	/**
	 * Set the overall amount of bytes sent in the VM lifetime
	 * 
	 * @return
	 */
	public void setOverallNetworkIn(Long netIn) throws InvalidValueException {
		if(netIn==null)
			return;
		if(netIn<0)
			throw new InvalidValueException("network usage cannot be less than zero");
		this.setResourceSpecificProperty(OVERALL_NETWORK_IN, netIn.toString());
	}

	/**
	 * Return the overall amount of bytes sent in the VM lifetime
	 * 
	 * @return
	 */
	// @XmlElement(name = "overallNetworkIn")
	public Long getOverallNetworkIn() {
		try {
			return Long.parseLong(this
					.getResourceSpecificProperty(OVERALL_NETWORK_IN));
		} catch (NumberFormatException e) {
			return 0L;
		}
	}

	/**
	 * Set the overall amount of bytes sent in the VM lifetime
	 * 
	 * @return
	 */
	public void setOverallNetworkOut(Long netOut) throws InvalidValueException {
		if(netOut==null)
			return;
		if(netOut<0)
			throw new InvalidValueException("network usage cannot be less than zero");
		this
				.setResourceSpecificProperty(OVERALL_NETWORK_OUT, netOut
						.toString());
	}

	/**
	 * Return the overall amount of bytes sent in the VM lifetime
	 * 
	 * @return
	 */
	// @XmlElement(name = "overallNetworkOut")
	public Long getOverallNetworkOut() {
		try {
			return Long.parseLong(this
					.getResourceSpecificProperty(OVERALL_NETWORK_OUT));
		} catch (NumberFormatException e) {
			return 0L;
		}

	}

	/**
	 * Return the id of the Virtual Machine
	 * 
	 * @return
	 */
	public String getRefVM() {
		return this.getResourceSpecificProperty(REF_VM);
	}

	/**
	 * Set the id of the virtual machine
	 * 
	 * @return
	 */
	public void setRefVM(String refVM) {
		this.setResourceSpecificProperty(REF_VM, refVM);
	}

	/**
	 * in bytes
	 * 
	 * @return
	 */
	public Long getDisk() {
		try {
			return Long.parseLong(this.getResourceSpecificProperty(DISK));
		} catch (NumberFormatException e) {
			return 0L;
		}
	}

	/**
	 * in bytes
	 * 
	 * @param duration
	 */
	public void setDisk(Long diskUsed) throws InvalidValueException {
		if(diskUsed==null)
			return;
		if(diskUsed<0)
			throw new InvalidValueException("disk usage cannot be less than zero");
		this.setResourceSpecificProperty(DISK, diskUsed.toString());
	}

	public String getJobName() {
		return this.getResourceSpecificProperty(JOB_NAME);
	}

	public void setJobName(String name) {
		this.setResourceSpecificProperty(JOB_NAME, name);
	}

	/**
	 * in bytes
	 * 
	 * @return
	 */
	public Long getMemory() {
		try {
			return Long.parseLong(this.getResourceSpecificProperty(MEMORY));
		} catch (NumberFormatException e) {
			return 0L;
		}
	}

	/**
	 * in bytes
	 * 
	 * @param duration
	 */
	public void setMemory(Long memory) throws InvalidValueException {
		if(memory==null)
			return;
		if(memory<0)
			throw new InvalidValueException("memory usage cannot be less than zero");
		this.setResourceSpecificProperty(MEMORY, memory.toString());
	}

	/**
	 * Return the date when the job started
	 * 
	 * @return
	 */
	public Calendar getJobStart() {
		String s = this.getResourceSpecificProperty(JOB_START);
		if(s==null)
			return null;
		try {
			return TimeHelper.getCalendar(s);
		} catch (NumberFormatException e) {
			return null;
		}
	}

	/**
	 * Return the date when the job ended
	 * 
	 * @return
	 */
	public Calendar getJobEnd() {
		String s = this.getResourceSpecificProperty(JOB_END);
		if(s==null)
			return null;
		try {
			return TimeHelper.getCalendar(s);
		} catch (NumberFormatException e) {
			return null;
		}
	}

	/**
	 * Set the date when the job started
	 * 
	 * @param usageStart
	 */
	public void setJobStart(Calendar jobStart) throws InvalidValueException {
		if(jobStart==null)
			return;
		if(this.getJobEnd()!=null && jobStart.after(this.getJobEnd()))
			throw new InvalidValueException("jobStart cannot be after jobEnd");
		this.setResourceSpecificProperty(JOB_START, Long.toString(jobStart
				.getTimeInMillis()));
	}

	/**
	 * Set the date when the job ended
	 * 
	 * @param usageEnd
	 */
	public void setJobEnd(Calendar jobEnd) throws InvalidValueException {
		if(jobEnd==null)
			return;
		if(this.getJobStart()!=null && jobEnd.before(this.getJobStart()))
			throw new InvalidValueException("jobEnd cannot be before jobStart");
		this.setResourceSpecificProperty(JOB_END, Long.toString(jobEnd
				.getTimeInMillis()));
	}

	public Integer getProcessors() {
		String rsp = this.getResourceSpecificProperty(PROCESSORS);
		if(rsp==null)
			return null;
		try {
			return Integer.parseInt(rsp);
		} catch (NumberFormatException e) {
			return null;
		}
	}

	public void setProcessors(Integer processors) throws InvalidValueException {
		if(processors==null)
			return;
		if(processors<=0)
			throw new InvalidValueException("processors number must be greater than zero");
		if(processors==null)
			throw new InvalidValueException("processors cannot be null");
		this.setResourceSpecificProperty(PROCESSORS, processors.toString());
	}

	public Integer getCores() {
		String rsp = this.getResourceSpecificProperty(CORES);
		if(rsp==null)
			return null;
		try {
			return Integer.parseInt(rsp);
		} catch (NumberFormatException e) {
			return null;
		}
	}

	public void setCores(Integer cores) throws InvalidValueException {
		if(cores==null)
			return;
		if(cores<=0)
			throw new InvalidValueException("cores number must be greater than zero");
		if(cores==null)
			throw new InvalidValueException("cores cannot be null");
		this.setResourceSpecificProperty(CORES, cores.toString());
	}

	public Long getInputFilesNumber() {
		try {
			return Long.parseLong(this
					.getResourceSpecificProperty(INPUT_FILES_NUMBER));
		} catch (NumberFormatException e) {
			return 0L;
		}
	}

	public void setInputFilesNumber(Long n) throws InvalidValueException {
		if(n==null)
			return;
		if(n<0)
			throw new InvalidValueException("number of input files cannot be negative");
		this.setResourceSpecificProperty(INPUT_FILES_NUMBER, n.toString());
	}

	public Long getInputFilesSize() {
		try {
			return Long.parseLong(this
					.getResourceSpecificProperty(INPUT_FILES_SIZE));
		} catch (NumberFormatException e) {
			return 0L;
		}
	}

	public void setInputFilesSize(Long s) throws InvalidValueException {
		if(s==null)
			return;
		if(s<0)
			throw new InvalidValueException("size of input files cannot be negative");
		this.setResourceSpecificProperty(INPUT_FILES_SIZE, s.toString());
	}

	public Long getOutputFilesNumber() {
		try {
			return Long.parseLong(this
					.getResourceSpecificProperty(OUTPUT_FILES_NUMBER));
		} catch (NumberFormatException e) {
			return 0L;
		}
	}

	public void setOutputFilesNumber(Long n) throws InvalidValueException {
		if(n==null)
			return;
		if(n<0)
			throw new InvalidValueException("number of output files cannot be negative");
		this.setResourceSpecificProperty(OUTPUT_FILES_NUMBER, n.toString());
	}

	public Long getOutputFilesSize() {
		try {
			return Long.parseLong(this
					.getResourceSpecificProperty(OUTPUT_FILES_SIZE));
		} catch (NumberFormatException e) {
			return 0L;
		}
	}

	public void setOutputFilesSize(Long s) throws InvalidValueException {
		if(s==null)
			return;
		if(s<0)
			throw new InvalidValueException("size of output files cannot be negative");
		this.setResourceSpecificProperty(OUTPUT_FILES_SIZE, s.toString());
	}

	public void validate() throws InvalidValueException {
		super.validate();
		if(!TYPE_JOB.toString().equals(this.getResourceType()))
			throw new InvalidValueException("field 'resourceType' must be '"+TYPE_JOB.toString()+"'");
	}		
	
}
