/*
 * Copyright (C) 2012 Engineering Ingegneria Informatica S.p.A.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gcube.accounting.datamodel;

import java.util.Calendar;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;

import org.apache.log4j.Logger;
import org.gcube.accounting.common.helper.TimeHelper;
import org.gcube.accounting.exception.InvalidValueException;

@XmlRootElement(name = "usageRecord")
@XmlAccessorType(XmlAccessType.PROPERTY)
public class VMUsageRecord extends AbstractDelegateUsageRecord {

	public static final String TYPE_VM = "VM";

	private static final String REF_VM = "refVM";
	private static final String REF_HOST = "refHost";
	private static final String USAGE_START = "usageStart";
	private static final String USAGE_END = "usageEnd";
	private static final String CORES = "cores";

	public VMUsageRecord(UsageRecord ur) {
		super(ur);
		this.setResourceType(TYPE_VM);
	}

	public VMUsageRecord() {
		this(new RawUsageRecord());
	}

	/**
	 * Return the id of the Virtual Machine
	 * 
	 * @return
	 */
	public String getRefVM() {
		return this.getResourceSpecificProperty(REF_VM);
	}

	/**
	 * Set the id of the virtual machine
	 * 
	 * @return
	 */
	public void setRefVM(String refVM) {
		this.setResourceSpecificProperty(REF_VM, refVM);
	}

	/**
	 * Return the number of cores allocated to the Virtual Machine
	 * 
	 * @return
	 */
	public Integer getCores() {
		try {
			return Integer.parseInt(this.getResourceSpecificProperty(CORES));
		} catch (NumberFormatException e) {
			return null;
		}
	}

	/**
	 * Set the number of cores allocated to the Virtual Machine
	 * 
	 * @return
	 */	
	public void setCores(Integer cores) throws InvalidValueException {
		if(cores==null)
			return;
		if(cores<0)
			throw new InvalidValueException("number of cores cannot be less than zero");
		this.setResourceSpecificProperty(CORES, cores.toString());
	}
	
	/**
	 * Return a reference to the machine hosting the virtual machine
	 * 
	 * @return
	 */
	public String getRefHost() {
		return this.getResourceSpecificProperty(REF_HOST);
	}

	/**
	 * Set a reference to the machine hosting the virtual machine
	 * 
	 * @param refHost
	 */
	public void setRefHost(String refHost) {
		this.setResourceSpecificProperty(REF_HOST, refHost);
	}

	/**
	 * Return the date when the machine usage started
	 * 
	 * @return
	 */
	public Calendar getUsageStart() {
		try {
			String usageStart = this.getResourceSpecificProperty(USAGE_START);
			if (usageStart != null) {
				return TimeHelper.getCalendar(usageStart);
			} else {
				return null;
			}
		} catch (NumberFormatException e) {
			return null;
		}
	}

	/**
	 * Return the date when the machine usage ended
	 * 
	 * @return
	 */
	public Calendar getUsageEnd() {
		try {
			String usageEnd = this.getResourceSpecificProperty(USAGE_END);
			if (usageEnd != null) {
				return TimeHelper.getCalendar(usageEnd);
			} else {
				return null;
			}
		} catch (NumberFormatException e) {
			return null;
		}
	}

	/**
	 * Set the date when the machine usage started
	 * 
	 * @param usageStart
	 */
	public void setUsageStart(Calendar usageStart) throws InvalidValueException {
		if(this.getStartTime()!=null && usageStart.before(this.getStartTime()))
			throw new InvalidValueException("usageStart cannot be before startTime");
		if(this.getUsageEnd()!=null && usageStart.after(this.getUsageEnd()))
			throw new InvalidValueException("usageStart cannot be after usageEnd");
		if(this.getEndTime()!=null && usageStart.after(this.getEndTime()))
			throw new InvalidValueException("usageStart cannot be after endTime");
		this.setResourceSpecificProperty(USAGE_START, Long.toString(usageStart.getTimeInMillis()));
	}

	/**
	 * Set the date when the machine usage ended
	 * 
	 * @param usageEnd
	 */
	public void setUsageEnd(Calendar usageEnd) throws InvalidValueException {
		if(this.getStartTime()!=null && usageEnd.before(this.getStartTime()))
			throw new InvalidValueException("usageEnd cannot be before startTime");
		if(this.getUsageStart()!=null && usageEnd.before(this.getUsageStart()))
			throw new InvalidValueException("usageEnd cannot be before usageStart");
		if(this.getEndTime()!=null && usageEnd.after(this.getEndTime()))
			throw new InvalidValueException("usageEnd cannot be after endTime");
		this.setResourceSpecificProperty(USAGE_END, Long.toString(usageEnd.getTimeInMillis()));
	}
	
	public void setStartTime(Calendar startTime) throws InvalidValueException {
		if(this.getUsageStart()!=null && startTime.after(this.getUsageStart()))
			throw new InvalidValueException("startTime cannot be after usageStart");
		if(this.getUsageEnd()!=null && startTime.after(this.getUsageEnd()))
			throw new InvalidValueException("startTime cannot be after usageEnd");
		super.setStartTime(startTime);
	}

	public void validate() throws InvalidValueException {
		super.validate();
		if(!TYPE_VM.toString().equals(this.getResourceType()))
			throw new InvalidValueException("field 'resourceType' must be 'VM'");
		if(this.getUsageStart()==null)
			throw new InvalidValueException("field 'usageStart' is mandatory");
		if(this.getUsageEnd()==null)
			throw new InvalidValueException("field 'usageEnd' is mandatory");
		if(this.getRefVM()==null)
			throw new InvalidValueException("field 'refVM' is mandatory");
//		if(this.getRefHost()==null)
//			throw new InvalidValueException("field 'refHost' is mandatory");
	}

}
