/*
 * Decompiled with CFR 0.152.
 */
package com.couchbase.client.core.tracing;

import com.couchbase.client.core.logging.CouchbaseLogger;
import com.couchbase.client.core.logging.CouchbaseLoggerFactory;
import com.couchbase.client.core.logging.RedactableArgument;
import com.couchbase.client.core.tracing.ThresholdLogSpan;
import com.couchbase.client.core.utils.DefaultObjectMapper;
import com.couchbase.client.deps.io.netty.util.internal.shaded.org.jctools.queues.MpscUnboundedArrayQueue;
import io.opentracing.tag.Tags;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

public class ThresholdLogReporter {
    private static final CouchbaseLogger LOGGER = CouchbaseLoggerFactory.getInstance(ThresholdLogReporter.class);
    private static final AtomicInteger REPORTER_ID = new AtomicInteger();
    private static final long MIN_LOG_INTERVAL = TimeUnit.SECONDS.toNanos(1L);
    public static final String SERVICE_KV = "kv";
    public static final String SERVICE_N1QL = "n1ql";
    public static final String SERVICE_FTS = "search";
    public static final String SERVICE_VIEW = "view";
    public static final String SERVICE_ANALYTICS = "analytics";
    public static final String KEY_TOTAL_MICROS = "total_us";
    public static final String KEY_DISPATCH_MICROS = "last_dispatch_us";
    public static final String KEY_ENCODE_MICROS = "encode_us";
    public static final String KEY_DECODE_MICROS = "decode_us";
    public static final String KEY_SERVER_MICROS = "server_us";
    private final Queue<ThresholdLogSpan> overThresholdQueue;
    private final long kvThreshold;
    private final long n1qlThreshold;
    private final long viewThreshold;
    private final long ftsThreshold;
    private final long analyticsThreshold;
    private final long logIntervalNanos;
    private final int sampleSize;
    private final boolean pretty;
    private final Thread worker;
    private volatile boolean running;

    public static Builder builder() {
        return new Builder();
    }

    public static ThresholdLogReporter disabled() {
        return ThresholdLogReporter.builder().logInterval(0L, TimeUnit.SECONDS).build();
    }

    public static ThresholdLogReporter create() {
        return ThresholdLogReporter.builder().build();
    }

    ThresholdLogReporter(Builder builder) {
        this.logIntervalNanos = builder.logIntervalUnit.toNanos(builder.logInterval);
        this.sampleSize = builder.sampleSize;
        if (this.logIntervalNanos > 0L && this.logIntervalNanos < this.minLogInterval()) {
            throw new IllegalArgumentException("The log interval needs to be either 0 or greater than " + MIN_LOG_INTERVAL + " micros");
        }
        this.overThresholdQueue = new MpscUnboundedArrayQueue<ThresholdLogSpan>(builder.spanQueueSize);
        this.kvThreshold = builder.kvThreshold;
        this.analyticsThreshold = builder.analyticsThreshold;
        this.ftsThreshold = builder.ftsThreshold;
        this.viewThreshold = builder.viewThreshold;
        this.n1qlThreshold = builder.n1qlThreshold;
        this.pretty = builder.pretty;
        this.running = true;
        if (this.logIntervalNanos > 0L) {
            this.worker = new Thread(new Worker());
            this.worker.setDaemon(true);
            this.worker.start();
        } else {
            this.worker = null;
            LOGGER.debug("ThresholdLogReporter disabled via config.");
        }
    }

    long minLogInterval() {
        return MIN_LOG_INTERVAL;
    }

    public void report(ThresholdLogSpan span) {
        if (this.isOverThreshold(span) && !this.overThresholdQueue.offer(span)) {
            LOGGER.debug("Could not enqueue span {} for over threshold reporting, discarding.", (Object)span);
        }
    }

    private boolean isOverThreshold(ThresholdLogSpan span) {
        String service = (String)span.tag(Tags.PEER_SERVICE.getKey());
        if (SERVICE_KV.equals(service)) {
            return span.durationMicros() >= this.kvThreshold;
        }
        if (SERVICE_N1QL.equals(service)) {
            return span.durationMicros() >= this.n1qlThreshold;
        }
        if (SERVICE_VIEW.equals(service)) {
            return span.durationMicros() >= this.viewThreshold;
        }
        if (SERVICE_FTS.equals(service)) {
            return span.durationMicros() >= this.ftsThreshold;
        }
        if (SERVICE_ANALYTICS.equals(service)) {
            return span.durationMicros() >= this.analyticsThreshold;
        }
        return false;
    }

    public void shutdown() {
        this.running = false;
        if (this.worker != null) {
            this.worker.interrupt();
        }
    }

    void logOverThreshold(List<Map<String, Object>> toLog) {
        try {
            String result = this.pretty ? DefaultObjectMapper.prettyWriter().writeValueAsString(toLog) : DefaultObjectMapper.writer().writeValueAsString(toLog);
            LOGGER.warn("Operations over threshold: {}", (Object)result);
        }
        catch (Exception ex) {
            LOGGER.warn("Could not write threshold log.", ex);
        }
    }

    class Worker
    implements Runnable {
        private final long workerSleepMs = Long.parseLong(System.getProperty("com.couchbase.thresholdLogReporterSleep", "100"));
        private final SortedSet<ThresholdLogSpan> kvThresholdSet = new TreeSet<ThresholdLogSpan>();
        private final SortedSet<ThresholdLogSpan> n1qlThresholdSet = new TreeSet<ThresholdLogSpan>();
        private final SortedSet<ThresholdLogSpan> viewThresholdSet = new TreeSet<ThresholdLogSpan>();
        private final SortedSet<ThresholdLogSpan> ftsThresholdSet = new TreeSet<ThresholdLogSpan>();
        private final SortedSet<ThresholdLogSpan> analyticsThresholdSet = new TreeSet<ThresholdLogSpan>();
        private int kvThresholdCount = 0;
        private int n1qlThresholdCount = 0;
        private int viewThresoldCount = 0;
        private int ftsThresholdCount = 0;
        private int analyticsThresholdCount = 0;
        private long lastThresholdLog;
        private boolean hasThresholdWritten;

        Worker() {
        }

        @Override
        public void run() {
            Thread.currentThread().setName("cb-tracing-" + REPORTER_ID.incrementAndGet());
            while (ThresholdLogReporter.this.running) {
                try {
                    this.handleOverThresholdQueue();
                    Thread.sleep(this.workerSleepMs);
                }
                catch (InterruptedException ex) {
                    if (!ThresholdLogReporter.this.running) {
                        return;
                    }
                    Thread.currentThread().interrupt();
                }
                catch (Exception ex) {
                    LOGGER.warn("Got exception on slow operation reporter, ignoring.", ex);
                }
            }
        }

        private void handleOverThresholdQueue() {
            long now = System.nanoTime();
            if (now > this.lastThresholdLog + ThresholdLogReporter.this.logIntervalNanos) {
                this.prepareAndlogOverThreshold();
                this.lastThresholdLog = now;
            }
            ThresholdLogSpan span;
            while ((span = (ThresholdLogSpan)ThresholdLogReporter.this.overThresholdQueue.poll()) != null) {
                String service = (String)span.tag(Tags.PEER_SERVICE.getKey());
                if (ThresholdLogReporter.SERVICE_KV.equals(service)) {
                    this.updateSet(this.kvThresholdSet, span);
                    ++this.kvThresholdCount;
                    continue;
                }
                if (ThresholdLogReporter.SERVICE_N1QL.equals(service)) {
                    this.updateSet(this.n1qlThresholdSet, span);
                    ++this.n1qlThresholdCount;
                    continue;
                }
                if (ThresholdLogReporter.SERVICE_VIEW.equals(service)) {
                    this.updateSet(this.viewThresholdSet, span);
                    ++this.viewThresoldCount;
                    continue;
                }
                if (ThresholdLogReporter.SERVICE_FTS.equals(service)) {
                    this.updateSet(this.ftsThresholdSet, span);
                    ++this.ftsThresholdCount;
                    continue;
                }
                if (ThresholdLogReporter.SERVICE_ANALYTICS.equals(service)) {
                    this.updateSet(this.analyticsThresholdSet, span);
                    ++this.analyticsThresholdCount;
                    continue;
                }
                LOGGER.warn("Unknown service in span {}", (Object)service);
            }
            return;
        }

        private void prepareAndlogOverThreshold() {
            if (!this.hasThresholdWritten) {
                return;
            }
            this.hasThresholdWritten = false;
            ArrayList<Map<String, Object>> output = new ArrayList<Map<String, Object>>();
            if (!this.kvThresholdSet.isEmpty()) {
                output.add(this.convertThresholdSet(this.kvThresholdSet, this.kvThresholdCount, ThresholdLogReporter.SERVICE_KV));
                this.kvThresholdSet.clear();
                this.kvThresholdCount = 0;
            }
            if (!this.n1qlThresholdSet.isEmpty()) {
                output.add(this.convertThresholdSet(this.n1qlThresholdSet, this.n1qlThresholdCount, ThresholdLogReporter.SERVICE_N1QL));
                this.n1qlThresholdSet.clear();
                this.n1qlThresholdCount = 0;
            }
            if (!this.viewThresholdSet.isEmpty()) {
                output.add(this.convertThresholdSet(this.viewThresholdSet, this.viewThresoldCount, ThresholdLogReporter.SERVICE_VIEW));
                this.viewThresholdSet.clear();
                this.viewThresoldCount = 0;
            }
            if (!this.ftsThresholdSet.isEmpty()) {
                output.add(this.convertThresholdSet(this.ftsThresholdSet, this.ftsThresholdCount, ThresholdLogReporter.SERVICE_FTS));
                this.ftsThresholdSet.clear();
                this.ftsThresholdCount = 0;
            }
            if (!this.analyticsThresholdSet.isEmpty()) {
                output.add(this.convertThresholdSet(this.analyticsThresholdSet, this.analyticsThresholdCount, ThresholdLogReporter.SERVICE_ANALYTICS));
                this.analyticsThresholdSet.clear();
                this.analyticsThresholdCount = 0;
            }
            ThresholdLogReporter.this.logOverThreshold(output);
        }

        private Map<String, Object> convertThresholdSet(SortedSet<ThresholdLogSpan> set, int count, String ident) {
            HashMap<String, Object> output = new HashMap<String, Object>();
            ArrayList top = new ArrayList();
            for (ThresholdLogSpan span : set) {
                String server_duration;
                String dispatch_duration;
                String encode_duration;
                String decode_duration;
                String localId;
                String operationName;
                HashMap<String, Object> entry = new HashMap<String, Object>();
                entry.put(ThresholdLogReporter.KEY_TOTAL_MICROS, span.durationMicros());
                String spanId = span.request().operationId();
                if (spanId != null) {
                    entry.put("last_operation_id", spanId);
                }
                if ((operationName = span.operationName()) != null) {
                    entry.put("operation_name", operationName);
                }
                String local = span.request().lastLocalSocket();
                String peer = span.request().lastRemoteSocket();
                if (local != null) {
                    entry.put("last_local_address", RedactableArgument.system(local).toString());
                }
                if (peer != null) {
                    entry.put("last_remote_address", RedactableArgument.system(peer).toString());
                }
                if ((localId = span.request().lastLocalId()) != null) {
                    entry.put("last_local_id", RedactableArgument.system(localId).toString());
                }
                if ((decode_duration = span.getBaggageItem(ThresholdLogReporter.KEY_DECODE_MICROS)) != null) {
                    entry.put(ThresholdLogReporter.KEY_DECODE_MICROS, Long.parseLong(decode_duration));
                }
                if ((encode_duration = span.getBaggageItem(ThresholdLogReporter.KEY_ENCODE_MICROS)) != null) {
                    entry.put(ThresholdLogReporter.KEY_ENCODE_MICROS, Long.parseLong(encode_duration));
                }
                if ((dispatch_duration = span.getBaggageItem(ThresholdLogReporter.KEY_DISPATCH_MICROS)) != null) {
                    entry.put(ThresholdLogReporter.KEY_DISPATCH_MICROS, Long.parseLong(dispatch_duration));
                }
                if ((server_duration = span.getBaggageItem(ThresholdLogReporter.KEY_SERVER_MICROS)) != null) {
                    entry.put(ThresholdLogReporter.KEY_SERVER_MICROS, Long.parseLong(server_duration));
                }
                top.add(entry);
            }
            output.put("service", ident);
            output.put("count", count);
            output.put("top", top);
            return output;
        }

        private void updateSet(SortedSet<ThresholdLogSpan> set, ThresholdLogSpan span) {
            set.add(span);
            while (set.size() > ThresholdLogReporter.this.sampleSize) {
                set.remove(set.first());
            }
            this.hasThresholdWritten = true;
        }
    }

    public static class Builder {
        private static final long DEFAULT_LOG_INTERVAL = 10L;
        private static final TimeUnit DEFAULT_LOG_INTERVAL_UNIT = TimeUnit.SECONDS;
        private static final int DEFAULT_SPAN_QUEUE_SIZE = 1024;
        private static final long DEFAULT_KV_THRESHOLD = TimeUnit.MILLISECONDS.toMicros(500L);
        private static final long DEFAULT_N1QL_THRESHOLD = TimeUnit.SECONDS.toMicros(1L);
        private static final long DEFAULT_VIEW_THRESHOLD = TimeUnit.SECONDS.toMicros(1L);
        private static final long DEFAULT_FTS_THRESHOLD = TimeUnit.SECONDS.toMicros(1L);
        private static final long DEFAULT_ANALYTICS_THRESHOLD = TimeUnit.SECONDS.toMicros(1L);
        private static final int DEFAULT_SAMPLE_SIZE = 10;
        private static final boolean DEFAULT_PRETTY = false;
        private long logInterval = 10L;
        private TimeUnit logIntervalUnit = DEFAULT_LOG_INTERVAL_UNIT;
        private int spanQueueSize = 1024;
        private int sampleSize = 10;
        private boolean pretty = false;
        private long kvThreshold = DEFAULT_KV_THRESHOLD;
        private long n1qlThreshold = DEFAULT_N1QL_THRESHOLD;
        private long viewThreshold = DEFAULT_VIEW_THRESHOLD;
        private long ftsThreshold = DEFAULT_FTS_THRESHOLD;
        private long analyticsThreshold = DEFAULT_ANALYTICS_THRESHOLD;

        public ThresholdLogReporter build() {
            return new ThresholdLogReporter(this);
        }

        public Builder logInterval(long interval, TimeUnit unit) {
            this.logInterval = interval;
            this.logIntervalUnit = unit;
            return this;
        }

        public Builder spanQueueSize(int spanQueueSize) {
            this.spanQueueSize = spanQueueSize;
            return this;
        }

        public Builder kvThreshold(long kvThreshold, TimeUnit timeUnit) {
            this.kvThreshold = timeUnit.toMicros(kvThreshold);
            return this;
        }

        public Builder n1qlThreshold(long n1qlThreshold, TimeUnit timeUnit) {
            this.n1qlThreshold = timeUnit.toMicros(n1qlThreshold);
            return this;
        }

        public Builder viewThreshold(long viewThreshold, TimeUnit timeUnit) {
            this.viewThreshold = timeUnit.toMicros(viewThreshold);
            return this;
        }

        public Builder ftsThreshold(long ftsThreshold, TimeUnit timeUnit) {
            this.ftsThreshold = timeUnit.toMicros(ftsThreshold);
            return this;
        }

        public Builder analyticsThreshold(long analyticsThreshold, TimeUnit timeUnit) {
            this.analyticsThreshold = timeUnit.toMicros(analyticsThreshold);
            return this;
        }

        public Builder sampleSize(int sampleSize) {
            this.sampleSize = sampleSize;
            return this;
        }

        public Builder pretty(boolean pretty) {
            this.pretty = pretty;
            return this;
        }
    }
}

