/*
 * Copyright (C) 2012 Engineering Ingegneria Informatica S.p.A.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gcube.accounting.usagetracker.persistence;

import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.regex.Pattern;

import org.apache.log4j.Logger;
import org.gcube.accounting.usagetracker.configuration.Configuration;
import org.gcube.accounting.datamodel.RawUsageRecord;
import org.gcube.accounting.datamodel.UsageRecord;
import org.gcube.accounting.datamodel.query.QueryClause;
import org.gcube.accounting.datamodel.query.QueryOperator;
import org.gcube.accounting.exception.InvalidValueException;
import org.gcube.accounting.exception.NotFoundException;

import com.mongodb.BasicDBList;
import com.mongodb.BasicDBObject;
import com.mongodb.DB;
import com.mongodb.DBCollection;
import com.mongodb.DBCursor;
import com.mongodb.DBObject;
import com.mongodb.Mongo;
import com.mongodb.WriteConcern;


public class UsageTrackerDB {

	// a db-specific prefix for resource-specific properties
	private static final String RSP_PREFIX = "rsp:";

	// a db-specific prefix for custom properties
	private static final String CP_PREFIX = "cp:";
	
	// encoding for special characters
	
	private static final String DOLLAR = "\\\\DOL";
	private static final String DOT = "\\\\DOT";
	private static final String BACKSLASH = "\\\\\\\\";

	/**
	 * The singleton instance
	 */
	private static UsageTrackerDB utdb;
	private Mongo mongo;
	private DB db;

	/**
	 * The list of mandatory fields 
	 */
	private static DBField[] baseFields = {
		DBField.RESOURCE_TYPE,
		DBField.CONSUMER_ID,
		DBField.FULLY_QUALIFIED_CONSUMER_ID,
		DBField.START_TIME, 
		DBField.END_TIME, 
		DBField.CREATOR_ID, 
		DBField.UR_ID,
		DBField.CREATE_TIME,
		DBField.MODIFY_TIME,
		DBField.RESOURCE_OWNER
	};

	private static Logger logger = Logger.getLogger(UsageTrackerDB.class);

	/**
	 * Constructor is private to implement singleton
	 * @param hostname
	 * @param dbname
	 * @throws UnknownHostException
	 */
	private UsageTrackerDB(String hostname, String dbname, int port)
			throws UnknownHostException {
		this.mongo = new Mongo(hostname, port);
		
		this.db = mongo.getDB(dbname);
	}
	
	/**
	 * Constructor is private to implement singleton
	 * @param hostname
	 * @param dbname
	 * @throws UnknownHostException
	 */
	private UsageTrackerDB(String hostname, String dbname, int port, String username, String password)
			throws UnknownHostException {
		this(hostname, dbname, port);
		this.db.authenticate(username, password.toCharArray());
	}
	
	/**
	 * Check whether the field 'field' is a basic field
	 * 
	 * @param f
	 * @return
	 */
	private boolean isBaseField(String field) {
		for (DBField f : baseFields)
			if (f.toString().equals(field))
				return true;
		return false;
	}

	/**
	 * Singleton
	 * 
	 * @return
	 * @throws UnknownHostException
	 */
	public static UsageTrackerDB getStorage() throws UnknownHostException {
		if (utdb == null) {
			utdb = new UsageTrackerDB(Configuration.getInstance().getDBHost(), Configuration.getInstance()
					.getDBName(), Configuration.getInstance().getDBPort());
		}
		return utdb;
	}

	/**
	 * Create a storage object starting from the UR
	 * @param ur
	 * @return
	 */
	private DBObject createDBObject(UsageRecord ur) {
		BasicDBObject doc = new BasicDBObject();
		doc.put(DBField.UR_ID.toString(), ur.getId());
		if(ur.getCreateTime()!=null)
			doc.put(DBField.CREATE_TIME.toString(), ur.getCreateTime().getTimeInMillis());
		if(ur.getModifyTime()!=null)
			doc.put(DBField.MODIFY_TIME.toString(), ur.getModifyTime().getTimeInMillis());
		doc.put(DBField.CREATOR_ID.toString(), ur.getCreatorId());
		doc.put(DBField.CONSUMER_ID.toString(), ur.getConsumerId());
		doc.put(DBField.FULLY_QUALIFIED_CONSUMER_ID.toString(), ur.getFullyQualifiedConsumerId());
		doc.put(DBField.RESOURCE_TYPE.toString(), ur.getResourceType());
		doc.put(DBField.RESOURCE_OWNER.toString(), ur.getResourceOwner());

		if (ur.getStartTime() != null)
			doc.put(DBField.START_TIME.toString(), ur.getStartTime().getTimeInMillis());
		if (ur.getEndTime() != null)
			doc.put(DBField.END_TIME.toString(), ur.getEndTime().getTimeInMillis());
		Map<String, String> properties = ur.getResourceSpecificProperties();
		for (String key : properties.keySet()) {
			if (!doc.keySet().contains(key)) {
				try {
					Long value = Long.parseLong(properties.get(key));
					doc.put(RSP_PREFIX+encodeKey(key), value);
				}
				catch(Exception e) {
					try {
						Double value = Double.parseDouble(properties.get(key));
						doc.put(RSP_PREFIX+encodeKey(key), value);
					}
					catch(Exception e2) {
						doc.put(RSP_PREFIX+encodeKey(key), properties.get(key));
					}
					doc.put(RSP_PREFIX+encodeKey(key), properties.get(key));
				}
			}
		}
		properties = ur.getCustomProperties();
		for (String key : properties.keySet()) {
			if (!doc.keySet().contains(key)) {
				try {
					Long value = Long.parseLong(properties.get(key));
					doc.put(CP_PREFIX+encodeKey(key), value);
				}
				catch(Exception e) {
					try {
						Double value = Double.parseDouble(properties.get(key));
						doc.put(CP_PREFIX+encodeKey(key), value);
					}
					catch(Exception e2) {
						doc.put(CP_PREFIX+encodeKey(key), properties.get(key));
					}
					doc.put(CP_PREFIX+encodeKey(key), properties.get(key));
				}
			}
		}
		return doc;
	}

	/**
	 * Create an UsageRecord object starting from the storage object
	 * @param dbo
	 * @return
	 */
	@SuppressWarnings("unchecked")
	private RawUsageRecord createUsageRecord(DBObject dbo) {
		RawUsageRecord out = new RawUsageRecord();
		out.setId((String) dbo.get(DBField.UR_ID.toString()));
		out.setConsumerId((String) dbo.get(DBField.CONSUMER_ID.toString()));
		out.setFullyQualifiedConsumerId((String)dbo.get(DBField.FULLY_QUALIFIED_CONSUMER_ID.toString()));
		out.setCreatorId((String) dbo.get(DBField.CREATOR_ID.toString()));
		out.setResourceType((String) dbo.get(DBField.RESOURCE_TYPE.toString()));
		out.setResourceOwner((String) dbo.get(DBField.RESOURCE_OWNER.toString()));
		if (dbo.get(DBField.START_TIME.toString()) != null) {
			Calendar c = Calendar.getInstance();
			c.setTimeInMillis((Long)dbo.get(DBField.START_TIME.toString()));
			try {
				out.setStartTime(c);
			}
			catch(InvalidValueException e) {
				logger.error(e.getMessage(), e);
			}
		}
		if (dbo.get(DBField.END_TIME.toString()) != null) {
			Calendar c = Calendar.getInstance();
			c.setTimeInMillis((Long)dbo.get(DBField.END_TIME.toString()));
			try {
				out.setEndTime(c);
			}
			catch(InvalidValueException e) {
				logger.error(e.getMessage(), e);
			}
		}
		if (dbo.get(DBField.CREATE_TIME.toString()) != null) {
			Calendar c = Calendar.getInstance();
			c.setTimeInMillis((Long)dbo.get(DBField.CREATE_TIME.toString()));
			out.setCreateTime(c);
		}
		if (dbo.get(DBField.MODIFY_TIME.toString()) != null) {
			Calendar c = Calendar.getInstance();
			c.setTimeInMillis((Long)dbo.get(DBField.MODIFY_TIME.toString()));
			out.setModifyTime(c);
		}
		for (String key : dbo.keySet()) {
			if (!this.isBaseField(key)) {
				Object value = dbo.get(key);
				if (value instanceof String || value instanceof Double || value instanceof Long) {
					if(key.startsWith(RSP_PREFIX)) {
						//logger.debug("r-s property:<" + key.substring(RSP_PREFIX.length()) + ":" + dbo.get(key).toString()+">");
						out.setResourceSpecificProperty(decodeKey(key.substring(RSP_PREFIX.length())), dbo.get(key).toString());
					}
					if(key.startsWith(CP_PREFIX)) {
						//logger.debug("custom property:<" + key.substring(CP_PREFIX.length()) + ":" + dbo.get(key).toString()+">");
						out.setCustomProperty(decodeKey(key.substring(CP_PREFIX.length())), dbo.get(key).toString());
					}
				}
			}
		}
		return out;
	}

	/**
	 * Computes the nr of URs in the store
	 * @return
	 */
	public long size() {
		return this.getRecordsCollection().count();
//		return db.getCollection(USAGE_RECORDS).count();
	}

	/**
	 * Add the given usage record
	 * @param ur
	 * @throws JMSException 
	 */
	public void addUsageRecord(UsageRecord ur) {
		// set the create time to 'now'
		ur.setCreateTime(Calendar.getInstance());
		// set also the modify time to 'now'
		try {
			ur.setModifyTime(ur.getCreateTime());
		}
		catch(InvalidValueException e) {
			// this should never happen
			logger.warn(e.getMessage(), e);
		}
		// retrieve collection
		DBCollection coll = getRecordsCollection();
		// insert object
		logger.info(String.format("inserting new record %s", ur.getId()));
		coll.insert(this.createDBObject(ur));
		// record the group id
		this.addConsumerGroup(ur.getFullyQualifiedConsumerId());
	}
	
	private DBCollection getRecordsCollection() {
		DBCollection coll = this.db.getCollection(DBField.USAGE_RECORDS.toString());
		logger.debug("enforcing indexes");
		coll.ensureIndex(new BasicDBObject(DBField.UR_ID.toString(), 1).append("unique", true).append("dropDups", true));
		coll.ensureIndex(new BasicDBObject(DBField.FULLY_QUALIFIED_CONSUMER_ID.toString(), 1));
		return coll;
	}
	
	/**
	 * Update the given UR (replacement based on id)
	 * @param ur
	 */
	public void updateUsageRecord(UsageRecord ur) {
		DBCollection coll = getRecordsCollection();
		try {
			// retrieving existing record
			UsageRecord oldUR = this.getUsageRecord(ur.getId());
			// retrieving current consumer group (might not be in new record)
			// NO: updating group is allowed
			// ur.setFullyQualifiedConsumerId(oldUR.getFullyQualifiedConsumerId());
			// avoid clients to change the create time
			ur.setCreateTime(oldUR.getCreateTime());
			// update modify time (to 'now')
			ur.setModifyTime(Calendar.getInstance());
			// update object
			DBObject newUR = this.createDBObject(ur);
			logger.info(String.format("updating record %s", ur.getId()));
			coll.update(new BasicDBObject().append("urId", ur.getId()), newUR);
		}
		catch(Exception e) {
			logger.error(e.getMessage(), e);
			e.printStackTrace();
		}
	}

	/**
	 * Retrieve URs with the given <key,value> pair
	 * @param key
	 * @param value
	 * @return
	 */
	/*
	public Collection<RawUsageRecord> getUsageRecordByProperty(String key, String value) {
		Collection<RawUsageRecord> out = new Vector<RawUsageRecord>();
		BasicDBObject query = new BasicDBObject();
		query.put(key, value);
		DBCollection coll = this.db.getCollection(USAGE_RECORDS);
		DBCursor cur = coll.find(query);
		while(cur.hasNext()) {
			DBObject dbo = cur.next();
			out.add(this.createUsageRecord(dbo));
		}
		return out;
	}
	*/

	/*
	public Collection<RawUsageRecord> getUsageRecordByProperties(Map<String, String> props) {
		Collection<RawUsageRecord> out = new Vector<RawUsageRecord>();
		BasicDBObject query = new BasicDBObject();
		for(String key:props.keySet()) {
			logger.debug("adding " + key + ":"+props.get(key));
			query.put(key, props.get(key));
		}
		DBCollection coll = this.db.getCollection(USAGE_RECORDS);
		DBCursor cur = coll.find(query);
		while(cur.hasNext()) {
			DBObject dbo = cur.next();
			out.add(this.createUsageRecord(dbo));
		}
		return out;
	}
	 */

	private BasicDBObject getDBObject(String key, String op, String value) {
		try {
			Long d = Long.parseLong(value);
			return new BasicDBObject(op, d);
		}
		catch(NumberFormatException e) {
			try {
				Double d = Double.parseDouble(value);
				return new BasicDBObject(op, d);
			}
			catch(NumberFormatException e2) {
				return new BasicDBObject(op, value);
			}
		}
	}

	public Collection<RawUsageRecord> getUsageRecordByQuery(Map<String, QueryClause> props) {
		Collection<RawUsageRecord> out = new Vector<RawUsageRecord>();
		BasicDBObject query = new BasicDBObject();
//		List<String> consumerGroupQueryValues = new ArrayList<String>();
		for(QueryClause clause:props.values()) {
			logger.debug("adding " + clause.getKey() + ":" + clause.getValue());
			if(QueryOperator.eq.equals(clause.getOperator())) {
				//special case for consumer group
				if(clause.getKey().equals(DBField.FULLY_QUALIFIED_CONSUMER_ID.toString())){
					// ensure index on consumerGroup
					DBCollection coll = this.getRecordsCollection();

					// prepare clause
					
					//String pattern = "^"+clause.getValue().replace(".", ",");
					String pattern = "^"+clause.getValue().replace(".", "\\.");
					Pattern match = Pattern.compile(pattern);
					query.put(clause.getKey(), match);
					
					// version 1
					/*
					String consumerGroupQuery = "";
					StringTokenizer st = new StringTokenizer(clause.getValue(), ".");
					if(st.hasMoreTokens()){
						while(st.hasMoreTokens()) {
							String currentValue = st.nextToken();
							consumerGroupQuery+=currentValue;
							
								consumerGroupQuery+=",";
						}
						// trim out last ','
						consumerGroupQuery=consumerGroupQuery.substring(0, consumerGroupQuery.length()-1);

						DBCollection coll = this.db.getCollection(USAGE_RECORDS);
						coll.ensureIndex(new BasicDBObject("id", 1).append("unique", true));
						coll.createIndex(new BasicDBObject("consumerGroup", 1));
						String pattern = "^"+consumerGroupQuery; 
						Pattern match = Pattern.compile(pattern);
						
						query.put(clause.getKey(), match);
					}
					*/
				}
				else{
					try {
						Long d = Long.parseLong(clause.getValue());
						query.put(clause.getKey(), d);
					}
					catch(NumberFormatException e) {
						try {
							Double d = Double.parseDouble(clause.getValue());
							query.put(clause.getKey(), d);
						}
						catch(NumberFormatException e2) {
							/*try {
								StringTokenizer st = new StringTokenizer(clause.getValue(), ",");
								if(st.hasMoreTokens()){

									List<String> values = new ArrayList<String>();
									while(st.hasMoreTokens()) {
										String currentValue = st.nextToken();
										values.add(currentValue);
									}
									BasicDBObject in = new BasicDBObject("$in", values);
									query.put(clause.getKey(), in);
								}
							}
							catch (Exception e3) {*/
							query.put(clause.getKey(), clause.getValue());
							//}

						}
					}
				}
			}
			else if(QueryOperator.lt.equals(clause.getOperator())) {
				query.put(clause.getKey(), getDBObject(clause.getKey(), "$lt", clause.getValue()));
			}
			else if(QueryOperator.lte.equals(clause.getOperator())) {
				query.put(clause.getKey(), getDBObject(clause.getKey(), "$lte", clause.getValue()));
			}
			else if(QueryOperator.gt.equals(clause.getOperator())) {
				query.put(clause.getKey(), getDBObject(clause.getKey(), "$gt", clause.getValue()));
			}
			else if(QueryOperator.gte.equals(clause.getOperator())) {
				query.put(clause.getKey(), getDBObject(clause.getKey(), "$gte", clause.getValue()));
			}
			else {
				logger.warn("operator '" + clause.getOperator() + "' is not yet supported");
			}
		}

		DBCollection coll = this.db.getCollection(DBField.USAGE_RECORDS.toString());

		DBCursor cursor = coll.find(query);
		
		
		
//								while(cursor.hasNext()) {
//									DBObject dbo = cursor.next();
//									@SuppressWarnings({ "unchecked", "rawtypes" })
//									List<String> consumerGroup = (List) dbo.get("consumerGroup");
//									int consumerGroupSize = consumerGroup.size();
//									if(valuesSize>consumerGroupSize)
//										continue;
//									consumerGroupSize = consumerGroup.size();
//									int i;
//									for(i=0; i<values.size(); i++) {
//										if(!values.get(i).equals(consumerGroup.get(i)))
//											break;
//									}
//									if(i==values.size()) {
//										out.add(this.createUsageRecord(dbo));
//									}
//								}
//								logger.debug("#record : "+out.size());
//								return out;
		
		
		
//		int valuesSize = consumerGroupQueryValues.size(); 
		logger.debug("found " + cursor.size() + " records");
		while(cursor.hasNext()) {
			DBObject dbo = cursor.next();
			out.add(this.createUsageRecord(dbo));
			/*DBObject dbo = cursor.next();
			@SuppressWarnings({ "unchecked", "rawtypes" })
			List<String> consumerGroup = (List) dbo.get("consumerGroup");
			int consumerGroupSize = consumerGroup.size();
			if(consumerGroupQueryValues.size()>consumerGroupSize)
				continue;
			consumerGroupSize = consumerGroup.size();
			int i;
			for(i=0; i<consumerGroupQueryValues.size(); i++) {
				if(!consumerGroupQueryValues.get(i).equals(consumerGroup.get(i)))
					break;
			}
			if(i==consumerGroupQueryValues.size()) {
				out.add(this.createUsageRecord(dbo));
			}*/
		}
		return out;
	}

	/*
	public Collection<UsageRecord> getUsageRecordByXPath(String xpathexpression) throws XPathException {
		XPath xpath = XPathFactory.newInstance().newXPath();
		XPathExpression expr = xpath.compile(xpathexpression);
		return null;
	}
	 */

	/**
	 * Return all records _entirely_ falling in the given time interval
	 * @param user
	 * @param fromDate
	 * @param toDate
	 * @return
	 */
	/*
	public Collection<RawUsageRecord> getUsageRecordsByUserByTime(String user, Calendar fromDate, Calendar toDate) {
		Collection<RawUsageRecord> out = new Vector<RawUsageRecord>();

		logger.info("looking for URs between " + fromDate.getTimeInMillis() + " and " + toDate.getTimeInMillis());

		// 1) completely inside the window
		BasicDBObject query = new BasicDBObject();
		query.put(CONSUMER_ID, user);
		query.put(START_TIME, new BasicDBObject("$gt", fromDate.getTimeInMillis()));
		query.put(END_TIME, new BasicDBObject("$lt", toDate.getTimeInMillis()));
		DBCollection coll = this.db.getCollection(USAGE_RECORDS);
		DBCursor cur = coll.find(query);
		while(cur.hasNext()) {
			DBObject dbo = cur.next();
			out.add(this.createUsageRecord(dbo));
		}

		// 2) completely covering the window
		query = new BasicDBObject();
		query.put(CONSUMER_ID, user);
		query.put(START_TIME, new BasicDBObject("$lt", fromDate.getTimeInMillis()));
		query.put(END_TIME, new BasicDBObject("$gt", toDate.getTimeInMillis()));
		coll = this.db.getCollection(USAGE_RECORDS);
		cur = coll.find(query);
		while(cur.hasNext()) {
			DBObject dbo = cur.next();
			out.add(this.createUsageRecord(dbo));
		}

		// 3) ending in the period
		query = new BasicDBObject();
		query.put(CONSUMER_ID, user);
		query.put(START_TIME, new BasicDBObject("$lt", fromDate.getTimeInMillis()));
		query.put(END_TIME, new BasicDBObject("$gt", fromDate.getTimeInMillis()).append("$lt", toDate.getTimeInMillis()));
		coll = this.db.getCollection(USAGE_RECORDS);
		cur = coll.find(query);
		while(cur.hasNext()) {
			DBObject dbo = cur.next();
			out.add(this.createUsageRecord(dbo));
		}

		// 4) started in the period
		query = new BasicDBObject();
		query.put(CONSUMER_ID, user);
		query.put(START_TIME, new BasicDBObject("$gt", fromDate.getTimeInMillis()).append("$lt", toDate.getTimeInMillis()));
		query.put(END_TIME, new BasicDBObject("$gt", toDate.getTimeInMillis()));
		coll = this.db.getCollection(USAGE_RECORDS);
		cur = coll.find(query);
		while(cur.hasNext()) {
			DBObject dbo = cur.next();
			out.add(this.createUsageRecord(dbo));
		}

		return out;
	}
	*/

	/**
	 * Retrieve the UR with the given ID
	 */
	public UsageRecord getUsageRecord(String id) throws NotFoundException {
		BasicDBObject query = new BasicDBObject();
		query.put(DBField.UR_ID.toString(), id);
		DBCollection coll = this.db.getCollection(DBField.USAGE_RECORDS.toString());
		DBCursor cur = coll.find(query);
		if (cur.hasNext()) {
			DBObject out = cur.next();
			
			return this.createUsageRecord(out);
		}
		throw new NotFoundException("id");
	}

	/**
	 * Delete the UR with the given ID
	 * @param id
	 */
	public void deleteUsageRecord(String id) {
		logger.info(String.format("deleting usage record '%s' ...", id));
		BasicDBObject query = new BasicDBObject();
		query.put(DBField.UR_ID.toString(), id);
		DBCollection coll = this.db.getCollection(DBField.USAGE_RECORDS.toString());
		coll.remove(query, WriteConcern.SAFE);
	}

	/**
	 * Remove all Usage Records
	 */
	public void clear() {
		this.getTrackedUsers();
		// remove all records
		DBCollection coll = this.db.getCollection(DBField.USAGE_RECORDS.toString());
		BasicDBObject query = new BasicDBObject();
		coll.remove(query, WriteConcern.SAFE);
		// remove all groups
		coll = this.db.getCollection(DBField.CONSUMERS_RECORDS.toString());
		query = new BasicDBObject();
		coll.remove(query, WriteConcern.SAFE);
	}

	/**
	 * Return all usage records in the store
	 * @return
	 */
	/*
	public Collection<RawUsageRecord> getAll() {
		Collection<RawUsageRecord> out = new Vector<RawUsageRecord>();
		DBCollection coll = this.db.getCollection(USAGE_RECORDS);
		BasicDBObject query = new BasicDBObject();
		DBCursor cur = coll.find(query);
		while(cur.hasNext()) {
			DBObject dbo = cur.next();
			out.add(this.createUsageRecord(dbo));
		}
		return out;
	}
	*/

	public Collection<String> getTrackedUsers() {
		return this.getDistinct(DBField.CONSUMER_ID.toString());
	}

	public Collection<String> getTrackedResourceTypes() {
		return this.getDistinct(DBField.RESOURCE_TYPE.toString());
	}

	public Collection<String> getTrackedResourceOwners() {
		return this.getDistinct(DBField.RESOURCE_OWNER.toString());
	}

	public Collection<String> getTrackedGroups() {
		return this.getDistinct(DBField.FULLY_QUALIFIED_CONSUMER_ID.toString());
	}

	private Collection<String> getDistinct(String key) {
		List<String> out = new Vector<String>();
		DBCollection coll = this.db.getCollection(DBField.USAGE_RECORDS.toString());
		List items = coll.distinct(key);
		for(Object o:items) {
			if(o!=null)
				out.add(o.toString());
		}
		return out;
	}

	/**
	 * Return the last update time for the given DCI.
	 * FIXME: for the moment the 'end_time' is used since we're using the simulator. Later on, we need to use the 'create_time'
	 * @param dci
	 * @return
	 */
	public Long getLastUpdate(String dci) {
		Long max = 0L;
		BasicDBObject query = new BasicDBObject();
		query.put(DBField.RESOURCE_OWNER.toString(), dci);
		DBCollection coll = this.db.getCollection(DBField.USAGE_RECORDS.toString());
		DBCursor cur = coll.find(query);
		while(cur.hasNext()) {
			DBObject dbo = cur.next();
			Long l = (Long)dbo.get(DBField.END_TIME.toString());
			if(l!=null && l>max)
				max = l;
		}
		return max;
	}

	public void addConsumerGroup(String consumerGroup) {
		List<String> l = new Vector<String>();
		StringTokenizer st = new StringTokenizer(consumerGroup, ".");
		while(st.hasMoreTokens()) {
			l.add(st.nextToken());
		}
		this.addConsumerGroup(l);
	}
	
	public void addConsumerGroup(List<String> consumerGroup) {
		logger.info("adding group: " + consumerGroup.toString());
		DBCollection coll = this.db.getCollection(DBField.CONSUMERS_RECORDS.toString());
		BasicDBObject query = new BasicDBObject();
		query.put(DBField.SUBSET_GROUP.toString(), consumerGroup);
		DBCursor cur = coll.find(query);
		if(!cur.hasNext()){
			BasicDBObject doc = new BasicDBObject();
			doc.put(DBField.SUBSET_GROUP.toString(), consumerGroup);
			coll.insert(doc);
		}	
	}

	public Collection<ArrayList<String>> getConsumersGroup(){
		Collection<ArrayList<String>> consumersGroup = new ArrayList<ArrayList<String>>();
		DBCollection collCG = this.db.getCollection(DBField.CONSUMERS_RECORDS.toString());
		collCG.ensureIndex(new BasicDBObject(DBField.SUBSET_GROUP.toString(), 1).append("unique", true));
		BasicDBObject query = new BasicDBObject();
		DBCursor cursor = collCG.find(query);
		while(cursor.hasNext()){
			DBObject dbo = cursor.next();
			BasicDBList dblist = (BasicDBList) dbo.get(DBField.SUBSET_GROUP.toString());
			ArrayList<String> current = new ArrayList<String>();
			for (Object object : dblist) {
				current.add((String) object);
			}
			consumersGroup.add(current);
			
		}
		return consumersGroup;
	}
	
	private String encodeKey(String key){
		key=key.replaceAll("\\\\", BACKSLASH);
		key=key.replaceAll("\\$", DOLLAR);
		key=key.replaceAll("\\.", DOT);
		return key;
	}
	
	private String decodeKey(String key){
		key=key.replaceAll(DOT, ".");
		key=key.replaceAll(DOLLAR,"\\$");
		key=key.replaceAll(BACKSLASH,"\\\\");
		return key;
	}

}