/*
 * Copyright (C) 2012 Engineering Ingegneria Informatica S.p.A.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gcube.accounting.usagetracker.rest.resources;

import java.net.UnknownHostException;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.PUT;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;

import org.apache.log4j.Logger;
import org.gcube.accounting.usagetracker.persistence.UsageTrackerDB;
import org.gcube.accounting.datamodel.UsageRecord;
import org.gcube.accounting.exception.InvalidValueException;
import org.gcube.accounting.exception.NotAuthorizedException;
import org.gcube.accounting.exception.NotFoundException;
import org.gcube.accounting.exception.rest.BadRequestException;
import org.gcube.accounting.exception.rest.ServerErrorException;
import org.gcube.accounting.security.SecurityManager;
import org.gcube.accounting.security.authz.Action;

public abstract class AbstractRecordResource<T extends UsageRecord> {

	private String id;
	private String userId;
	
//	private AuthorizationManager authz = SecurityManager.getAuthorizationManager();

	private static Logger logger = Logger.getLogger(AbstractRecordResource.class);

	public AbstractRecordResource() {
	}

	public AbstractRecordResource(String id) {
		this.id = id;
	}

	@PUT
	@Consumes(MediaType.APPLICATION_XML)
	public void update(T r) throws NotAuthorizedException, NotFoundException {
		// check if user can update
		if(!SecurityManager.isAllowed(userId,Action.UPDATE_RECORD)){
			logger.info("User "+userId+" doesn't have permission to perform "+Action.UPDATE_RECORD);
			throw new NotAuthorizedException();
		}
		try {
			UsageTrackerDB db = UsageTrackerDB.getStorage();
			String fullyQualifiedConsumerId = db.getUsageRecord(r.getId()).getFullyQualifiedConsumerId();
			// check if user can update the given record
			if(!SecurityManager.isAllowed(userId,Action.UPDATE_RECORD,fullyQualifiedConsumerId)){
				logger.info("User "+userId+" doesn't have permission to perform "+Action.UPDATE_RECORD+" on record "+r.getId());
				throw new NotAuthorizedException();
			}
			// also check that the new value for fqci are ok for the user
			if(!SecurityManager.isAllowed(userId,Action.UPDATE_RECORD,r.getFullyQualifiedConsumerId())) {
				logger.info("User "+userId+" doesn't have permission to perform "+Action.UPDATE_RECORD+" on record "+r.getId());
				throw new NotAuthorizedException();
			}
			// if authn is enabled, creatorId must be the same as the logged client
			if(SecurityManager.isAuthnEnabled()) {
				r.setCreatorId(this.getUserId());
				logger.debug("forcing creator id to '" + r.getCreatorId() + "'");
			}
			// check record fields
			try {
				this.validate(r);
			}
			catch(InvalidValueException e) {
				logger.info("invalid data for upload: " + e.getMessage());
				throw new BadRequestException(e.getMessage());
			}
			db.updateUsageRecord(r);
		} catch (UnknownHostException e) {
			logger.error(e.getMessage(), e);
			throw new ServerErrorException(e.getMessage());
		} catch(NotFoundException e) {
			logger.info(e.getMessage(), e);
			throw new org.gcube.accounting.exception.rest.NotFoundException(e.getMessage());
		}
	}

	@DELETE
	public void delete() throws NotAuthorizedException, NotFoundException {
		logger.info("deleting usage record...");
		// check whether the user can delete records
		if(!SecurityManager.isAllowed(userId,Action.DELETE_RECORD)){
			logger.info("User "+userId+" doesn't have permission to perform "+Action.DELETE_RECORD);
			throw new NotAuthorizedException();
		}
		try {
			UsageTrackerDB db = UsageTrackerDB.getStorage();
			String fullyQualifiedConsumerId = db.getUsageRecord(this.id).getFullyQualifiedConsumerId();
			if(!SecurityManager.isAllowed(userId,Action.DELETE_RECORD,fullyQualifiedConsumerId)){
				logger.info("User "+userId+" doesn't have permission to perform "+Action.DELETE_RECORD+" on record "+this.id);
				throw new NotAuthorizedException();
			}
			db.deleteUsageRecord(this.id);
		} catch (UnknownHostException e) {
			logger.error(e.getMessage(), e);
			throw new ServerErrorException(e.getMessage());
		} catch(NotFoundException e) {
			logger.info(e.getMessage(), e);
			throw new org.gcube.accounting.exception.rest.NotFoundException(e.getMessage());
		}
	}

	@GET
	@Produces({MediaType.APPLICATION_XML})
	public T getRecord() throws NotAuthorizedException {
		// check the user can read
		if(!SecurityManager.isAllowed(userId,Action.READ_RECORD)){
			logger.info("User "+userId+" doesn't have permission to perform "+Action.READ_RECORD);
			throw new NotAuthorizedException();
		}
		try {
			UsageTrackerDB db = UsageTrackerDB.getStorage();
			UsageRecord r = db.getUsageRecord(this.id);

			// check the user can retrieve this record
			String fullyQualifiedConsumerId = r.getFullyQualifiedConsumerId();
			if(!SecurityManager.isAllowed(userId,Action.READ_RECORD,fullyQualifiedConsumerId)){
				logger.info("User "+userId+" doesn't have permission to perform "+Action.READ_RECORD+" on record "+this.id);
				throw new NotAuthorizedException();
			}
			// check the record has the correct type
			if(!this.isCorrectType(r)) {
				String message = "record " + r.getId() + " does not match the requested resource type"; 
				logger.info(message);
				throw new org.gcube.accounting.exception.rest.NotFoundException(message);
			}
			// create and return record
			return this.createRecord(r);
		} catch (UnknownHostException e) {
			logger.error(e.getMessage(), e);
			throw new ServerErrorException(e.getMessage());
		} catch (NotFoundException e) {
			logger.info(e.getMessage(), e);
			throw new org.gcube.accounting.exception.rest.NotFoundException(e.getMessage());
		} catch (InvalidValueException e) {
			logger.info(e.getMessage(), e);
			throw new org.gcube.accounting.exception.rest.NotFoundException(e.getMessage());
		}
	}

	protected abstract void validate(T record) throws InvalidValueException;

	protected abstract T createRecord(UsageRecord record) throws InvalidValueException;

	protected abstract boolean isCorrectType(UsageRecord record);

	public String getUserId() {
		return userId;
	}

	public void setUserId(String userId) {
		this.userId = userId;
	}



}
