

package org.gcube.accounting.usagetracker.rest.resources;

import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.Vector;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;

import org.apache.log4j.Logger;
import org.gcube.accounting.usagetracker.configuration.Configuration;
import org.gcube.accounting.usagetracker.membership.MembershipRestAPI;
import org.gcube.accounting.usagetracker.persistence.UsageTrackerDB;
import org.gcube.accounting.datamodel.Group;
import org.gcube.accounting.datamodel.RawUsageRecord;
import org.gcube.accounting.datamodel.UsageRecord;
import org.gcube.accounting.datamodel.query.QueryClause;
import org.gcube.accounting.datamodel.query.QueryHelper;
import org.gcube.accounting.exception.InvalidValueException;
import org.gcube.accounting.exception.QueryFormatException;
import org.gcube.accounting.exception.rest.BadRequestException;
import org.gcube.accounting.exception.rest.NotAuthorizedException;
import org.gcube.accounting.exception.rest.ServerErrorException;
import org.gcube.accounting.security.SecurityManager;
import org.gcube.accounting.security.authz.Action;


public abstract class AbstractRecordsResource<T extends UsageRecord> {

	protected @Context HttpServletRequest request;

	private static Logger logger = Logger.getLogger(AbstractRecordsResource.class);

	/**
	 * Retrieve membership information from membership service (if available)
	 * 
	 * @param consumerId
	 * @return
	 */
	private List<String> getGroupPathFor(String consumerId) {

		List<String> path = new ArrayList<String>();
		path.add(consumerId);
		
		if(Configuration.getInstance().isMembershipServiceEnabled()) {
			String membershipServiceURL = Configuration.getInstance().getMembershipServiceURL();
			MembershipRestAPI membershipService = new MembershipRestAPI(membershipServiceURL);
			if(Configuration.getInstance().isMembershipServiceAuthnEnabled()) {
				logger.debug("accessing the membership service with username/password");
				String userName = Configuration.getInstance().getMembershipUsername();
				String password = Configuration.getInstance().getMembershipPassword();
				membershipService.setUserToken(userName, password);
			}
			else {
				logger.debug("accessing the membership service anonymously");
			}
			Group userGroup = membershipService.getGroups(consumerId);

			if(userGroup!=null){ 
				logger.debug("Membership found: " + userGroup.toString());
				path.addAll(userGroup.getPath());
			}
			else {
				logger.warn("Membership not found for consumer "+consumerId);
				path.add(Configuration.getInstance().getDefaultGroup());
			}
		}
		else {
			logger.info("No membership service available. Using consumerId");
			path.add(Configuration.getInstance().getDefaultGroup());
		}
		Collections.reverse(path);
		return path;
	}

	/**
	 * Convert a list of strings to a dotted notation. For example ["a","b","c"]
	 * becomes "a.b.c"
	 * 
	 * @param groupPath
	 * @return
	 */
	private String getStringPath(List<String> groupPath) {
		String groupPathString = "";
		for (String string : groupPath)
			groupPathString+="."+string;
		groupPathString=groupPathString.substring(1);
		return groupPathString;
	}
	
	@POST
	@Consumes(MediaType.APPLICATION_XML)
	public String insertUsageRecord(T record) throws NotAuthorizedException {
		logger.info("inserting new usage record...");
		// check if user can write
		if(!SecurityManager.isAllowed(this.getUserId(),Action.WRITE_RECORD)){
			logger.info("User "+this.getUserId()+" doesn't have permission to perform "+Action.WRITE_RECORD);
			throw new NotAuthorizedException();
		}
		try {
			// check the ur contains a resource type
			if(record.getResourceType()==null||record.getResourceType().isEmpty()) {
				String message = "unable to insert record: a 'resourceType' must be set";
				logger.warn(message);
				throw new BadRequestException(message);
			}
			// set fully qualified consumer id
			UsageTrackerDB db = UsageTrackerDB.getStorage();
			// only if not already provided in the record
			if(record.getFullyQualifiedConsumerId()==null) {
				// retrieve the group path up to the consumer (included)
				List<String> groupPath = this.getGroupPathFor(record.getConsumerId());
				// convert it to string (e.g. "org,example.john")
				String groupPathString = this.getStringPath(groupPath);
				// update the usage record
				record.setFullyQualifiedConsumerId(groupPathString);
			}
			// check if user can write in the given target
			if(!SecurityManager.isAllowed(this.getUserId(),Action.WRITE_RECORD,record.getFullyQualifiedConsumerId())){
				logger.info("User "+this.getUserId()+" doesn't have permission to perform '"+Action.WRITE_RECORD + "' over " + record.getFullyQualifiedConsumerId());
				throw new NotAuthorizedException();
			}
			// always override ids set by clients
			record.setId(UUID.randomUUID().toString());
			logger.debug("setting record id to '" + record.getId() + "'");
			// if authn is enabled, creatorId must be the same as the logged client
			if(SecurityManager.isAuthnEnabled()) {
				record.setCreatorId(this.getUserId());
				logger.debug("forcing creator id to '" + record.getCreatorId() + "'");
			}
			

			
			// check record fields
			try {
				this.validate(record);
			}
			catch(InvalidValueException e) {
				logger.info("invalid data for upload: " + e.getMessage());
				throw new BadRequestException(e.getMessage());
			}
			// upload the group to the database
			//			db.addConsumerGroup(groupPath);
			// upload the record to the db
			db.addUsageRecord(record);
			logger.info("record added: " + record.getId());
			return record.getId();
		} catch (UnknownHostException e) {
			logger.error(e.getMessage(), e);
			throw new ServerErrorException(e.getMessage());
		}
	}
	
	protected abstract void validate(T record) throws InvalidValueException;

	public Collection<T> getRecords(Map<String, QueryClause> clauses) {
		try {
			UsageTrackerDB db = UsageTrackerDB.getStorage();
			return this.convert(db.getUsageRecordByQuery(clauses));
		} catch (UnknownHostException e) {
			logger.error(e.getMessage(), e);
			throw new ServerErrorException(e.getMessage());
		}
	}
	
//	public AggregatedInformation getAggregatedRecords(Map<String, QueryClause> clauses) {
//		try {
//			UsageTrackerDB db = UsageTrackerDB.getStorage();
//			return db.getAggregatedUsageRecordsByQuery(clauses);
//		} catch (UnknownHostException e) {
//			logger.error(e.getMessage(), e);
//			throw new ServerErrorException(e.getMessage());
//		}
//	}

	private List<QueryClause> getTypeSpecificClause() {
		List<QueryClause> out = new Vector<QueryClause>();
		QueryClause typeClause = this.getTypeClause();
		if(typeClause!=null)
			out.add(typeClause);
		return out;
	}

	protected QueryClause getTypeClause() {
		return null;
	}

//	@GET
//	@Produces({MediaType.APPLICATION_XML})
//	public AggregatedInformation getAggregatedInformation(@QueryParam("query") String query) throws NotAuthorizedException {
//		// check if user can read
//		if(!SecurityManager.isAllowed(this.getUserId(),Action.READ_RECORD)){
//			logger.info("User "+this.getUserId()+" doesn't have permission to perform "+Action.READ_RECORD);
//			throw new NotAuthorizedException();
//		}
//		try {
////logger.debug("query inside getRecords = "+query+" query.size() = "+query.size());
//			Map<String, QueryClause> params = QueryHelper.parseQuery(query);
//			// add type-specific clauses
//			for(QueryClause clause:this.getTypeSpecificClause()) {
//				params.put(clause.getKey(), clause);
//			}
//			
//			
//			Map<Integer, Map<String,Integer>> bigMap = new LinkedHashMap<Integer, Map<String,Integer>>();
//			Map<String,Integer> littleMap = new LinkedHashMap<String,Integer>();
//			
//
//			AggregatedInformation info = this.getAggregatedRecords(params);
//			
////			Collection<T> recordsToReturn = new ArrayList<T>();
////			for (T t : records) {
////				if(SecurityManager.isAllowed(this.getUserId(), Action.READ_RECORD, t.getFullyQualifiedConsumerId())) {
////					recordsToReturn.add(t);
////				}
////				else {
////					logger.debug("filtering out record " + t.getId());
////				}
////			}
////			return recordsToReturn;
//			
//			return info;
//
//		}
//		catch(QueryFormatException e) {
//			throw new BadRequestException(e.getMessage());
//		}
//	}
	
	@GET
	@Path("/records_size")
	public String getNumberOfRecords(@QueryParam("query") String query) throws NotAuthorizedException, QueryFormatException, UnknownHostException {
		Map<String, QueryClause> params = QueryHelper.parseQuery(query);
		for(QueryClause clause:this.getTypeSpecificClause()) {
			params.put(clause.getKey(), clause);
		}
		UsageTrackerDB db = UsageTrackerDB.getStorage();
		return String.valueOf(db.size(params));
		
	}

	
	@POST
	@Consumes({MediaType.APPLICATION_XML})
	@Produces({MediaType.APPLICATION_XML})
	public Collection<T> getRecordsWithPost(String x, QueryClause q) throws NotAuthorizedException {
		// check if user can read
		if(!SecurityManager.isAllowed(this.getUserId(),Action.READ_RECORD)){
			logger.info("User "+this.getUserId()+" doesn't have permission to perform "+Action.READ_RECORD);
			throw new NotAuthorizedException();
		}
		//logger.debug("query inside getRecords = "+query+" query.size() = "+query.size());
		Map<String, QueryClause> params = null;
		params.put(x, q);
				// add type-specific clauses
		//			for(QueryClause clause:this.getTypeSpecificClause()) {
		//				params.put(clause.getKey(), clause);
		//			}
					Collection<T> records = this.getRecords(params);
					
					Collection<T> recordsToReturn = new ArrayList<T>();
					for (T t : records) {
						if(SecurityManager.isAllowed(this.getUserId(), Action.READ_RECORD, t.getFullyQualifiedConsumerId())) {
							recordsToReturn.add(t);
						}
						else {
							logger.debug("filtering out record " + t.getId());
						}
					}
					return recordsToReturn;
	}
	
	@GET
	@Produces({MediaType.APPLICATION_XML})
	public Collection<T> getRecords(@QueryParam("query") String query) throws NotAuthorizedException {
		// check if user can read
		if(!SecurityManager.isAllowed(this.getUserId(),Action.READ_RECORD)){
			logger.info("User "+this.getUserId()+" doesn't have permission to perform "+Action.READ_RECORD);
			throw new NotAuthorizedException();
		}
		try {
//logger.debug("query inside getRecords = "+query+" query.size() = "+query.size());
Map<String, QueryClause> params = QueryHelper.parseQuery(query);
			// add type-specific clauses
			for(QueryClause clause:this.getTypeSpecificClause()) {
				params.put(clause.getKey(), clause);
			}
			Collection<T> records = this.getRecords(params);
			
			Collection<T> recordsToReturn = new ArrayList<T>();
			for (T t : records) {
				if(SecurityManager.isAllowed(this.getUserId(), Action.READ_RECORD, t.getFullyQualifiedConsumerId())) {
					recordsToReturn.add(t);
				}
				else {
					logger.debug("filtering out record " + t.getId());
				}
			}
			return recordsToReturn;
		}
		catch(QueryFormatException e) {
			throw new BadRequestException(e.getMessage());
		}
	}

	private String getUserId() {
		return (String)request.getAttribute("userId");
	}

	private Collection<T> convert(Collection<RawUsageRecord> records) {
		Collection<T> out = new Vector<T>();
		for(RawUsageRecord rawur:records) {
			out.add(this.createRecord(rawur));
		}
		return out;
	}

	protected abstract T createRecord(RawUsageRecord record);

}