package org.gcube.accounting.couchdb.query;

import java.net.SocketTimeoutException;
import java.util.Calendar;
import java.util.Map;
import java.util.Set;

import org.ektorp.DbAccessException;
import org.gcube.accounting.analytics.TemporalConstraint;
import org.gcube.accounting.analytics.TemporalConstraint.AggregationMode;
import org.gcube.accounting.analytics.persistence.AccountingPersistenceBackendQuery;
import org.gcube.accounting.analytics.persistence.AccountingPersistenceBackendQueryFactory;
import org.gcube.documentstore.records.AggregatedRecord;
import org.gcube.documentstore.records.RecordUtility;
import org.gcube.vremanagement.executor.plugin.Plugin;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Luca Frosini (ISTI - CNR) http://www.lucafrosini.com/
 */
public class CouchDBQueryPlugin extends Plugin<CouchDBQueryPluginDeclaration> {

	/**
	 * Logger
	 */
	private static Logger logger = LoggerFactory.getLogger(CouchDBQueryPlugin.class);
	
	public static final String DELAY_MILLIS = "delay";
	public static final long RETRY_DELAY_MILLIS = 1000*60; // 1 min
	public static final long DEFAULT_DELAY_MILLIS = 1000*60*2; // 2 min
	
	public static final int MONTH_INTERVAL = 3;
	public static final int QUERY_MONTH_INTERVAL = MONTH_INTERVAL - 1;
	
	/**
	 * @param runningPluginEvolution
	 */
	public CouchDBQueryPlugin(CouchDBQueryPluginDeclaration pluginDeclaration) {
		super(pluginDeclaration);
	}
	
	/**{@inheritDoc}*/
	@Override
	public void launch(Map<String, Object> inputs) throws Exception {
		logger.debug("Launching {}", this.getClass().getSimpleName());
		
		long delay = DEFAULT_DELAY_MILLIS;
		
		if(inputs != null && inputs.containsKey(DELAY_MILLIS)){
			try {
				delay = new Long(inputs.get(DELAY_MILLIS).toString());
			} catch(Exception e){
				logger.warn("The provided value {} for {} is not a long. Default value {} will be used", 
						inputs.get(DELAY_MILLIS), DELAY_MILLIS, DEFAULT_DELAY_MILLIS, e);
			}
		} else {
			logger.debug("No provided value for {}. Default value {} will be used", 
					DELAY_MILLIS, DEFAULT_DELAY_MILLIS);
		}
		
		
		AccountingPersistenceBackendQuery apq = AccountingPersistenceBackendQueryFactory.getInstance();
		@SuppressWarnings("rawtypes")
		Map<String, Class<? extends AggregatedRecord>> map = RecordUtility.getAggregatedRecordClassesFound();
		
		for(@SuppressWarnings("rawtypes") Class<? extends AggregatedRecord> recordClass : map.values()){
			try {
				Set<String> queryKey = apq.getKeys(recordClass);
				Calendar startTime = Calendar.getInstance();
				int month = startTime.get(Calendar.MONTH);
				if(month >= QUERY_MONTH_INTERVAL){
					startTime.set(Calendar.MONTH, month - QUERY_MONTH_INTERVAL);
				}else{
					startTime.set(Calendar.YEAR, startTime.get(Calendar.YEAR) - 1);
					startTime.set(Calendar.MONTH, Calendar.DECEMBER - (QUERY_MONTH_INTERVAL - startTime.get(Calendar.MONTH)));
				}
				Calendar endTime = Calendar.getInstance();
				boolean iterate = true;
				int i = 4;
				while(iterate) {
					try {
						TemporalConstraint temporalConstraint = new TemporalConstraint(startTime.getTimeInMillis(), endTime.getTimeInMillis(), AggregationMode.MONTHLY);
						apq.query(recordClass, temporalConstraint, null);
						logger.debug("Going to query possible values for retrieved keys");
						for(String key : queryKey){
							logger.debug("Getting possible values for key {}", key);
							apq.getPossibleValuesForKey(recordClass, key);
						}
						iterate = false;
					}catch(DbAccessException ex){
						if(ex.getCause() instanceof SocketTimeoutException && i<6){
							long retryInterval = RETRY_DELAY_MILLIS*i;
							logger.error("{} retry in {} millis", ex.getCause().getClass().getSimpleName(), retryInterval);
							iterate = true;
							i++;
							Thread.sleep(retryInterval);
						}else{
							throw ex;
						}
					}catch (Exception e) {
						throw e;
					}
				}
				logger.debug("Waiting {} millis before quering the next UsageRecord", delay);
				Thread.sleep(delay);
			}catch(Exception e){
				logger.warn("", e);
			}
		}
		
		logger.debug("{} has finished", this.getClass().getSimpleName());
	}

	/**{@inheritDoc}*/
	@Override
	protected void onStop() throws Exception {
		logger.debug("{} onStop() function", this.getClass().getSimpleName());
	}
	
}
