package org.gcube.datatransfer.agent.impl.handlers;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.net.io.CopyStreamEvent;
import org.apache.commons.net.io.CopyStreamException;
import org.apache.commons.net.io.CopyStreamListener;
import org.apache.commons.net.io.Util;
import org.apache.commons.vfs2.FileObject;
import org.gcube.datatransfer.agent.impl.context.ServiceContext;
import org.gcube.datatransfer.agent.impl.jdo.TransferObject;
import org.gcube.datatransfer.common.outcome.TransferStatus;
import org.gcube.datatransfer.agent.impl.utils.TransferUtils;
import org.gcube.datatransfer.agent.stubs.datatransferagent.DestData;
import org.gcube.datatransfer.agent.stubs.datatransferagent.PostProcessType;
import org.gcube.datatransfer.agent.stubs.datatransferagent.TransferType;

/**
 * 
 * @author Andrea Manzi(CERN)
 * 
 */
public class LocalFileTransferAsyncHandler extends TransferHandler {

	public static int bufferSize = Util.DEFAULT_COPY_BUFFER_SIZE * 1000;

	private ExecutorService pool;

	public LocalFileTransferAsyncHandler(String[] inputFiles, 
			String outPath, String transferId, TransferType type,
			DestData data, int startIndex, int endIndex) {
		this.inputFiles = inputFiles;
		this.timeout = data.getOutUri().getOptions().getTransferTimeout();
		this.outPath = outPath;
		this.transferId = transferId;
		this.transferType = type;
		this.destData = data;
		this.startIndex = startIndex;
		this.endIndex = endIndex;
		pool = Executors.newFixedThreadPool(1);

	}

	@Override
	public void run() {
		for (int i = startIndex; i <= endIndex; i++) {
			long startTime = 0;
			TransferObject transferObj = null;

			try {

				transferObj = TransferUtils.createTransferObjectJDO(transferId,
						transferType);
				transferObj.setSourceURI(inputFiles[i]);

				FileObject inputFile = TransferUtils
						.prepareFileObject(inputFiles[i]);
				transferObj.setSize(inputFile.getContent().getSize());

				// getting outfile info
				String outputFile = inputFile.getName().getBaseName();
				String relativeOutputFile = outPath + File.separator
						+ outputFile;

				logger.debug("Relative Output file " + relativeOutputFile);

				FileObject absoluteOutputFile = ServiceContext.getContext()
						.getLocalFSManager().resolveFile(relativeOutputFile);

				FileObject absolutePath = ServiceContext.getContext()
						.getLocalFSManager().resolveFile(outPath);

				absolutePath.createFolder();

				if (absoluteOutputFile.exists()
						&& !destData.getOutUri().getOptions().isOverwrite()) {
					String msg= "the file represented by the URL "
							+ inputFile.getURL().toString()
							+ " cannot be copied cause a file with the same name already exists";
					logger.error(msg);
					throw new Exception(msg);
				}

				transferObj.setDestURI(absoluteOutputFile.getName().getPath());

				logger.debug("Copying file from URL " + inputFile.getURL()
						+ " to : " + absoluteOutputFile.getName().getPath());

				startTime = System.currentTimeMillis();
				// parameters

				boolean terminate  = false;


				InputStream sourceFileIn = inputFile.getContent()
						.getInputStream();
				OutputStream destinationFileOut = absoluteOutputFile
						.getContent().getOutputStream();

				Handler handler = new Handler(sourceFileIn, destinationFileOut,
						inputFile.getContent().getSize(), listener);

				try {
					pool.execute(handler);
					pool.shutdown();
				} catch (Exception e) {
					pool.shutdownNow();
					absoluteOutputFile.delete();
					throw new Exception("Error while executing the transfer");
				}

				// waiting for transfer to complete
				terminate = pool.awaitTermination(this.timeout,	TimeUnit.MILLISECONDS);

				if (terminate) {
					// check the postprocess Options
					PostProcessType[] postProcesses = destData.getOutUri()
							.getOptions().getPostProcess();

					if (postProcesses != null)
						for (PostProcessType process : postProcesses)
							TransferUtils.applyPostProcess(process,
									absoluteOutputFile, absolutePath, destData
									.getOutUri().getOptions()
									.getConversionType());

					transferObj.setStatus(TransferStatus.DONE.name());
					transferObj.setOutcome("File succesfully copied to "
							+ absoluteOutputFile);
					logger.debug("File succesfully copied to "
							+ absoluteOutputFile);
					ServiceContext.getContext().getDbManager()
					.addTransferObjectCompleted(transferId);
				} else {
					String msg = "Transfer aborted because timeout has elapsed";
					logger.error(msg);
					absoluteOutputFile.delete();
					throw new Exception(msg);
				}

			}
			catch (Exception e) {
				e.printStackTrace();
				transferObj.setStatus(TransferStatus.FAILED.name());
				transferObj.setOutcome(e.toString());
				errorHappened = true;
			} finally {
				long endTime = System.currentTimeMillis();
				transferObj.setTransferTime(endTime - startTime);
				transferObjs.add(transferObj);

			}
		}
	}



	CopyStreamListener listener = new CopyStreamListener() {

		@Override
		public void bytesTransferred(long arg0, int arg1, long arg2) {
			try {
				ServiceContext.getContext().getDbManager()
				.updateTransferObjectInfo(transferId, arg1);
			} catch (Exception e) {
				logger.error("Error updating DB");
			}

		}

		@Override
		public void bytesTransferred(CopyStreamEvent arg0) {

		}
	};


	public static long copyLarge(InputStream input, OutputStream output)
			throws IOException {
		byte[] buffer = new byte[bufferSize];
		long count = 0;
		int n = 0;
		while (-1 != (n = input.read(buffer))) {
			output.write(buffer, 0, n);
			count += n;
		}
		return count;
	}

	public static void copyWithHttpClient(String inputURI, String outPutUri) throws Exception{
		HttpClient client = new HttpClient();
		GetMethod get = new GetMethod(inputURI);

		int status = client.executeMethod(get);

		InputStream is = get.getResponseBodyAsStream();

		FileOutputStream fos = new FileOutputStream(outPutUri);

		copyLarge(is, fos);
		fos.close();

	}
}

class Handler implements Runnable {
	private InputStream in;
	private OutputStream out;
	private long streamSize;
	CopyStreamListener listener;

	Handler(InputStream in, OutputStream out, long streamSize,
			CopyStreamListener listener) {
		this.in = in;
		this.out = out;
		this.streamSize = streamSize;
		this.listener = listener;

	}

	public void run() {
		try {
			Util.copyStream(in, out, LocalFileTransferAsyncHandler.bufferSize,
					streamSize, listener);
		} catch (CopyStreamException e) {
			e.printStackTrace();
		} finally {
			Util.closeQuietly(in);
			Util.closeQuietly(out);

		}

	}


}
