package com.moseph.gis.raster;


import java.io.*;
import java.text.NumberFormat;
import java.util.regex.*;

/**
 * Writes rasters out to files.
 * @author dmrust
 *
 */
public class RasterWriter
{
	NumberFormat cellFormat = null;
	String nodataString = Raster.DEFAULT_NODATA;
	
	/**
	 * Writes out the given Raster object to the given filename.
	 * 
	 * Throws the exceptions associated with filehandling
	 * @param filename
	 * @param r
	 * @throws IOException
	 */
	public void writeRaster( String filename, Raster r ) throws IOException
	{
		File f = new File( filename );
		if( f.exists() ) f.delete();
		if( ! f.createNewFile() ) throw new RuntimeException( "Could not create file for some reason!");
		PrintStream o = new PrintStream( f );
		o.println( "ncols " + r.getCols() );
		o.println( "nrows " + r.getRows() );
		o.println( "xllcorner " + r.getXll() );
		o.println( "yllcorner " + r.getYll());
		o.println( "cellsize " + r.getCellsize() );
		o.println( "NODATA_value " + r.getNDATA()  );
		
		for( double[] row : r.getData() )
		{
			StringBuffer b = new StringBuffer();
			for( int i = 0; i < row.length; i++ )
			{
				if( Double.isNaN( row[i] )  ) b.append( r.getNDATA() );
				else if( cellFormat != null ) b.append( cellFormat.format( row[i] ));
				else b.append( row[i] );
				if( i < row.length-1 ) b.append(  " "  );
			}
			o.println( b );
		}
		o.close();
	}
	
	/**
	 * Shortcut method, if you just have some data and want to write it out as a Raster.
	 * 
	 * There is no error checking at the moment (e.g. about equal size rows)
	 * @param filename
	 * @param data
	 * @param xll
	 * @param yll
	 * @param size
	 * @param ndata
	 * @throws IOException
	 */
	public void writeRaster( String filename, double[][] data, double xll, double yll, double size, String ndata ) throws IOException
	{
		writeRaster( filename, Raster.getTempRaster( data, xll, yll, size, ndata ) );
	}
	
	/**
	 * Can be used to set a number format for the cells. For example, if they are all integer
	 * values, you can set an integer format. This should help with roundtrippability for
	 * existing Raster files
	 * @param format
	 */
	public void setCellFormat( NumberFormat format )
	{
		cellFormat = format;
	}

}
