/*
 * Decompiled with CFR 0.152.
 */
package org.gcube.dataanalysis.oscar;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintWriter;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.Calendar;
import org.apache.commons.io.FileSystemUtils;
import org.apache.commons.io.FileUtils;
import org.gcube.dataanalysis.oscar.IncrementalOscarMerger;
import org.gcube.dataanalysis.oscar.util.FTPDownloader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ucar.nc2.Attribute;
import ucar.nc2.FileWriter2;
import ucar.nc2.NetcdfFile;
import ucar.nc2.NetcdfFileWriter;
import ucar.nc2.dataset.NetcdfDataset;
import ucar.nc2.util.CancelTask;
import ucar.nc2.util.CancelTaskImpl;

public class OscarMerger {
    private String yearlyOscarRemoteFilePattern = "ftp://podaac-ftp.jpl.nasa.gov/allData/oscar/preview/L4/resource/LAS/oscar_third_deg_180/oscar_vel${YYYY}_180.nc";
    private String yearlyOscarLocalFilePattern = "oscar_vel${YYYY}_180.nc";
    private static String yearlyOscarHTMLPagePattern = "https://thredds.jpl.nasa.gov/thredds/dodsC/OSCAR_L4_OC_third-deg/oscar_vel${YYYY}.nc.gz.html";
    private static String workdir = "/tmp/oscar-merger";
    private Integer startYear;
    private Integer endYear;
    private Boolean test = true;
    private static final Logger logger = LoggerFactory.getLogger(IncrementalOscarMerger.class);

    private Integer getStartYear() {
        if (this.startYear == null) {
            for (int year = 1990; year < Calendar.getInstance().get(1); ++year) {
                if (!this.checkRemoteOscarFile(year)) continue;
                this.startYear = year;
                break;
            }
        }
        return this.startYear;
    }

    private Integer getEndYear() {
        if (this.test.booleanValue()) {
            this.startYear = 1992;
            this.endYear = 1993;
        }
        if (this.endYear == null) {
            for (int year = Calendar.getInstance().get(1); year > 1990; --year) {
                if (!this.checkRemoteOscarFile(year)) continue;
                this.endYear = year;
                break;
            }
        }
        return this.endYear;
    }

    public void generateDescriptorFile() {
        this.generateDescriptorFile(new File(workdir), this.getDescriptorFileName());
    }

    private void generateDescriptorFile(File destinationDir, String fileName) {
        String out = "";
        out = out + "<netcdf xmlns=\"http://www.unidata.ucar.edu/namespaces/netcdf/ncml-2.2\">\n";
        out = out + "  <attribute name=\"title\" value=\"OSCAR Velocity Dataset\"/>\n";
        out = out + "  <aggregation type=\"joinExisting\" dimName=\"time\">\n";
        int startYear = this.getStartYear();
        int endYear = this.getEndYear();
        for (int year = startYear; year <= endYear; ++year) {
            boolean localFileExists = true;
            String url = this.yearlyOscarLocalFilePattern.replaceAll("\\$\\{YYYY\\}", year + "");
            File f = new File(destinationDir, url);
            if (localFileExists) {
                out = out + "    <netcdf location=\"" + f.getAbsolutePath() + "\"/>\n";
                continue;
            }
            logger.info("can't find remote file " + url);
        }
        out = out + "  </aggregation>\n";
        out = out + "</netcdf>";
        try (PrintWriter pw = new PrintWriter(this.getDescriptorFile());){
            pw.println(out);
        }
        catch (FileNotFoundException e) {
            e.printStackTrace();
        }
    }

    private File getDescriptorFile() {
        return new File(this.getWorkDir(), this.getDescriptorFileName());
    }

    private String getDescriptorFileName() {
        if (this.startYear == this.endYear) {
            return "oscar-vel-" + this.getStartYear() + ".xml";
        }
        return "oscar-vel-" + this.getStartYear() + "-" + this.getEndYear() + ".xml";
    }

    private boolean checkRemoteOscarFile(int year) {
        String url = yearlyOscarHTMLPagePattern.replaceAll("\\$\\{YYYY\\}", year + "");
        try {
            HttpURLConnection.setFollowRedirects(false);
            HttpURLConnection con = (HttpURLConnection)new URL(url).openConnection();
            con.setRequestMethod("HEAD");
            return con.getResponseCode() == 200;
        }
        catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

    private File getWorkDir() {
        File out = new File(workdir);
        out.mkdir();
        return out;
    }

    private String getMergedFileName() {
        if (this.startYear == this.endYear) {
            return "oscar-vel-" + this.getStartYear() + ".nc";
        }
        return "oscar-vel-" + this.getStartYear() + "-" + this.getEndYear() + ".nc";
    }

    private File getMergedFile() {
        return new File(this.getWorkDir(), this.getMergedFileName());
    }

    public void merge() throws Exception {
        this.checkReady();
        this.downloadRemoteFiles();
        this.generateDescriptorFile();
        this.createOscarMergedFile();
    }

    private void downloadRemoteFiles() {
        this.downloadRemoteFiles(this.getWorkDir());
    }

    private void downloadRemoteFiles(File destinationDir) {
        System.out.println("downloading remote files");
        int startYear = this.getStartYear();
        int endYear = this.getEndYear();
        for (int year = startYear; year <= endYear; ++year) {
            try {
                File localFile = new File(this.getWorkDir(), this.yearlyOscarLocalFilePattern.replaceAll("\\$\\{YYYY\\}", year + ""));
                System.out.println("writing to " + localFile.toString());
                this.downloadFile(this.yearlyOscarRemoteFilePattern.replaceAll("\\$\\{YYYY\\}", year + ""), localFile);
                System.out.println("File downloaded");
                continue;
            }
            catch (Exception ex) {
                ex.printStackTrace();
            }
        }
    }

    private void downloadFile(String source, File destination) throws IOException, Exception {
        if (!source.startsWith("ftp:")) {
            throw new Exception("only ftp downloads currently supported");
        }
        String tmp = source.substring(6);
        String host = tmp.split("/")[0];
        String file = tmp.split("/", 2)[1];
        FTPDownloader ftpDownloader = new FTPDownloader(host, "anonymous", null);
        ftpDownloader.downloadFile(file, destination.getAbsolutePath());
        System.out.println("FTP File downloaded successfully");
        ftpDownloader.disconnect();
    }

    private void createOscarMergedFile() throws Exception {
        String datasetIn = this.getDescriptorFile().getAbsolutePath();
        String datasetOut = this.getMergedFile().getAbsolutePath();
        CancelTaskImpl cancel = new CancelTaskImpl();
        NetcdfFile ncfileIn = NetcdfDataset.openFile((String)datasetIn, (CancelTask)cancel);
        logger.info(String.format("NetcdfDatataset read from %s write to %s ", datasetIn, datasetOut));
        NetcdfFileWriter.Version version = NetcdfFileWriter.Version.netcdf4;
        FileWriter2 writer = new FileWriter2(ncfileIn, datasetOut, version, null);
        writer.getNetcdfFileWriter().setLargeFile(true);
        NetcdfFile ncfileOut = writer.write((CancelTask)cancel);
        for (Attribute a : ncfileOut.getGlobalAttributes()) {
            System.out.println(a);
        }
        if (ncfileOut != null) {
            ncfileOut.close();
        }
        ncfileIn.close();
        cancel.setDone(true);
        logger.info(String.format("%s%n", cancel));
    }

    public void cleanup() {
        if (!this.getWorkDir().getAbsolutePath().startsWith("/tmp")) {
            logger.info("NOT REMOVING WORKING DIR AS IT'S NOT IN /TMP");
        } else {
            logger.info("Removing working directory " + this.getWorkDir().getAbsolutePath() + "...");
            try {
                FileUtils.deleteDirectory((File)this.getWorkDir());
                logger.info("Removed.");
            }
            catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    public void checkReady() throws Exception {
        this.checkEnoughDiskSpace();
    }

    private void checkEnoughDiskSpace() throws Exception {
        Long neededSpace = 387973120L * (long)(this.getEndYear() - this.getStartYear() + 1);
        neededSpace = neededSpace + (long)(1198006272 * (this.getEndYear() - this.getStartYear() + 1));
        logger.info("Needed disk space: " + neededSpace + " bytes");
        Long availableSpace = FileSystemUtils.freeSpaceKb((String)this.getWorkDir().getAbsolutePath()) * 1024L;
        logger.info("Available disk space: " + availableSpace + " bytes");
        if (neededSpace >= availableSpace) {
            Double gb = (double)neededSpace.longValue() / 1024.0 / 1024.0 / 1024.0;
            String message = String.format("Not enough disk space. At least %2.2fGB of available disk needed.", gb);
            throw new Exception(message);
        }
        logger.info("There's enough disk space to proceed");
        Double percent = (double)neededSpace.longValue() * 1.0 / (double)availableSpace.longValue() * 1.0 * 100.0;
        String message = String.format("I'm about to use %.0f%% of the available disk space (but I'll release it afterwards, I promise)", percent);
        logger.info(message);
    }
}

