package org.gcube.couchbase.helpers;

import gr.uoa.di.madgik.grs.buffer.IBuffer.Status;
import gr.uoa.di.madgik.grs.events.KeyValueEvent;
import gr.uoa.di.madgik.grs.proxy.tcp.TCPWriterProxy;
import gr.uoa.di.madgik.grs.reader.ForwardReader;
import gr.uoa.di.madgik.grs.reader.GRS2ReaderException;
import gr.uoa.di.madgik.grs.record.GenericRecord;
import gr.uoa.di.madgik.grs.record.GenericRecordDefinition;
import gr.uoa.di.madgik.grs.record.Record;
import gr.uoa.di.madgik.grs.record.RecordDefinition;
import gr.uoa.di.madgik.grs.record.field.FieldDefinition;
import gr.uoa.di.madgik.grs.record.field.StringField;
import gr.uoa.di.madgik.grs.record.field.StringFieldDefinition;
import gr.uoa.di.madgik.grs.writer.GRS2WriterException;
import gr.uoa.di.madgik.grs.writer.RecordWriter;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import org.gcube.couchbase.entities.ForwardIndexDocument;
import org.gcube.couchbase.entities.MetaIndex;
import org.gcube.couchbase.entities.Operator;
import org.gcube.couchbase.helpers.QueryHelper.OrderBy;
import org.gcube.indexmanagement.bdbwrapper.BDBGcqlProcessor;
import org.gcube.indexmanagement.bdbwrapper.BDBGcqlQueryContainer;
import org.gcube.indexmanagement.bdbwrapper.BDBGcqlQueryContainer.SingleTerm;
import org.gcube.indexmanagement.common.IndexException;
import org.gcube.indexmanagement.common.IndexType;
import org.gcube.indexmanagement.common.XMLProfileParser;
import org.gcube.indexmanagement.resourceregistry.RRadaptor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.couchbase.client.CouchbaseClient;

/**
 * 
 * @author Alex Antoniadis
 * 
 */
public class CouchBaseHelper {

	private static final long RSTIMEOUT = 5;
	private static final Logger logger = LoggerFactory.getLogger(CouchBaseHelper.class);

	/**
	 * Creates one view-index for each key. The view-index will have the name
	 * bucketName_fieldName_datatype
	 * 
	 * @param client
	 * @param bucketName
	 * @param designDocumentName
	 * @param keys
	 */
	public static void createIndexes(CouchbaseClient client, String bucketName, String designDocumentName,
			Map<String, CouchBaseDataTypesHelper.DataType> keys) {
		logger.info("Creating indexes...");
		ViewHelper.createAllIndexes(client, bucketName, designDocumentName, keys);
		logger.info("Creating indexes...OK");
	}

	/**
	 * Insert a key-value pair
	 * 
	 * @param client
	 * @param key
	 * @param value
	 * @param dataType
	 */
	public static void insertSimple(CouchbaseClient client, String key, String value,
			CouchBaseDataTypesHelper.DataType dataType) {
		// 0 means that it will last forever
		String val = CouchBaseDataTypesHelper.getDoc(value, dataType);

		logger.info("Adding doc " + key + " : " + value);

		client.set(key, 0, val);
	}

	/**
	 * Inserts a ForwardIndexDocument document
	 * 
	 * @param client
	 * @param doc
	 */
	public static void insertSimple(CouchbaseClient client, ForwardIndexDocument doc) {
		String id = doc.getID();
		String value = doc.toJSON();

		//logger.info("Adding " + id + " : " + value);
		logger.info("Adding record with " + id);
		
		client.set(id, 0, value);
	}

	/**
	 * Deletes the record with the given key
	 * 
	 * @param key
	 * @param dataType
	 * @param client
	 */
	public static void delete(String key, CouchBaseDataTypesHelper.DataType dataType, CouchbaseClient client) {
		// 0 means that it will last forever
		// String val = CouchBaseDataTypesHelper.getDoc(value, dataType);

		client.delete(key);
	}

	/**
	 * Deletes all the documents with key in the given keys list
	 * 
	 * @param client
	 * @param keys
	 */
	public static void deleteDocsCouchBase(CouchbaseClient client, List<String> keys) {
		for (String key : keys) {
			logger.info("Deleting document with ID : " + key);
			client.delete(key);
		}
	}
	
	/**
	 * Deletes all the documents of the given collection
	 * 
	 * @param client
	 * @param keys
	 */
	public static Boolean deleteCollectionCouchBase(CouchbaseClient client, final String bucketName,
			final String designDocName, final Map<String, CouchBaseDataTypesHelper.DataType> keys, MetaIndex meta, String collID) {
		logger.info("Deleting documents of collection with ID : " + collID);
		List<String> docIDs = collectionDocIDs(client, bucketName, designDocName, keys, collID);
		if (docIDs == null || docIDs.size() == 0) {
			logger.warn("No documents found for collection with ID : " + collID);
			//return false;
		} else {
			deleteDocsCouchBase(client, docIDs);
		}
		
		logger.info("Deleting collection from meta index with id : " + collID);
		meta.removeCollection(collID);
		
		logger.info("Saving metaindex");
		meta.flushToDatabase(client);
		
		return true;
	}
	
	
	/**
	 * Returns the document IDs of the collection with the given ID
	 * 
	 * @param client
	 * @param bucketName
	 * @param designDocumentName
	 * @param keys
	 * @param collID
	 * @return list of the doc IDs of the collection
	 */
	public static List<String> collectionDocIDs(CouchbaseClient client, String bucketName, final String designDocumentName, final Map<String, CouchBaseDataTypesHelper.DataType> keys, String collID){
		
		logger.info("Searching for documents of collection : " + collID);
		String viewName = ViewHelper.constructViewName(bucketName, IndexType.COLLECTION_FIELD, keys);
		logger.info("Searching for collections in view : " + viewName);
		
		List<String> docIDs = QueryHelper.queryCouchBase(client, designDocumentName, viewName, Operator.EQUAL, collID);
		logger.info("Found " + docIDs.size() + " docs.");
		logger.info("Found docs are : " + docIDs);
		
		return docIDs;
	}

	/**
	 * Executes a dummyQuery on each index to "warm" it. This warming will
	 * update the indexes so any update in the database will be reflected to the
	 * indexes and the queries will be consistent.
	 * 
	 * @param client
	 * @param bucketName
	 * @param designDocumentName
	 * @param keys
	 */
	public static void commit(CouchbaseClient client, String bucketName, String designDocumentName,
			Map<String, CouchBaseDataTypesHelper.DataType> keys) {
		if (keys == null || keys.size() == 0) {
			logger.warn("Keys not set. Commit will be skipped");
			return;
		}
		
		for (String field : keys.keySet()) {
			logger.info("Performing a dummy query on field : " + field);

			String viewName = ViewHelper.constructViewName(bucketName, field, keys);
			
			int tries = 5;
			boolean success = false;
			Exception ex = null;
			while (tries > 0) {
				try {
					QueryHelper.dummyQuery(client, designDocumentName, viewName);
					logger.warn("Dummy query perform succeded.");
					success = true;
					break;
				} catch (Exception e) {
					ex = e;
					tries--;
					logger.warn("Dummy query perform failed. Probably timeout");
					logger.warn("Retrying in 2 secs.. Tries left : " + tries);
					try {
						Thread.sleep(2000);
					} catch (Exception e2) {}
				}
			}
			if (!success) {
				logger.warn("Commit failed. Index for field " + field +  "  may not be ready. Although this is not a problem. Latest exception : ", ex);
			}
		}
	}

	/**
	 * Executes the queryString and returns a gRS2 endpoint locator were the
	 * results will written. The meta index is needed in order to get the
	 * presentables and the searchables fields that are known The rradaptor is
	 * used for the query parsing from BDBGcqlProcessor
	 * 
	 * @param client
	 * @param meta
	 * @param bucketName
	 * @param designDocName
	 * @param keys
	 * @param rradaptor
	 * @param queryString
	 * @return gRS2 locator of the response
	 */
	public static String query(final CouchbaseClient client, MetaIndex meta, final String bucketName,
			final String designDocName, final Map<String, CouchBaseDataTypesHelper.DataType> keys, final RRadaptor rradaptor,
			String queryString) {
		try {

			final ArrayList<String> presentables = new ArrayList<String>(meta.getPresentables());
			final ArrayList<String> searchables = new ArrayList<String>(meta.getSearchables());

			logger.info("Executing query : " + queryString);
			logger.info("presentables : " + presentables);
			logger.info("searchables  : " + searchables);
			logger.info("keys : " + keys);

			final String orderByField = QueryHelper.getOrderByField(queryString);
			logger.info("orderByField : " + orderByField);
			
			
			
			final OrderBy orderby = QueryHelper.getOrderBy(queryString);
			logger.info("orderby : " + orderby);
			
			if (orderByField != null)
				queryString = QueryHelper.deleteOrderBy(queryString, orderByField, orderby);
			logger.info("query string after orderby removal : " + queryString);
			
			
			BDBGcqlProcessor processor = new BDBGcqlProcessor();

			BDBGcqlQueryContainer queryCont = (BDBGcqlQueryContainer) processor.processQuery(presentables, searchables,
					queryString, rradaptor);
			final List<ArrayList<SingleTerm>> queries = queryCont.getBdbQueries();
			final List<String> projections = new ArrayList<String>(queryCont.getProjectedFields().values());

			if (projections.contains(IndexType.WILDCARD)){
				logger.info("projections contain wildcard");
				projections.clear();
				projections.addAll(presentables);
			}
			logger.info("projections 1/2  : " + projections);
			if (orderByField != null && !projections.contains(orderByField)) {
				//add orderby field in projections
				projections.add(rradaptor.getFieldNameById(orderByField));
			}
			logger.info("projections 2/2 : " + projections);
			
			QueryHelper.addExcludedFields(queries, projections);

			final boolean distinct = queryCont.getDistinct();
			logger.info("distinct : " + distinct);
			// give the definition for the record
			final RecordWriter<GenericRecord> rsWriter = initRSWriterForSearchHits(presentables, projections, rradaptor);// createFieldDefinition(projections, distinct);

			if (!projections.contains(IndexType.DOCID_FIELD))
				projections.add(IndexType.DOCID_FIELD);

			Runnable writerRun = new Runnable() {

				public void run() {
					try {
						List<Map<String, String>> resultDocs = new ArrayList<Map<String, String>>();

						for (ArrayList<SingleTerm> query : queries) {
							logger.debug(" >>> process Query");
							logger.debug("Query received: " + query);
							// prune query to throw field == *

							String subQueryString = QueryHelper.pruneQuery(query);

							List<String> docIDs = null;
							try {
								logger.info("executing query : " + subQueryString);
								docIDs = QueryHelper.queryString(client, bucketName, designDocName, keys,
										subQueryString);
								logger.info("query returned : " + docIDs.size());
								logger.trace("query returned : " + docIDs);
								Map<String, Object> docs = QueryHelper.multiGetCouchBase(client, docIDs, distinct);

								// logger.info("multiget returned : " + docs);

								// extract presentables from docs
								Collection<Map<String, String>> projectedDocuments = QueryHelper.applyProjection(docs,
										projections, distinct);
								resultDocs.addAll(projectedDocuments);

							} catch (Exception e) {
								logger.error("Error while searching the index", e);
							}
						}
						
						//sorting
						if (orderByField != null){
							logger.info("sorting on field : " + orderByField + " " + orderby );

							String sortField = rradaptor.getFieldNameById(orderByField);
							logger.info("sorting on field : " + sortField + " " + orderby + " datatype : " + keys.get(sortField));
							
							Comparator<Map<String, String>> comp = new RecordComparator(sortField, keys.get(sortField));
							if (orderby.equals(OrderBy.DESC))
								Collections.sort(resultDocs, Collections.reverseOrder(comp));
							else
								Collections.sort(resultDocs, comp);
							
							logger.info("sorting completed");
						}

						rsWriter.emit(new KeyValueEvent(IndexType.RESULTSNOFINAL_EVENT, "" + resultDocs.size()));
						for (Map<String, String> resultDoc : resultDocs) {
							if (!writeSearchHitInResultSet(resultDoc, rsWriter, projections, presentables, RSTIMEOUT)) {
								logger.error("ERROR writing doc " + resultDoc + " in RS");
								break;
							}
						}
						logger.info("writing results completed");

						if (rsWriter.getStatus() != Status.Dispose)
							rsWriter.close();
					} catch (Exception e) {
						logger.error("Error during search.", e);
						try {
							if (rsWriter.getStatus() != Status.Dispose)
								rsWriter.close();
						} catch (Exception ex) {
							logger.error("Error while closing RS writer.", ex);
						}
					}
				}
			};

			new Thread(writerRun).start();

			logger.info("results locator : " + rsWriter.getLocator());

			return rsWriter.getLocator().toString();
		} catch (Exception e) {
			logger.error("Error in query : ", e);
		}

		return null;
	}

	/**
	 * Inserts one rowset in the index and updates the meta index. This method
	 * is part of the feedLocator-bulk insert. Note that meta-index is not
	 * saved, the updates are done in memory. feedLocator is doing the saving.
	 * 
	 * @param client
	 * @param rowset
	 * @param meta
	 * @throws Exception
	 */
	private static void feedRowset(CouchbaseClient client, String rowset, MetaIndex meta) throws Exception {
		ForwardIndexDocument doc = new ForwardIndexDocument(rowset);
		CouchBaseHelper.insertSimple(client, doc);
		meta.updateFromDoc(doc);
	}

	/**
	 * Inserts all the records that are read from the resultSetLocation gRS2
	 * endpoint in the record, updates the meta-index and saves it
	 * 
	 * @param client
	 * @param resultSetLocation
	 * @param meta
	 * @return true on success, false otherwise
	 * @throws GRS2ReaderException
	 * @throws URISyntaxException
	 */
	public static boolean feedLocator(CouchbaseClient client, String resultSetLocation, MetaIndex meta)
			throws GRS2ReaderException, URISyntaxException {
		long beforeFeed, afterFeed;

		logger.info("Initializing reader at resultset : " + resultSetLocation);
		ForwardReader<Record> reader = new ForwardReader<Record>(new URI(resultSetLocation));
		reader.setIteratorTimeout(RSTIMEOUT);
		reader.setIteratorTimeUnit(TimeUnit.MINUTES);

		int rowSetCount = 1;
		boolean success = true;

		beforeFeed = System.currentTimeMillis();

		try {
			logger.info("Initializing resultset reader iterator");
			Iterator<Record> it = reader.iterator();
			long before, after;

			while (it.hasNext()) {
				logger.info("Getting result : " + rowSetCount);

				before = System.currentTimeMillis();
				Record result = it.next();

				after = System.currentTimeMillis();
				logger.info("Time for getting record from Result Set : " + (after - before) / 1000.0 + " secs");

				before = System.currentTimeMillis();

				String rowset = RowsetParser.getRowsetFromResult(result);
				feedRowset(client, rowset, meta);

				after = System.currentTimeMillis();
				logger.info("Time for getting rowset from record : " + (after - before) / 1000.0 + " secs");
				// logger.info("Result rowset : " + rowset);

				if (success == false) {
					logger.info("feed rowset failed");
					break;
				}

				logger.info("Result " + rowSetCount + " inserted");
				rowSetCount++;
			}
		} catch (Exception e) {
			logger.info("Exception while feeding", e);
		}

		reader.close();

		afterFeed = System.currentTimeMillis();

		logger.info("Total feed time : " + (afterFeed - beforeFeed) / 1000.0 + " secs");
		//System.out.println("Total feed time : " + (afterFeed - beforeFeed) / 1000.0 + " secs");

		if (success) {
			// add collection fields to resource
			logger.info("Feeding succeded!!!");
			meta.saveToDatabase(client);
		} else {
			logger.error("Feeding failed!!!");
		}
		return success;
	}

	/**
	 * Writes a search hit at the gRS2 writer 
	 * 
	 * @param docMap
	 * @param rsWriter
	 * @param projections
	 * @param presentables
	 * @param rsTimeout
	 * @return false if writer is not open, true if everything is ok
	 * @throws GRS2WriterException
	 */
	private static boolean writeSearchHitInResultSet(Map<String, String> docMap, RecordWriter<GenericRecord> rsWriter,
			List<String> projections, List<String> presentables, long rsTimeout) throws GRS2WriterException {

		if (rsWriter.getStatus() != Status.Open)
			return false;
		// the current RS record
		GenericRecord rec = new GenericRecord();
		// the fields for this record
		ArrayList<gr.uoa.di.madgik.grs.record.field.Field> fields = new ArrayList<gr.uoa.di.madgik.grs.record.field.Field>();

		String fieldContentDocID = docMap.containsKey(IndexType.DOCID_FIELD) ? docMap.get(IndexType.DOCID_FIELD)
				: "NoMetaId";// TODO:

		logger.info("Adding " + IndexType.DOCID_FIELD.toLowerCase() + " field with value : " + fieldContentDocID);
		fields.add(new StringField(fieldContentDocID));

		if (projections != null && projections.size() > 0) {

			List<String> returnFields = null;
			// in case there is the wildcard in projections
			if (projections.contains(IndexType.WILDCARD))
				returnFields = presentables;
			else
				returnFields = projections;

			logger.trace("returnFields : " + returnFields);

			for (String fieldName : returnFields) {
				if (fieldName.equalsIgnoreCase(IndexType.DOCID_FIELD)
						|| fieldName.equalsIgnoreCase(IndexType.PAYLOAD_FIELD))
					continue;

				String fieldContent = docMap.containsKey(fieldName) ? docMap.get(fieldName).toString() : "";

				fieldContent = XMLProfileParser.escapeForXML(fieldContent);

				logger.trace("fieldContent : " + fieldContent);

				fields.add(new StringField(fieldContent));
			}
		}

		// while the reader hasn't stopped reading
		if (rsWriter.getStatus() != Status.Open)
			return false;

		// set the fields in the record
		rec.setFields(fields.toArray(new gr.uoa.di.madgik.grs.record.field.Field[fields.size()]));

		while (!rsWriter.put(rec, rsTimeout, TimeUnit.SECONDS)) {
			// while the reader hasn't stopped reading
			if (rsWriter.getStatus() != Status.Open)
				break;
		}

		return true;
	}

	/**
	 * Creates the gRS2 writer for the given projections and presentables. It
	 * will create the definitions that are needed to write the records.
	 * 
	 * @param presentables
	 * @param projections
	 * @param adaptor
	 * @throws IndexException
	 * @throws GRS2WriterException
	 * @return an initialized RecordWriter
	 */
	public static RecordWriter<GenericRecord> initRSWriterForSearchHits(List<String> presentables,
			List<String> projections, RRadaptor adaptor) throws IndexException, GRS2WriterException {
		logger.info("Initializing gRS2 writer");
		logger.info("(1/3) getting field definitions");
		FieldDefinition[] fieldDef = null;
		try {
			fieldDef = createFieldDefinition(presentables, projections, adaptor);
		} catch (Exception e) {
			logger.error("Could not create field definition: ", e);
			throw new IndexException(e);
		}

		logger.info("(2/3) creating record definitions");
		RecordDefinition[] definition = new RecordDefinition[] { new GenericRecordDefinition(fieldDef) };

		logger.info("(3/3) creating rsWriter");
		return new RecordWriter<GenericRecord>(new TCPWriterProxy(), definition);

	}

	/**
	 * Creates the actual field definitions that will be used to initialize the
	 * gRS2 writer
	 * 
	 * @param presentable
	 * @param projections
	 * @param adaptor
	 * @return array of FieldDefinitions
	 * @throws Exception
	 */
	private static FieldDefinition[] createFieldDefinition(List<String> presentable, List<String> projections, RRadaptor adaptor) throws Exception {
		ArrayList<FieldDefinition> fieldDef = new ArrayList<FieldDefinition>();
		FieldDefinition[] fd = null;

		// if it is not the distinct case
		// if(!distinct) {
		// //add a field for the docID
		// fieldDef.add(new
		// StringFieldDefinition(ForwardIndexDocument.DOCID_FIELD));
		// System.out.println("field : " + ForwardIndexDocument.DOCID_FIELD);
		// }

		fieldDef.add(new StringFieldDefinition(ForwardIndexDocument.DOCID_FIELD));

		if (projections == null || projections.size() == 0) {
			fd = fieldDef.toArray(new FieldDefinition[fieldDef.size()]);
			logger.info("No projections found");
		} else {
			// TODO: fix code. 2 branches in one
			if (projections.contains(IndexType.WILDCARD)) {
				// return all the presentable fields (we assume that its the
				// updater's responsibility
				// to check for the fields to be returnable, stored) except for
				// the
				// full payload
				for (String fieldName : presentable) {

					String fieldID = adaptor.getFieldIDFromName(fieldName);
					// if a field is not the ObjectID or full payload field

					if (!fieldName.equalsIgnoreCase(IndexType.DOCID_FIELD)
							&& !fieldName.equalsIgnoreCase(IndexType.PAYLOAD_FIELD)) {
						fieldDef.add(new StringFieldDefinition(fieldID));
					}
				}
			} else {
				for (String fieldName : projections) {
					String fieldID = adaptor.getFieldIDFromName(fieldName);
					// if a field is not the ObjectID or full payload field

					if (!fieldName.equalsIgnoreCase(IndexType.DOCID_FIELD)
							&& !fieldName.equalsIgnoreCase(IndexType.PAYLOAD_FIELD)) {
						fieldDef.add(new StringFieldDefinition(fieldID));
					}

					// String fieldID = adaptor.getFieldIDFromName(current);
					// fieldDef.add(new StringFieldDefinition(fieldID));
				}
			}
		}

		fd = fieldDef.toArray(new FieldDefinition[fieldDef.size()]);

		return fd;
	}

}
