package org.gcube.portlets.admin.software_upload_wizard.server.softwaremanagers;

import java.io.File;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.gcube.common.core.contexts.GHNContext;
import org.gcube.common.core.resources.GCUBEService;
import org.gcube.common.core.resources.common.PlatformDescription;
import org.gcube.common.core.resources.service.MainPackage;
import org.gcube.common.core.resources.service.Package.GHNRequirement;
import org.gcube.common.core.resources.service.Package.GHNRequirement.Category;
import org.gcube.common.core.resources.service.Package.GHNRequirement.OpType;
import org.gcube.common.core.resources.service.Package.ScopeLevel;
import org.gcube.common.core.resources.service.PortType;
import org.gcube.common.core.resources.service.Software;
import org.gcube.common.core.resources.service.Software.Type;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.portlets.admin.software_upload_wizard.server.aslmanagers.ASLSessionManager;
import org.gcube.portlets.admin.software_upload_wizard.server.data.ScriptTransformFunction;
import org.gcube.portlets.admin.software_upload_wizard.server.data.SoftwareFile;
import org.gcube.portlets.admin.software_upload_wizard.server.logging.InjectLogger;
import org.gcube.portlets.admin.software_upload_wizard.server.softwaremanagers.filesmanager.FileManager;
import org.gcube.portlets.admin.software_upload_wizard.server.softwaremanagers.maven.deploy.IMavenDeployer;
import org.gcube.portlets.admin.software_upload_wizard.server.softwaremanagers.maven.is.IMavenRepositoryIS;
import org.gcube.portlets.admin.software_upload_wizard.server.softwaremanagers.registrationmanagers.ISoftwareSubmissionTask;
import org.gcube.portlets.admin.software_upload_wizard.server.softwaremanagers.softwaregateway.ISoftwareGatewayRegistrationManager;
import org.gcube.portlets.admin.software_upload_wizard.server.softwareprofile.Package;
import org.gcube.portlets.admin.software_upload_wizard.server.softwareprofile.ServiceProfile;
import org.gcube.portlets.admin.software_upload_wizard.server.util.XmlFormatter;
import org.gcube.portlets.admin.software_upload_wizard.shared.IOperationProgress;
import org.gcube.portlets.admin.software_upload_wizard.shared.OperationProgress;
import org.gcube.portlets.admin.software_upload_wizard.shared.OperationState;
import org.gcube.portlets.admin.software_upload_wizard.shared.filetypes.FileType;
import org.gcube.portlets.admin.software_upload_wizard.shared.filetypes.GarFileType;
import org.gcube.portlets.admin.software_upload_wizard.shared.filetypes.InstallScriptFileType;
import org.gcube.portlets.admin.software_upload_wizard.shared.filetypes.JarFileType;
import org.gcube.portlets.admin.software_upload_wizard.shared.filetypes.MiscFileType;
import org.gcube.portlets.admin.software_upload_wizard.shared.filetypes.RebootScriptFileType;
import org.gcube.portlets.admin.software_upload_wizard.shared.filetypes.UninstallScriptFileType;
import org.gcube.portlets.admin.software_upload_wizard.shared.rpc.maven.IMavenRepositoryInfo;
import org.gcube.portlets.admin.software_upload_wizard.shared.rpc.maven.MavenCoordinates;
import org.gcube.portlets.admin.software_upload_wizard.shared.softwareprofile.PackageData;
import org.gcube.portlets.admin.software_upload_wizard.shared.softwareprofile.ServiceData;
import org.gcube.portlets.admin.software_upload_wizard.shared.softwareprofile.PackageData.PackageType;
import org.gcube.portlets.admin.software_upload_wizard.shared.softwaretypes.ISoftwareTypeInfo;
import org.gcube.portlets.admin.software_upload_wizard.shared.softwaretypes.SoftwareTypeCode;
import org.gcube.portlets.admin.software_upload_wizard.shared.softwaretypes.SoftwareTypeInfo;
import org.slf4j.Logger;

import com.allen_sauer.gwt.log.client.Log;
import com.google.common.collect.Collections2;
import com.google.inject.Inject;

public class GCubeWebServiceSoftwareManager extends AbstractSoftwareManager {

	@InjectLogger
	Logger logger;

	@Inject
	IMavenDeployer mavenDeployer;

	@Inject
	private IMavenRepositoryIS mavenRepositoryIS;

	@Inject
	private FileManager fileManager;

	@Inject
	private ISoftwareGatewayRegistrationManager sgRegistrationManager;

	private static final SoftwareTypeCode CODE = SoftwareTypeCode.gCubeWebService;
	private static final String NAME = "gCube Web Service";
	private static final String DESCRIPTION = "<h1>gCube Web Service</h1>"
			+ "<p>A Web Service of the GCube Infrastructure.</p>"
			+ "<p>Java archives provided for the Main and Stubs packages will be uploaded on a maven repository with a generated Service Archive. A Service Profile with will be created and registered on the Software Gateway.</p>"
			+ "<h2>Wizard steps</h2>"
			+ "<ul>"
			+ "<li>User enters Service Profile data related to the Service</li>"
			+ "<li>User enters Service Profile data related to the Main Package</li>"
			+ "<li>User uploads a GAR and other optional files related to the Main Package</li>"
			+ "<li>User edits maven coordinates related to the Main Package artifact</li>"
			+ "<li>User edits maven dependencies related to the Main Package artifact</li>"
			+ "<li>User enters Service Profile data related to the Stubs Package</li>"
			+ "<li>User uploads a Jar archive related to the Stubs Package</li>"
			+ "<li>User edits maven coordinates related to the Stubs Package artifact</li>"
			+ "<li>User edits maven dependencies related to the Stubs Package artifact</li>"
			+ "<li>User enters generic software info, documentation and source code/binary URLs</li>"
			+ "<li>User specifies package maintainers and software changes</li>"
			+ "<li>User enters package installation, uninstallation and configuration notes and specifies dependencies</li>"
			+ "<li>User enters license agreement</li>"
			+ "<li>User reviews XML Service Profile and generated deliverables and submits the software to the platform.</li>"
			+ "</ul>";

	private static final String GROUPID_PREFIX = "org.gcube.";

	@Override
	public ServiceProfile generateInitialSoftwareProfile() {
		ServiceProfile profile = new ServiceProfile();

		ArrayList<FileType> mainPackageAllowedFileTypes = new ArrayList<FileType>();
		mainPackageAllowedFileTypes.add(new GarFileType());
		mainPackageAllowedFileTypes.add(new InstallScriptFileType(false,false));
		mainPackageAllowedFileTypes.add(new UninstallScriptFileType(false,false));
		mainPackageAllowedFileTypes.add(new RebootScriptFileType(false,false));
		mainPackageAllowedFileTypes.add(new MiscFileType());
		Package mainPackage = new Package(PackageType.Main,
				mainPackageAllowedFileTypes);
		profile.getService().getPackages().add(mainPackage);
		ArrayList<FileType> stubsPackageAllowedFileTypes = new ArrayList<FileType>();
		stubsPackageAllowedFileTypes.add(new JarFileType());
		Package stubsPackage = new Package(PackageType.Software,
				stubsPackageAllowedFileTypes);
		profile.getService().getPackages().add(stubsPackage);
		return profile;
	}

	@Override
	public ISoftwareTypeInfo getSoftwareTypeInfo() {
		return new SoftwareTypeInfo(CODE, NAME, DESCRIPTION);
	}

	@Override
	public String getPOM(Package softwarePackage) throws Exception {
		StringBuilder stringBuilder = new StringBuilder();
		stringBuilder
				.append("<project xmlns=\"http://maven.apache.org/POM/4.0.0\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://maven.apache.org/POM/4.0.0 http://maven.apache.org/maven-v4_0_0.xsd\">\n");
		stringBuilder.append("\t<modelVersion>4.0.0</modelVersion>\n");
		stringBuilder.append("\t<groupId>"
				+ getMavenCoordinates(softwarePackage).getGroupId()
				+ "</groupId>\n");
		stringBuilder.append("\t<artifactId>"
				+ getMavenCoordinates(softwarePackage).getArtifactId()
				+ "</artifactId>\n");
		stringBuilder.append("\t<version>"
				+ getMavenCoordinates(softwarePackage).getVersion()
				+ "</version>\n");
		stringBuilder.append("\t<packaging>"
				+ getMavenCoordinates(softwarePackage).getPackaging()
				+ "</packaging>\n");
		if (softwarePackage == getImportSession().getServiceProfile()
				.getService().getPackages().get(0)) {
			// Add stubs package coords
			stringBuilder.append("\t<dependencies>\n");
			MavenCoordinates stubsPackageCoordinates = getImportSession()
					.getServiceProfile().getService().getPackages().get(1)
					.getArtifactCoordinates();
			stringBuilder.append("\t\t<dependency>\n");
			stringBuilder.append("\t\t\t<groupId>"
					+ stubsPackageCoordinates.getGroupId() + "</groupId>\n");
			stringBuilder.append("\t\t\t<artifactId>"
					+ stubsPackageCoordinates.getArtifactId() + "</artifactId>\n");
			stringBuilder.append("\t\t\t<version>"
					+ stubsPackageCoordinates.getVersion() + "</version>\n");
			stringBuilder.append("\t\t\t<type>"+stubsPackageCoordinates.getPackaging()+"</type>\n");
			stringBuilder.append("\t\t</dependency>\n");
			
			//Add other user defined deps
			if (softwarePackage.getMavenDependencies().size() > 0) {
				for (MavenCoordinates dep : softwarePackage
						.getMavenDependencies()) {
					stringBuilder.append("\t\t<dependency>\n");
					stringBuilder.append("\t\t\t<groupId>" + dep.getGroupId()
							+ "</groupId>\n");
					stringBuilder.append("\t\t\t<artifactId>"
							+ dep.getArtifactId() + "</artifactId>\n");
					stringBuilder.append("\t\t\t<version>" + dep.getVersion()
							+ "</version>\n");
					stringBuilder.append("\t\t\t<type>jar</type>\n");
					stringBuilder.append("\t\t</dependency>\n");
				}

			}
			stringBuilder.append("\t</dependencies>\n");
		} else {
			if (softwarePackage.getMavenDependencies().size() > 0) {
				stringBuilder.append("\t<dependencies>\n");
				for (MavenCoordinates dep : softwarePackage
						.getMavenDependencies()) {
					stringBuilder.append("\t\t<dependency>\n");
					stringBuilder.append("\t\t\t<groupId>" + dep.getGroupId()
							+ "</groupId>\n");
					stringBuilder.append("\t\t\t<artifactId>"
							+ dep.getArtifactId() + "</artifactId>\n");
					stringBuilder.append("\t\t\t<version>" + dep.getVersion()
							+ "</version>\n");
					stringBuilder.append("\t\t\t<type>jar</type>\n");
					stringBuilder.append("\t\t</dependency>\n");
				}
				stringBuilder.append("\t</dependencies>\n");
			}
		}
		stringBuilder.append("</project>");
		return stringBuilder.toString();
	}

	@Override
	public MavenCoordinates getMavenCoordinates(Package softwarePackage)
			throws Exception {
		if (softwarePackage.getArtifactCoordinates() != null)
			return softwarePackage.getArtifactCoordinates();

		PackageData packageData = softwarePackage.getData();

		String artifactId = packageData.getName().toLowerCase();
		String groupId = GROUPID_PREFIX
				+ getImportSession().getServiceProfile().getService().getData()
						.getClazz();
		String version = packageData.getVersion().toString();
		String packaging = "jar";
		if (packageData.getPackageType() == PackageType.Main) {
			packaging = "gar";
		}

		// gcube infrastructure
		if (getImportSession().getScope().getInfrastructure().getName()
				.equals(ASLSessionManager.GCUBE_INFRASTRUCTURE))
			return new MavenCoordinates(groupId, artifactId, version
					+ SNAPSHOT_SUFFIX, packaging);

		// d4science infrastructure
		if (getImportSession().getScope().getInfrastructure().getName()
				.equals(ASLSessionManager.D4SCIENCE_INFRASTRUCTURE))
			return new MavenCoordinates(groupId, artifactId, version, packaging);

		throw new Exception("Unmanaged scope infrastructure");
	}

	@Override
	public String getServiceProfile(boolean withHeader) throws Exception {
		GCUBEService gcubeService = GHNContext
				.getImplementation(GCUBEService.class);

		ServiceData serviceData = getImportSession().getServiceProfile()
				.getService().getData();

		// Service attributes
		gcubeService.setServiceName(serviceData.getName());
		gcubeService.setDescription(serviceData.getDescription());
		gcubeService.setServiceClass(serviceData.getClazz());
		gcubeService.setVersion(serviceData.getVersion().toString());

		/** Start of Main Package section **/

		// Main Package - Attributes
		MainPackage gCubeMainSoftwarePackage = new MainPackage();
		Package mainSoftwarePackage = getImportSession().getServiceProfile()
				.getService().getPackages().get(0);
		PackageData mainSoftwarePackageData = mainSoftwarePackage.getData();

		gCubeMainSoftwarePackage.setName(mainSoftwarePackageData.getName());
		gCubeMainSoftwarePackage.setDescription(mainSoftwarePackageData
				.getDescription());

		String infrastructureName = getImportSession().getScope()
				.getInfrastructure().getName();
		if (infrastructureName.equals(ASLSessionManager.GCUBE_INFRASTRUCTURE))
			gCubeMainSoftwarePackage.setVersion(mainSoftwarePackageData
					.getVersion().toString() + SNAPSHOT_SUFFIX);
		else if (infrastructureName
				.equals(ASLSessionManager.D4SCIENCE_INFRASTRUCTURE))
			gCubeMainSoftwarePackage.setVersion(mainSoftwarePackageData
					.getVersion().toString());
		else
			throw new Exception(
					"Unmanaged scope infrastructure, unable to evaluate main package version");

		// Main Package - Maven coordinates
		MavenCoordinates mavenCoordinates = getMavenCoordinates(mainSoftwarePackage);
		gCubeMainSoftwarePackage
				.setMavenCoordinates(mavenCoordinates.getGroupId(),
						mavenCoordinates.getArtifactId(),
						mavenCoordinates.getVersion());

		// Main Package - Target Platform
		PlatformDescription targetPlatform = new PlatformDescription();
		targetPlatform.setName("Tomcat");
		targetPlatform.setVersion((short) 6);
		targetPlatform.setMinorVersion((short) 0);
		gCubeMainSoftwarePackage.setTargetPlatform(targetPlatform);

		gCubeMainSoftwarePackage.setMandatoryLevel(ScopeLevel.NONE);

		// Main Package - Platform requirement
		GHNRequirement requirement = new GHNRequirement();
		requirement.setCategory(Category.SITE_LOCATION);
		requirement.setOperator(OpType.GE);
		requirement.setRequirement("string");
		requirement.setValue("java1.6");
		gCubeMainSoftwarePackage.setGHNRequirements(Collections
				.singletonList(requirement));

		// Main Package - GAR Archive

		List<SoftwareFile> garFiles = mainSoftwarePackage.getFilesContainer()
				.getFilesWithFileType(GarFileType.NAME);
		if (garFiles.size() != 1)
			throw new Exception("Number of GAR files is " + garFiles.size());
		gCubeMainSoftwarePackage.setGarArchive(garFiles.get(0).getFilename());

		// Main Package - Port types
		for (org.gcube.portlets.admin.software_upload_wizard.shared.softwareprofile.PortType pt : mainSoftwarePackageData
				.getPortTypes()) {
			PortType tmp = new PortType();
			tmp.setName(pt.getName());
			gCubeMainSoftwarePackage.getPorttypes().add(tmp);
		}

		// Main Package - Scripts
		List<SoftwareFile> scripts;

		// Install
		scripts = mainSoftwarePackage.getFilesContainer().getFilesWithFileType(
				InstallScriptFileType.NAME);
		gCubeMainSoftwarePackage
				.setInstallScripts(new ArrayList<String>(Collections2
						.transform(scripts, new ScriptTransformFunction())));

		// Uninstall
		scripts = mainSoftwarePackage.getFilesContainer().getFilesWithFileType(
				UninstallScriptFileType.NAME);
		gCubeMainSoftwarePackage
				.setUninstallScripts(new ArrayList<String>(Collections2
						.transform(scripts, new ScriptTransformFunction())));

		// Reboot
		scripts = mainSoftwarePackage.getFilesContainer().getFilesWithFileType(
				RebootScriptFileType.NAME);
		gCubeMainSoftwarePackage
				.setRebootScripts(new ArrayList<String>(Collections2.transform(
						scripts, new ScriptTransformFunction())));

		gcubeService.getPackages().add(gCubeMainSoftwarePackage);
		/** End of Main Package section **/

		/** Start of Stubs Package section **/

		// Stubs Package
		Software gCubeStubsSoftwarePackage = new Software();
		Package stubsSoftwarePackage = getImportSession().getServiceProfile()
				.getService().getPackages().get(1);
		PackageData stubsSoftwarePackageData = stubsSoftwarePackage.getData();

		gCubeStubsSoftwarePackage.setName(stubsSoftwarePackageData.getName());
		gCubeStubsSoftwarePackage.setDescription(stubsSoftwarePackageData
				.getDescription());

		if (infrastructureName.equals(ASLSessionManager.GCUBE_INFRASTRUCTURE))
			gCubeStubsSoftwarePackage.setVersion(stubsSoftwarePackageData
					.getVersion().toString() + SNAPSHOT_SUFFIX);
		else if (infrastructureName
				.equals(ASLSessionManager.D4SCIENCE_INFRASTRUCTURE))
			gCubeStubsSoftwarePackage.setVersion(stubsSoftwarePackageData
					.getVersion().toString());
		else
			throw new Exception(
					"Unmanaged scope infrastructure, unable to evaluate main package version");

		gCubeStubsSoftwarePackage.setType(Type.library);

		// Stubs Package - Maven coordinates
		mavenCoordinates = getMavenCoordinates(stubsSoftwarePackage);
		gCubeStubsSoftwarePackage
				.setMavenCoordinates(mavenCoordinates.getGroupId(),
						mavenCoordinates.getArtifactId(),
						mavenCoordinates.getVersion());

		// Stubs Package - Files

		for (SoftwareFile file : stubsSoftwarePackage.getFilesContainer()
				.getFiles()) {
			gCubeStubsSoftwarePackage.getFiles().add(file.getFilename());
		}

		gcubeService.getPackages().add(gCubeStubsSoftwarePackage);

		/** Additional packages section **/

		if (getImportSession().getServiceProfile().getService().getPackages()
				.size() > 2) {
			// For each additional package
			for (int i = 2; i < getImportSession().getServiceProfile()
					.getService().getPackages().size(); i++) {

				// Stubs Package
				Software additionalStubsSoftwarePackage = new Software();
				Package additionalSoftwarePackage = getImportSession()
						.getServiceProfile().getService().getPackages().get(i);
				PackageData additionalSoftwarePackageData = additionalSoftwarePackage
						.getData();

				additionalStubsSoftwarePackage
						.setName(additionalSoftwarePackageData.getName());
				additionalStubsSoftwarePackage
						.setDescription(additionalSoftwarePackageData
								.getDescription());

				if (infrastructureName
						.equals(ASLSessionManager.GCUBE_INFRASTRUCTURE))
					additionalStubsSoftwarePackage
							.setVersion(additionalSoftwarePackageData
									.getVersion().toString() + SNAPSHOT_SUFFIX);
				else if (infrastructureName
						.equals(ASLSessionManager.D4SCIENCE_INFRASTRUCTURE))
					additionalStubsSoftwarePackage
							.setVersion(additionalSoftwarePackageData
									.getVersion().toString());
				else
					throw new Exception(
							"Unmanaged scope infrastructure, unable to evaluate main package version");

				additionalStubsSoftwarePackage.setType(Type.library);

				// Stubs Package - Maven coordinates
				mavenCoordinates = getMavenCoordinates(additionalSoftwarePackage);
				additionalStubsSoftwarePackage.setMavenCoordinates(
						mavenCoordinates.getGroupId(),
						mavenCoordinates.getArtifactId(),
						mavenCoordinates.getVersion());

				// Stubs Package - Files

				for (SoftwareFile file : additionalSoftwarePackage
						.getFilesContainer().getFiles()) {
					additionalStubsSoftwarePackage.getFiles().add(
							file.getFilename());
				}

				gcubeService.getPackages().add(additionalStubsSoftwarePackage);

			}
		}

		/** End of additional packages section **/

		StringWriter xml = new StringWriter();
		gcubeService.store(xml);

		String resultXML = XmlFormatter
				.prettyFormat(xml.toString(), withHeader);
		Log.trace("XML profile generated:\n\n" + resultXML);

		return resultXML;
	}

	@Override
	protected IMavenRepositoryInfo getTargetRepository() throws Exception {
		if (getImportSession().getScope().getInfrastructure().getName()
				.equals(ASLSessionManager.D4SCIENCE_INFRASTRUCTURE))
			return mavenRepositoryIS
					.getMavenRepository(IMavenRepositoryIS.RELEASES_REPO_ID);
		if (getImportSession().getScope().getInfrastructure().getName()
				.equals(ASLSessionManager.GCUBE_INFRASTRUCTURE))
			return mavenRepositoryIS
					.getMavenRepository(IMavenRepositoryIS.SNAPSHOTS_REPO_ID);

		throw new Exception("Unmanaged scope infrastructure");
	}

	@Override
	public boolean isAvailableForScope(GCUBEScope scope) {
		if (scope.getInfrastructure().getName()
				.equals(ASLSessionManager.GCUBE_INFRASTRUCTURE))
			return true;
		if (scope.getInfrastructure().getName()
				.equals(ASLSessionManager.D4SCIENCE_INFRASTRUCTURE))
			return true;
		return false;
	}

	@Override
	protected ISoftwareSubmissionTask createSofwareSubmissionTask() {
		try{
			ISoftwareSubmissionTask task = new GCubeWebServiceSubmissionTask();
			task.setTargetRepository(getTargetRepository());
			return task;
		}catch (Exception ex) {
			logger.error("Error occurred while creating software submission task.",ex);
			return null;
		}
	}

	private class GCubeWebServiceSubmissionTask implements
			ISoftwareSubmissionTask {

		private IOperationProgress operationProgress = new OperationProgress();
		private IMavenRepositoryInfo targetRepository;

		@Override
		public void run() {
			File serviceArchiveFile = null;
			File serviceArchivePomFile = null;
			ArrayList<File> artifactsPomFiles = new ArrayList<File>();

			try {
				logger.debug("Starting software deployment");
				// Deploy primary artifact
				operationProgress.setProgress(100, 0);
				operationProgress.setDetails("Deploying packages artifacts...");

				ArrayList<Package> packages = getImportSession()
						.getServiceProfile().getService().getPackages();

				// Deploy an artifact for each package
				for (int i = 0; i < packages.size(); i++) {
					logger.trace("Creating artifact POM for package #" + i);
					artifactsPomFiles.add(fileManager
							.createPomFile(getPOM(getImportSession()
									.getServiceProfile().getService()
									.getPackages().get(i))));
				}

				logger.trace("Deploying artifact #" + 0
						+ " on maven repository "
						+ targetRepository.getId());
				mavenDeployer.deploy(targetRepository,
						getImportSession().getServiceProfile().getService()
								.getPackages().get(0).getFilesContainer()
								.getFilesWithFileType(GarFileType.NAME).get(0)
								.getFile(), artifactsPomFiles.get(0), true);

				for (int i = 1; i < packages.size(); i++) {

					logger.trace("Deploying artifact #" + i
							+ "on maven repository "
							+ targetRepository.getId());

					mavenDeployer.deploy(targetRepository,
							getImportSession().getServiceProfile().getService()
									.getPackages().get(i).getFilesContainer()
									.getFilesWithFileType(JarFileType.NAME)
									.get(0).getFile(),
							artifactsPomFiles.get(i), true);
				}

				// Deploy service archive
				operationProgress.setProgress(100, 33);
				operationProgress.setDetails("Creating Service Archive...");

				serviceArchiveFile = fileManager.createServiveArchive(
						getServiceProfile(true), getMiscFiles(),
						getImportSession().getServiceProfile());

				logger.trace("Creating service archive POM file...");
				serviceArchivePomFile = fileManager
						.createPomFile(getPOM(getImportSession()
								.getServiceProfile()));

				// Deploy service archive
				operationProgress.setProgress(100, 66);
				operationProgress.setDetails("Deploying Service Archive...");

				logger.trace("Deploying service archive on maven repository "
						+ targetRepository.getId());
				mavenDeployer
						.deploy(targetRepository, serviceArchiveFile,
								serviceArchivePomFile, false,
								SERVICEARCHIVE_CLASSIFIER);

				// Register Profile
				operationProgress.setProgress(100, 99);
				operationProgress.setDetails("Registering Service Profile...");

				logger.trace("Registering Service Profile on software gateway...");
				sgRegistrationManager.registerProfile(getServiceProfile(true),
						getImportSession().getScope());

				operationProgress.setProgress(100, 100);
				operationProgress.setState(OperationState.COMPLETED);

				logger.debug("Deploy completed succesfully");
			} catch (Exception e) {
				logger.error("Error encountered during software submission.", e);
				operationProgress.setProgress(100, 0);
				operationProgress.setState(OperationState.FAILED);
				operationProgress
						.setDetails("Error encountered during software submission. "
								+ e.getMessage());
			} finally {
				// Delete garbage
				if (serviceArchiveFile != null)
					serviceArchiveFile.delete();
				if (serviceArchivePomFile != null)
					serviceArchivePomFile.delete();
				for (int i = 0; i < artifactsPomFiles.size(); i++) {
					if (artifactsPomFiles.get(i) != null)
						artifactsPomFiles.get(i).delete();
				}
			}
		}

		@Override
		public IOperationProgress getOperationProgress() {
			return operationProgress;
		}
		
		@Override
		public void setTargetRepository(IMavenRepositoryInfo targetRepository) {
			this.targetRepository = targetRepository;
		}

	}

}
