package org.gcube.portlets.admin.software_upload_wizard.server.softwaremanagers;

import java.io.File;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;

import org.gcube.common.core.contexts.GHNContext;
import org.gcube.common.core.resources.GCUBEService;
import org.gcube.common.core.resources.service.Package.ScopeLevel;
import org.gcube.common.core.resources.service.Software;
import org.gcube.common.core.resources.service.Software.Type;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.portlets.admin.software_upload_wizard.server.aslmanagers.ASLSessionManager;
import org.gcube.portlets.admin.software_upload_wizard.server.data.ScriptTransformFunction;
import org.gcube.portlets.admin.software_upload_wizard.server.data.SoftwareFile;
import org.gcube.portlets.admin.software_upload_wizard.server.logging.InjectLogger;
import org.gcube.portlets.admin.software_upload_wizard.server.softwaremanagers.filesmanager.FileManager;
import org.gcube.portlets.admin.software_upload_wizard.server.softwaremanagers.filesmanager.JarArchiveManager;
import org.gcube.portlets.admin.software_upload_wizard.server.softwaremanagers.maven.deploy.IMavenDeployer;
import org.gcube.portlets.admin.software_upload_wizard.server.softwaremanagers.maven.is.IMavenRepositoryIS;
import org.gcube.portlets.admin.software_upload_wizard.server.softwaremanagers.registrationmanagers.ISoftwareSubmissionTask;
import org.gcube.portlets.admin.software_upload_wizard.server.softwaremanagers.softwaregateway.ISoftwareGatewayRegistrationManager;
import org.gcube.portlets.admin.software_upload_wizard.server.softwareprofile.Package;
import org.gcube.portlets.admin.software_upload_wizard.server.softwareprofile.Service;
import org.gcube.portlets.admin.software_upload_wizard.server.softwareprofile.ServiceProfile;
import org.gcube.portlets.admin.software_upload_wizard.server.util.XmlFormatter;
import org.gcube.portlets.admin.software_upload_wizard.shared.IOperationProgress;
import org.gcube.portlets.admin.software_upload_wizard.shared.OperationProgress;
import org.gcube.portlets.admin.software_upload_wizard.shared.OperationState;
import org.gcube.portlets.admin.software_upload_wizard.shared.filetypes.FileType;
import org.gcube.portlets.admin.software_upload_wizard.shared.filetypes.InstallScriptFileType;
import org.gcube.portlets.admin.software_upload_wizard.shared.filetypes.JarFileType;
import org.gcube.portlets.admin.software_upload_wizard.shared.filetypes.MiscFileType;
import org.gcube.portlets.admin.software_upload_wizard.shared.filetypes.RebootScriptFileType;
import org.gcube.portlets.admin.software_upload_wizard.shared.filetypes.UninstallScriptFileType;
import org.gcube.portlets.admin.software_upload_wizard.shared.rpc.maven.IMavenRepositoryInfo;
import org.gcube.portlets.admin.software_upload_wizard.shared.rpc.maven.MavenCoordinates;
import org.gcube.portlets.admin.software_upload_wizard.shared.softwareprofile.PackageData;
import org.gcube.portlets.admin.software_upload_wizard.shared.softwareprofile.ServiceData;
import org.gcube.portlets.admin.software_upload_wizard.shared.softwareprofile.PackageData.PackageType;
import org.gcube.portlets.admin.software_upload_wizard.shared.softwaretypes.ISoftwareTypeInfo;
import org.gcube.portlets.admin.software_upload_wizard.shared.softwaretypes.SoftwareTypeCode;
import org.gcube.portlets.admin.software_upload_wizard.shared.softwaretypes.SoftwareTypeInfo;
import org.slf4j.Logger;

import com.allen_sauer.gwt.log.client.Log;
import com.google.common.collect.Collections2;
import com.google.inject.Inject;

public class LibrarySoftwareManager extends AbstractSoftwareManager {

	@InjectLogger
	Logger logger;
	
	@Inject
	IMavenDeployer mavenDeployer;
	
	@Inject
	private IMavenRepositoryIS mavenRepositoryIS;
	
	@Inject 
	private ISoftwareGatewayRegistrationManager sgRegistrationManager;

	@Inject
	private FileManager fileManager;

	private static final String THIRD_PARTY_SOFTWARE_CLASS = "externals";

	private static final SoftwareTypeCode CODE = SoftwareTypeCode.Library;
	private static final String NAME = "Library";
	// TODO Remove inline of variables
	private static String DESCRIPTION = "<h1>Library</h1>"
			+ "<p>A collection of reusable java software packages. Third party libraries are supported.</p>"
			+ "<p>A Software Archive will be created and registered on a gCube Maven repository along with the provided library jars. A Service Profile with a single service package will be created and registered on the Software Gateway. </p>"
			+ "<h2>Wizard steps</h2>"
			+ "<ul>"
			+ "<li>User enters Service Profile data</li>"
			+ "<li>User uploads several package related files</li>"
			+ "<li>User enters generic software info, documentation and source code/binary URLs</li>"
			+ "<li>User specifies package maintainers and software changes</li>"
			+ "<li>User enters package installation, uninstallation and configuration notes and specifies dependencies</li>"
			+ "<li>User enters license agreement</li>"
			+ "<li>User reviews XML Service Profile and generated deliverables and submits the software to the platform.</li>"
			+ "</ul>"
			+ "<h2>Requirements</h2>"
			+ "<ul>"
			+ "<li>The library must be compatible with JRE 1.6</li>"
			+ "</ul>"
			+ "<h2>Disclaimer</h2>"
			+ "<p>If multiple jar library files are provided, those archives will be repackaged into a single jar, thereby losing any signature on the original jar files.</p> ";

	@Override
	public ServiceProfile generateInitialSoftwareProfile() {
		ServiceProfile profile = new ServiceProfile();

		ArrayList<FileType> allowedFileTypes = new ArrayList<FileType>();
		allowedFileTypes.add(new JarFileType(true, true));
		allowedFileTypes.add(new InstallScriptFileType());
		allowedFileTypes.add(new UninstallScriptFileType());
		allowedFileTypes.add(new MiscFileType(false));
		Package pack = new Package(PackageType.Software, allowedFileTypes);
		profile.getService().getPackages().add(pack);

		return profile;
	}

	@Override
	public ISoftwareTypeInfo getSoftwareTypeInfo() {
		return new SoftwareTypeInfo(CODE, NAME, DESCRIPTION);
	}

	@Override
	public MavenCoordinates getMavenCoordinates(Package softwarePackage)
			throws Exception {
		Service service = getImportSession().getServiceProfile().getService();
		ServiceData serviceData = service.getData();
		PackageData mainPackageData = softwarePackage.getData();

		String artifactId = mainPackageData.getName().toLowerCase();
		String version = mainPackageData.getVersion().toString();

		// third party software
		if (getImportSession().getServiceProfile().isThirdPartySoftware())
			return new MavenCoordinates("org.gcube."
					+ THIRD_PARTY_SOFTWARE_CLASS, artifactId, version);

		String normalizedServiceClass = serviceData.getClazz().toLowerCase()
				.replaceAll("-", "").replaceAll("\\.", "");
		// gcube infrastructure
		if (getImportSession().getScope().getInfrastructure().getName()
				.equals(ASLSessionManager.GCUBE_INFRASTRUCTURE))
			return new MavenCoordinates("org.gcube." + normalizedServiceClass,
					artifactId, version + SNAPSHOT_SUFFIX);

		// d4science infrastructure
		if (getImportSession().getScope().getInfrastructure().getName()
				.equals(ASLSessionManager.D4SCIENCE_INFRASTRUCTURE))
			return new MavenCoordinates("org.gcube." + normalizedServiceClass,
					artifactId, version);

		throw new Exception("Unmanaged scope infrastructure");
	}

	@Override
	public String getServiceProfile(boolean withHeader) throws Exception {
		GCUBEService gcubeService = GHNContext
				.getImplementation(GCUBEService.class);

		gcubeService.setServiceName(getImportSession().getServiceProfile()
				.getService().getData().getName());
		gcubeService.setDescription(getImportSession().getServiceProfile()
				.getService().getData().getDescription());
		gcubeService.setVersion(getImportSession().getServiceProfile()
				.getService().getData().getVersion().toString());
		gcubeService.setServiceClass(getImportSession().getServiceProfile()
				.getService().getData().getClazz());

		Software softwarePackage = new Software();
		Package mainPackage = getImportSession().getServiceProfile()
				.getService().getPackages().get(0);
		PackageData mainSoftwarePackageData = mainPackage.getData();

		softwarePackage.setName(mainSoftwarePackageData.getName());
		softwarePackage
				.setDescription(mainSoftwarePackageData.getDescription());

		if (getImportSession().getServiceProfile().isThirdPartySoftware())
			softwarePackage.setVersion(mainSoftwarePackageData.getVersion()
					.toString());
		else {
			String infrastructure = getImportSession().getScope()
					.getInfrastructure().getName();
			if (infrastructure.equals(ASLSessionManager.GCUBE_INFRASTRUCTURE))
				softwarePackage.setVersion(mainSoftwarePackageData.getVersion()
						.toString() + SNAPSHOT_SUFFIX);
			else if (infrastructure
					.equals(ASLSessionManager.D4SCIENCE_INFRASTRUCTURE))
				softwarePackage.setVersion(mainSoftwarePackageData.getVersion()
						.toString());
			else
				throw new Exception("Unmanaged scope infrastructure");
		}

		MavenCoordinates mavenCoordinates = getMavenCoordinates(mainPackage);
		softwarePackage
				.setMavenCoordinates(mavenCoordinates.getGroupId(),
						mavenCoordinates.getArtifactId(),
						mavenCoordinates.getVersion());

		softwarePackage.setMandatoryLevel(ScopeLevel.NONE);

		softwarePackage.setType(Type.library);

		// Files list
		for (SoftwareFile file : mainPackage.getFilesContainer().getFiles()) {
			softwarePackage.getFiles().add(file.getFilename());
		}

		/** Set scripts **/
		List<SoftwareFile> scripts;

		// Install
		scripts = mainPackage.getFilesContainer().getFilesWithFileType(
				InstallScriptFileType.NAME);
		softwarePackage.setInstallScripts(new ArrayList<String>(Collections2
				.transform(scripts, new ScriptTransformFunction())));

		// Uninstall
		scripts = mainPackage.getFilesContainer().getFilesWithFileType(
				UninstallScriptFileType.NAME);
		softwarePackage.setUninstallScripts(new ArrayList<String>(Collections2
				.transform(scripts, new ScriptTransformFunction())));

		// Reboot
		scripts = mainPackage.getFilesContainer().getFilesWithFileType(
				RebootScriptFileType.NAME);
		softwarePackage.setRebootScripts(new ArrayList<String>(Collections2
				.transform(scripts, new ScriptTransformFunction())));

		gcubeService.getPackages().add(softwarePackage);

		StringWriter xml = new StringWriter();
		gcubeService.store(xml);

		String resultXML = XmlFormatter
				.prettyFormat(xml.toString(), withHeader);
		Log.trace("XML profile generated:\n\n" + resultXML);

		return resultXML;
	}

	@Override
	protected IMavenRepositoryInfo getTargetRepository() throws Exception {
		if (getImportSession().getServiceProfile().isThirdPartySoftware())
			return mavenRepositoryIS
					.getMavenRepository(IMavenRepositoryIS.EXTERNALS_REPO_ID);
		else {
			if (getImportSession().getScope().getInfrastructure().getName()
					.equals(ASLSessionManager.D4SCIENCE_INFRASTRUCTURE))
				return mavenRepositoryIS
						.getMavenRepository(IMavenRepositoryIS.RELEASES_REPO_ID);
			if (getImportSession().getScope().getInfrastructure().getName()
					.equals(ASLSessionManager.GCUBE_INFRASTRUCTURE))
				return mavenRepositoryIS
						.getMavenRepository(IMavenRepositoryIS.SNAPSHOTS_REPO_ID);
		}
		throw new Exception("Unmanaged scope infrastructure");
	}

	@Override
	public boolean isAvailableForScope(GCUBEScope scope) {
		if (scope.getInfrastructure().getName()
				.equals(ASLSessionManager.GCUBE_INFRASTRUCTURE))
			return true;
		if (scope.getInfrastructure().getName()
				.equals(ASLSessionManager.D4SCIENCE_INFRASTRUCTURE))
			return true;
		return false;
	}

	@Override
	protected ISoftwareSubmissionTask createSofwareSubmissionTask() {
		try{
			ISoftwareSubmissionTask task = new LibrarySubmissionTask();
			task.setTargetRepository(getTargetRepository());
			return task;
		}catch (Exception ex) {
			logger.error("Error occurred while creating software submission task.",ex);
			return null;
		}
	}

	private class LibrarySubmissionTask implements ISoftwareSubmissionTask {

		private IOperationProgress operationProgress = new OperationProgress();

		@Override
		public void run() {
			File serviceArchiveFile = null;
			File primaryArtifactPomFile = null;
			File serviceArchivePomFile = null;
			File repackagedJarFile = null;
			try {
				logger.debug("Starting software deployment");
				
				Package mainPackage = getImportSession().getServiceProfile()
						.getService().getPackages().get(0);
				List<SoftwareFile> jarSoftwareFiles = mainPackage
						.getFilesContainer().getFilesWithFileType(
								JarFileType.NAME);

				if (jarSoftwareFiles.size() == 1) {
					// If a single jar is provided

					// Deploy jar
					operationProgress.setProgress(100, 0);
					operationProgress
							.setDetails("Deploying primary artifact...");

					logger.trace("Creating primary artifact POM...");
					logger.trace("Generated POM for primary artifact:\n" + getPOM(mainPackage));
					primaryArtifactPomFile = fileManager
							.createPomFile(getPOM(mainPackage));
					
					logger.trace("Deploying primary artifact on maven repository " + targetRepository.getId());
					mavenDeployer.deploy(
							targetRepository, jarSoftwareFiles.get(0)
							.getFile(), primaryArtifactPomFile);
					
				} else if (jarSoftwareFiles.size() > 1) {
					// If multiple jars are provided
					// Repackage jars
					operationProgress.setProgress(100, 0);
					operationProgress
							.setDetails("Repackaging jar primary artifact...");
					
					logger.trace("Found multiple jars in package, repackaging jars...");
					ArrayList<File> jarFiles = new ArrayList<File>();
					for (SoftwareFile sf : jarSoftwareFiles) {
						jarFiles.add(sf.getFile());
					}
					repackagedJarFile = JarArchiveManager.mergeJars(jarFiles);

					// Deploy jar
					operationProgress.setProgress(100, 25);
					operationProgress
							.setDetails("Deploying primary artifact...");
					
					logger.trace("Creating primary artifact POM...");
					primaryArtifactPomFile = fileManager
							.createPomFile(getPOM(mainPackage));

					logger.trace("Deploying primary artifact on maven repository " + targetRepository.getId());
					mavenDeployer.deploy(
							targetRepository, repackagedJarFile,
							primaryArtifactPomFile);
				} else
					throw new Exception("Number of jar files in package < 0");

				// Create service archive
				operationProgress.setProgress(100, 50);
				operationProgress.setDetails("Creating Service Archive...");
				
				logger.trace("Creating Service Archive...");
				serviceArchiveFile = fileManager.createServiveArchive(
						getServiceProfile(true), getMiscFiles(),
						getImportSession().getServiceProfile());
				
				logger.trace("Creating service archive POM file...");
				serviceArchivePomFile = fileManager
						.createPomFile(getPOM(getImportSession()
								.getServiceProfile()));

				// Deploy service archive
				operationProgress.setProgress(100, 75);
				operationProgress.setDetails("Deploying Service Archive...");
				
				logger.trace("Deploying service archive on maven repository " + targetRepository.getId() );
				mavenDeployer.deploy(
						targetRepository, serviceArchiveFile,
						serviceArchivePomFile, false, SERVICEARCHIVE_CLASSIFIER);

				// Register Profile
				operationProgress.setProgress(100, 90);
				operationProgress.setDetails("Registering Service Profile...");
				
				logger.trace("Registering Service Profile on software gateway...");
				sgRegistrationManager.registerProfile(getServiceProfile(true),
						getImportSession().getScope());

				operationProgress.setProgress(100, 100);
				operationProgress.setState(OperationState.COMPLETED);

				logger.debug("Deploy completed succesfully");
			} catch (Exception e) {
				logger.error("Error encountered during software submission", e);
				operationProgress.setProgress(100, 0);
				operationProgress.setDetails("Error encountered during software submission. " + e.getMessage());
				operationProgress.setState(OperationState.FAILED);
			} finally {
				// Delete garbage
				if (serviceArchiveFile != null)
					serviceArchiveFile.delete();
				if (serviceArchivePomFile != null)
					serviceArchivePomFile.delete();
				if (primaryArtifactPomFile != null)
					primaryArtifactPomFile.delete();
				if (repackagedJarFile != null)
					repackagedJarFile.delete();
			}

		}

		@Override
		public IOperationProgress getOperationProgress() {
			return operationProgress;
		}
		
		private IMavenRepositoryInfo targetRepository;
		@Override
		public void setTargetRepository(IMavenRepositoryInfo targetRepository) {
			this.targetRepository = targetRepository;
		}
	}

}
