package org.gcube.portlets.admin.software_upload_wizard.server.softwaremanagers;

import java.io.File;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.gcube.common.core.contexts.GHNContext;
import org.gcube.common.core.resources.GCUBEService;
import org.gcube.common.core.resources.common.PlatformDescription;
import org.gcube.common.core.resources.service.Package.GHNRequirement;
import org.gcube.common.core.resources.service.Package.GHNRequirement.Category;
import org.gcube.common.core.resources.service.Package.GHNRequirement.OpType;
import org.gcube.common.core.resources.service.Package.ScopeLevel;
import org.gcube.common.core.resources.service.Software;
import org.gcube.common.core.resources.service.Software.Type;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.portlets.admin.software_upload_wizard.server.aslmanagers.ASLSessionManager;
import org.gcube.portlets.admin.software_upload_wizard.server.data.ScriptTransformFunction;
import org.gcube.portlets.admin.software_upload_wizard.server.data.SoftwareFile;
import org.gcube.portlets.admin.software_upload_wizard.server.logging.InjectLogger;
import org.gcube.portlets.admin.software_upload_wizard.server.softwaremanagers.filesmanager.FileManager;
import org.gcube.portlets.admin.software_upload_wizard.server.softwaremanagers.maven.deploy.IMavenDeployer;
import org.gcube.portlets.admin.software_upload_wizard.server.softwaremanagers.maven.is.IMavenRepositoryIS;
import org.gcube.portlets.admin.software_upload_wizard.server.softwaremanagers.registrationmanagers.ISoftwareSubmissionManager;
import org.gcube.portlets.admin.software_upload_wizard.server.softwaremanagers.registrationmanagers.ISoftwareSubmissionTask;
import org.gcube.portlets.admin.software_upload_wizard.server.softwaremanagers.softwaregateway.ISoftwareGatewayRegistrationManager;
import org.gcube.portlets.admin.software_upload_wizard.server.softwareprofile.Package;
import org.gcube.portlets.admin.software_upload_wizard.server.softwareprofile.ServiceProfile;
import org.gcube.portlets.admin.software_upload_wizard.server.util.XmlFormatter;
import org.gcube.portlets.admin.software_upload_wizard.shared.IOperationProgress;
import org.gcube.portlets.admin.software_upload_wizard.shared.OperationProgress;
import org.gcube.portlets.admin.software_upload_wizard.shared.OperationState;
import org.gcube.portlets.admin.software_upload_wizard.shared.filetypes.FileType;
import org.gcube.portlets.admin.software_upload_wizard.shared.filetypes.InstallScriptFileType;
import org.gcube.portlets.admin.software_upload_wizard.shared.filetypes.MiscFileType;
import org.gcube.portlets.admin.software_upload_wizard.shared.filetypes.RebootScriptFileType;
import org.gcube.portlets.admin.software_upload_wizard.shared.filetypes.UninstallScriptFileType;
import org.gcube.portlets.admin.software_upload_wizard.shared.filetypes.WarFileType;
import org.gcube.portlets.admin.software_upload_wizard.shared.rpc.maven.IMavenRepositoryInfo;
import org.gcube.portlets.admin.software_upload_wizard.shared.rpc.maven.MavenCoordinates;
import org.gcube.portlets.admin.software_upload_wizard.shared.softwareprofile.PackageData;
import org.gcube.portlets.admin.software_upload_wizard.shared.softwareprofile.ServiceData;
import org.gcube.portlets.admin.software_upload_wizard.shared.softwareprofile.PackageData.PackageType;
import org.gcube.portlets.admin.software_upload_wizard.shared.softwaretypes.ISoftwareTypeInfo;
import org.gcube.portlets.admin.software_upload_wizard.shared.softwaretypes.SoftwareTypeCode;
import org.gcube.portlets.admin.software_upload_wizard.shared.softwaretypes.SoftwareTypeInfo;
import org.slf4j.Logger;

import com.allen_sauer.gwt.log.client.Log;
import com.google.common.collect.Collections2;
import com.google.inject.Inject;

public class WebAppSoftwareManager extends AbstractSoftwareManager implements
		ISoftwareSubmissionManager {

	@InjectLogger
	Logger logger;
	
	@Inject
	IMavenDeployer mavenDeployer;
	
	@Inject
	private IMavenRepositoryIS mavenRepositoryIS;
	
	@Inject
	private FileManager fileManager;
	
	@Inject 
	private ISoftwareGatewayRegistrationManager sgRegistrationManager;

	private static final SoftwareTypeCode CODE = SoftwareTypeCode.WebApp;
	private static final String NAME = "Web Application";
	private static final String DESCRIPTION = "<h1>Web Application</h1>"
			+ "<p>A java application utilizing web browser technologies to accomplish one or more tasks over a network, through a web browser. Third party applications are supported.</p>"
			+ "<p>The user provided war archive and the generated Service Archive will be registered on a gCube Maven repository. A Service Profile with a single service package will be created and registered on the Software Gateway.</p>"
			+ "<h2>Wizard steps</h2>"
			+ "<ul>"
			+ "<li>User enters Service Profile data</li>"
			+ "<li>User uploads several package related files</li>"
			+ "<li>User enters generic software info, documentation and source code/binary URLs</li>"
			+ "<li>User specifies package maintainers and software changes</li>"
			+ "<li>User enters package installation, uninstallation and configuration notes and specifies dependencies</li>"
			+ "<li>User enters license agreement</li>"
			+ "<li>User reviews XML Service Profile and generated deliverables and submits the software to the platform.</li>"
			+ "</ul>"
			+ "<h2>Requirements</h2>"
			+ "<ul>"
			+ "<li>The web application must be compatible with Tomcat 6.0 platform</li>"
			+ "<li>The web application must be compatible with JRE 1.6</li>"
			+ "</ul>";

	public static final String CLASS = "WebApp";

	@Override
	public ServiceProfile generateInitialSoftwareProfile() {
		ServiceProfile profile = new ServiceProfile();
		profile.getService().getData().setClazz("WebApp");

		ArrayList<FileType> allowedFileTypes = new ArrayList<FileType>();
		allowedFileTypes.add(new WarFileType(false, true));
		allowedFileTypes.add(new InstallScriptFileType());
		allowedFileTypes.add(new UninstallScriptFileType());
		allowedFileTypes.add(new MiscFileType(false));
		Package pack = new Package(PackageType.Software, allowedFileTypes);
		profile.getService().getPackages().add(pack);
		return profile;
	}

	@Override
	public ISoftwareTypeInfo getSoftwareTypeInfo() {
		return new SoftwareTypeInfo(CODE, NAME, DESCRIPTION);
	}

//	@Override
//	public String getPOM(Package softwarePackage) throws Exception {
//		ArrayList<Package> packages = getImportSession().getServiceProfile()
//				.getService().getPackages();
//		if (!(softwarePackage == packages.get(0)))
//			throw new Exception(
//					"Cannot accept package for POM creation, invalid package");
//
//		StringBuilder stringBuilder = new StringBuilder();
//		stringBuilder
//				.append("<project xmlns=\"http://maven.apache.org/POM/4.0.0\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://maven.apache.org/POM/4.0.0 http://maven.apache.org/maven-v4_0_0.xsd\">\n");
//		stringBuilder.append("\t<modelVersion>4.0.0</modelVersion>\n");
//		stringBuilder.append("\t<groupId>"
//				+ getMavenCoordinates(softwarePackage).getGroupId()
//				+ "</groupId>\n");
//		stringBuilder.append("\t<artifactId>"
//				+ getMavenCoordinates(softwarePackage).getArtifactId()
//				+ "</artifactId>\n");
//		stringBuilder.append("\t<version>"
//				+ getMavenCoordinates(softwarePackage).getVersion()
//				+ "</version>\n");
//		stringBuilder.append("\t<packaging>war</packaging>\n");
//		if (softwarePackage.getMavenDependencies().size() > 0) {
//			stringBuilder.append("\t<dependencies>\n");
//			for (MavenCoordinates dep : softwarePackage.getMavenDependencies()) {
//				stringBuilder.append("\t\t<dependency>\n");
//				stringBuilder.append("\t\t\t<groupId>" + dep.getGroupId()
//						+ "</groupId>\n");
//				stringBuilder.append("\t\t\t<artifactId>" + dep.getArtifactId()
//						+ "</artifactId>\n");
//				stringBuilder.append("\t\t\t<version>" + dep.getVersion()
//						+ "</version>\n");
//				stringBuilder.append("\t\t\t<type>war</type>\n");
//				stringBuilder.append("\t\t</dependency>\n");
//			}
//			stringBuilder.append("\t</dependencies>\n");
//		}
//		stringBuilder.append("</project>");
//		return stringBuilder.toString();
//
//	}

	@Override
	public MavenCoordinates getMavenCoordinates(Package softwarePackage)
			throws Exception {
		PackageData mainPackage = softwarePackage.getData();

		String artifactId = mainPackage.getName().toLowerCase();
		String version = mainPackage.getVersion().toString();

		// third party software
		if (getImportSession().getServiceProfile().isThirdPartySoftware())
			return new MavenCoordinates("org.gcube.externals", artifactId,
					version, "war");

		// gcube infrastructure
		if (getImportSession().getScope().getInfrastructure().getName()
				.equals(ASLSessionManager.GCUBE_INFRASTRUCTURE))
			return new MavenCoordinates("org.gcube.webapps", artifactId,
					version + SNAPSHOT_SUFFIX, "war");

		// d4science infrastructure
		if (getImportSession().getScope().getInfrastructure().getName()
				.equals(ASLSessionManager.D4SCIENCE_INFRASTRUCTURE))
			return new MavenCoordinates("org.gcube.webapps", artifactId,
					version, "war");

		throw new Exception("Unmanaged scope infrastructure");
	}

	@Override
	public String getServiceProfile(boolean withHeader) throws Exception {
		GCUBEService gcubeService = GHNContext
				.getImplementation(GCUBEService.class);

		ServiceData serviceData = getImportSession().getServiceProfile()
				.getService().getData();

		/*
		 * <Description>**DESCRIPTION**</Description> <Class>WebApp</Class>
		 * <Name>**APP CATEGORY**</Name> <Version>1.0.0</Version>
		 */
		gcubeService.setServiceName(serviceData.getName());
		gcubeService.setDescription(serviceData.getDescription());
		gcubeService.setServiceClass(serviceData.getClazz());
		gcubeService.setVersion(serviceData.getVersion().toString());

		/*
		 * <Packages> <Software> <Description>**DESCRIPTION**</Description>
		 * <Name>**APP NAME**</Name> <Version>**VERSION**</Version>
		 */
		Software softwarePackage = new Software();
		Package mainSoftwarePackage = getImportSession().getServiceProfile()
				.getService().getPackages().get(0);
		PackageData mainSoftwarePackageData = mainSoftwarePackage.getData();

		softwarePackage.setName(mainSoftwarePackageData.getName());
		softwarePackage
				.setDescription(mainSoftwarePackageData.getDescription());
		if (getImportSession().getServiceProfile().isThirdPartySoftware())
			softwarePackage.setVersion(mainSoftwarePackageData.getVersion()
					.toString());
		else {
			String infrastructure = getImportSession().getScope()
					.getInfrastructure().getName();
			if (infrastructure.equals(ASLSessionManager.GCUBE_INFRASTRUCTURE))
				softwarePackage.setVersion(mainSoftwarePackageData.getVersion()
						.toString() + SNAPSHOT_SUFFIX);
			else if (infrastructure
					.equals(ASLSessionManager.D4SCIENCE_INFRASTRUCTURE))
				softwarePackage.setVersion(mainSoftwarePackageData.getVersion()
						.toString());
			else
				throw new Exception("Unmanaged scope infrastructure");
		}

		/*
		 * <Main> <Description>...</Description> <Name>myapp</Name>
		 * <Version>1.0.0</Version> <MavenCoordinates>
		 * <groupId>org.gcube.customwars</groupId>
		 * <artifactId>myapp</artifactId> <version>1.0.0</version>
		 * </MavenCoordinates> </Main>
		 */

		MavenCoordinates mavenCoordinates = getMavenCoordinates(mainSoftwarePackage);
		softwarePackage
				.setMavenCoordinates(mavenCoordinates.getGroupId(),
						mavenCoordinates.getArtifactId(),
						mavenCoordinates.getVersion());

		/*
		 * <TargetPlatform> <Name>Tomcat</Name> <Version>6</Version>
		 * <MinorVersion>0</MinorVersion> </TargetPlatform>
		 */
		PlatformDescription targetPlatform = new PlatformDescription();
		targetPlatform.setName("Tomcat");
		targetPlatform.setVersion((short) 6);
		targetPlatform.setMinorVersion((short) 0);
		softwarePackage.setTargetPlatform(targetPlatform);

		/*
		 * <MultiVersion value="true" /> <Mandatory level="NONE" /> <Shareable
		 * level="NONE" /> <GHNRequirements> <Requirement category="Site"
		 * operator="ge" requirement="string" value="java1.6" />
		 * </GHNRequirements> <SpecificData/> <Type>webapplication</Type>
		 */

		// softwarePackage.setMultiVersion(true);
		softwarePackage.setMandatoryLevel(ScopeLevel.NONE);
		// softwarePackage.setSharingLevel(ScopeLevel.NONE);

		GHNRequirement requirement = new GHNRequirement();
		requirement.setCategory(Category.SITE_LOCATION);
		requirement.setOperator(OpType.GE);
		requirement.setRequirement("string");
		requirement.setValue("java1.6");
		softwarePackage.setGHNRequirements(Collections
				.singletonList(requirement));
		softwarePackage.setType(Type.webapplication);

		/*
		 * <EntryPoint>**APPLICATION ENTRY POINT 1**</EntryPoint>
		 * <EntryPoint>**APPLICATION ENTRY POINT N**</EntryPoint> <Files>
		 * <File>**NAME OF THE WAR**</File> </Files> </Software> </Packages>
		 * </Profile> </Resource>
		 */

		softwarePackage.getEntrypoints().addAll(
				mainSoftwarePackageData.getEntrypoints());
		for (SoftwareFile file : mainSoftwarePackage.getFilesContainer()
				.getFiles()) {
			softwarePackage.getFiles().add(file.getFilename());
		}

		/** Set scripts **/
		List<SoftwareFile> scripts;

		// Install
		scripts = mainSoftwarePackage.getFilesContainer().getFilesWithFileType(
				InstallScriptFileType.NAME);
		softwarePackage.setInstallScripts(new ArrayList<String>(Collections2
				.transform(scripts, new ScriptTransformFunction())));

		// Uninstall
		scripts = mainSoftwarePackage.getFilesContainer().getFilesWithFileType(
				UninstallScriptFileType.NAME);
		softwarePackage.setUninstallScripts(new ArrayList<String>(Collections2
				.transform(scripts, new ScriptTransformFunction())));

		// Reboot
		scripts = mainSoftwarePackage.getFilesContainer().getFilesWithFileType(
				RebootScriptFileType.NAME);
		softwarePackage.setRebootScripts(new ArrayList<String>(Collections2
				.transform(scripts, new ScriptTransformFunction())));

		gcubeService.getPackages().add(softwarePackage);

		StringWriter xml = new StringWriter();
		gcubeService.store(xml);

		String resultXML = XmlFormatter
				.prettyFormat(xml.toString(), withHeader);
		Log.trace("XML profile generated:\n\n" + resultXML);

		return resultXML;
	}

	@Override
	protected IMavenRepositoryInfo getTargetRepository() throws Exception {
		if (getImportSession().getServiceProfile().isThirdPartySoftware())
			return mavenRepositoryIS
					.getMavenRepository(IMavenRepositoryIS.EXTERNALS_REPO_ID);
		else {
			if (getImportSession().getScope().getInfrastructure().getName()
					.equals(ASLSessionManager.D4SCIENCE_INFRASTRUCTURE))
				return mavenRepositoryIS
						.getMavenRepository(IMavenRepositoryIS.RELEASES_REPO_ID);
			if (getImportSession().getScope().getInfrastructure().getName()
					.equals(ASLSessionManager.GCUBE_INFRASTRUCTURE))
				return mavenRepositoryIS
						.getMavenRepository(IMavenRepositoryIS.SNAPSHOTS_REPO_ID);
		}
		throw new Exception("Unmanaged scope infrastructure");
	}

	@Override
	public boolean isAvailableForScope(GCUBEScope scope) {
		return true;
	}

	@Override
	protected ISoftwareSubmissionTask createSofwareSubmissionTask() {
		try{
			ISoftwareSubmissionTask task = new WebAppSubmissionTask();
			task.setTargetRepository(getTargetRepository());
			return task;
		}catch (Exception ex) {
			logger.error("Error occurred while creating software submission task.",ex);
			return null;
		}
	}

	private class WebAppSubmissionTask implements ISoftwareSubmissionTask {

		private IOperationProgress operationProgress = new OperationProgress();
		private IMavenRepositoryInfo targetRepository;

		@Override
		public void run() {
			File serviceArchiveFile = null;
			File primaryArtifactPomFile = null;
			File serviceArchivePomFile = null;
			try {
				logger.debug("Starting software deployment");
				// Deploy primary artifact
				operationProgress.setProgress(100, 0);
				operationProgress.setDetails("Deploying primary artifact...");
				
				logger.trace("Creating primary artifact POM...");
				primaryArtifactPomFile = fileManager
						.createPomFile(getPOM(getImportSession()
								.getServiceProfile().getService().getPackages()
								.get(0)));

				logger.trace("Deploying primary artifact on maven repository " + targetRepository.getId());
				mavenDeployer.deploy(targetRepository, getImportSession()
						.getServiceProfile().getService().getPackages()
						.get(0).getFilesContainer()
						.getFilesWithFileType(WarFileType.NAME).get(0)
						.getFile(), primaryArtifactPomFile, true);

				// Deploy service archive
				operationProgress.setProgress(100, 33);
				operationProgress.setDetails("Creating Service Archive...");

				
				serviceArchiveFile = fileManager.createServiveArchive(
						getServiceProfile(true), getMiscFiles(),
						getImportSession().getServiceProfile());
				
				logger.trace("Creating service archive POM file...");
				serviceArchivePomFile = fileManager
						.createPomFile(getPOM(getImportSession()
								.getServiceProfile()));

				// Deploy service archive
				operationProgress.setProgress(100, 66);
				operationProgress.setDetails("Deploying Service Archive...");

				logger.trace("Deploying service archive on maven repository " + targetRepository.getId());
				mavenDeployer.deploy(targetRepository,
						serviceArchiveFile, serviceArchivePomFile, false, SERVICEARCHIVE_CLASSIFIER);

				// Register Profile
				operationProgress.setProgress(100, 99);
				operationProgress.setDetails("Registering Service Profile...");

				logger.trace("Registering Service Profile on software gateway...");
				sgRegistrationManager.registerProfile(getServiceProfile(true),
						getImportSession().getScope());

				operationProgress.setProgress(100, 100);
				operationProgress.setState(OperationState.COMPLETED);
				
				logger.debug("Deploy completed succesfully");
			} catch (Exception e) {
				logger.error("Error encountered during software submission.", e);
				operationProgress.setProgress(100, 0);
				operationProgress.setState(OperationState.FAILED);
				operationProgress.setDetails("Error encountered during software submission. " + e.getMessage());
			} finally {
				// Delete garbage
				if (serviceArchiveFile != null)
					serviceArchiveFile.delete();
				if (serviceArchivePomFile != null)
					serviceArchivePomFile.delete();
				if (primaryArtifactPomFile != null)
					primaryArtifactPomFile.delete();
			}
		}

		@Override
		public IOperationProgress getOperationProgress() {
			return operationProgress;
		}

		@Override
		public void setTargetRepository(IMavenRepositoryInfo targetRepository) {
			this.targetRepository = targetRepository;
		}
	}

}
