package org.gcube.vremanagement.executor.stubs;

import java.util.ArrayList;
import java.util.List;

import org.apache.axis.message.addressing.EndpointReferenceType;
import org.gcube.common.core.contexts.GCUBERemotePortTypeContext;
import org.gcube.common.core.faults.GCUBERetrySameException;
import org.gcube.common.core.faults.GCUBEUnrecoverableException;
import org.gcube.common.core.informationsystem.client.RPDocument;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.security.GCUBESecurityManager;
import org.gcube.common.core.types.VOID;
import org.gcube.common.core.utils.logging.GCUBEClientLog;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.vremanagement.executor.stubs.service.TaskServiceAddressingLocator;

/**
 * A high-level call for interacting with a running task.
 * The call can return local proxies to the remote task, subscribe {@link TaskMonitor monitors} to its key changes, and stop its execution.
 *  
 * @author Fabio Simeoni (University of Strathclyde)
 *
 */
public class TaskCall extends BaseCall {

	/** Class logger */
	static GCUBELog logger = new GCUBEClientLog(TaskCall.class);
	
	/**
	 * Creates an instance for a task with a given name and in a given scope, optionally using a security manager.
	 * @param taskName the name of the task.
	 * @param scope the scope.
	 * @param securityManager (optional) the security manager.
	 * @throws Exception if the instance could not be created.
	 */
	public TaskCall(String taskName, GCUBEScope scope, GCUBESecurityManager ... securityManager) throws Exception {
		super(taskName,scope,securityManager);
	}
	
	/**
	 * Creates an instance for a task at a given endpoint and in a given scope, optionally using a security manager.
	 * @param epr the endpoint.
	 * @param scope the scope.
	 * @param securityManager (optional) the security manager.
	 * @throws Exception if the instance could not be created.
	 */
	public TaskCall(EndpointReferenceType epr, GCUBEScope scope, GCUBESecurityManager ... securityManager) throws Exception {
		super(scope,securityManager);
		this.setEndpointReference(epr);
	}
		
	/**{@inheritDoc}*/ 
	protected String getPortTypeName() {return Utils.TASK_PT_NAME;}
		
	/**
	 * Stops the task.
	 * @throws Exception if the call failed due to a local error.
	 * @throws GCUBEUnrecoverableException if the task does not support this call.
	 * * @throws GCUBERetrySameException if the task could not be stopped.
	 */
	public synchronized void stop() throws Exception, GCUBEUnrecoverableException,GCUBERetrySameException{
		
		new WSCallHandler() {
			@Override public void interact(EndpointReferenceType epr) throws Exception {	
				try {
					TaskPortType task = new TaskServiceAddressingLocator().getTaskPortTypePort(epr);
					task = GCUBERemotePortTypeContext.getProxy(task, getScopeManager(), getSecurityManager());
					task.stop(new VOID());
				}
				catch(Exception e) { //we do not want best-effort here, just discovery...this a way to stop it at first failure
					throw new GCUBEUnrecoverableException(e);
				}
			}
			
			/**{@inheritDoc}*/ //@TODO remove at next gcore release
			@Override protected List<RPDocument> findWSResources() throws Exception {
				List<RPDocument> rpds = new ArrayList<RPDocument>();
				//filter by port-type name in case there are multiple ws-resource types per service
				for (RPDocument d : super.findWSResources())
					if (d.getEndpoint().getAddress().getPath().endsWith(TaskCall.this.getPortTypeName())) rpds.add(d);
				return rpds;
			}
		}.run();
		
	}
	
	@Override
	//@TODO remove at the next release of gCore
	public void setEndpointReference(EndpointReferenceType epr) {
		super.setEndpointReference(epr);
	}
	
	/**
	 * Returns a proxy for the remote task.
	 * @return the proxy.
	 * @throws Exception if the proxy could not be returned.
	 */
	public synchronized TaskProxy getProxy() throws Exception {
		
		final ResultHolder<TaskProxy> result = new ResultHolder<TaskProxy>();
		new WSCallHandler() {
			@Override public void interact(EndpointReferenceType epr) throws Exception {	
				try{
					result.value=new TaskProxy(epr);
				}
				catch(Exception e) { //we do not want best-effort here, just discovery...this a way to stop it at first failure
					throw new GCUBEUnrecoverableException(e);
				}
			}
			/**{@inheritDoc}*/ //@TODO remove at next gcore release
			@Override protected List<RPDocument> findWSResources() throws Exception {
				List<RPDocument> rpds = new ArrayList<RPDocument>();
				for (RPDocument d : super.findWSResources())
					if (d.getEndpoint().getAddress().getPath().endsWith(TaskCall.this.getPortTypeName())) rpds.add(d);
				return rpds;
			}
		}.run();
		return result.value;
	}
	
	/**
	 * Subscribes a monitor for changes to the task.
	 * The call is then synchronised to compensate for state changes that may have occurred prior or during subscription. 
	 * @param monitor the monitor.
	 * @throws Exception if the monitor could not be subscribed.
	 */
	public void subscribe(final TaskMonitor monitor) throws Exception {
		
		new WSCallHandler() {
			@Override public void interact(EndpointReferenceType epr) throws Exception {	
				try {
					monitor.subscribe(epr);
				}
				catch(Exception e) { //we do not want best-effort here, just discovery...this a way to stop it at first failure
					throw new GCUBEUnrecoverableException(e);
				}
			}
			/**{@inheritDoc}*/ //@TODO remove at next gcore release
			@Override protected List<RPDocument> findWSResources() throws Exception {
				List<RPDocument> rpds = new ArrayList<RPDocument>();
				for (RPDocument d : super.findWSResources())
					if (d.getEndpoint().getAddress().getPath().endsWith(TaskCall.this.getPortTypeName())) rpds.add(d);
				return rpds;
			}
		}.run();
	}


}
