package org.gcube.vremanagement.executor.stubs;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;

import org.apache.axis.message.MessageElement;
import org.apache.axis.message.addressing.EndpointReferenceType;
import org.gcube.common.core.utils.logging.GCUBEClientLog;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.vremanagement.executor.stubs.service.TaskServiceAddressingLocator;
import org.oasis.wsrf.properties.GetMultipleResourcePropertiesResponse;
import org.oasis.wsrf.properties.GetMultipleResourceProperties_Element;

/**
 * A local call for a remote running task.
 * The call can synchronise its state with the remote task, subscribe {@link TaskMonitor monitors} to its key changes, and stop its execution.
 *  
 * @author Fabio Simeoni (University of Strathclyde)
 *
 */
public class TaskProxy {

	/** Class logger */
	static GCUBELog logger = new GCUBEClientLog(TaskProxy.class);
	
	/** Names of static RPs*/
	List<QName> staticRPs = Arrays.asList(Utils.TASK_TYPE_RP, Utils.TASK_STARTED_RP, Utils.TASK_INPUT_RP);
	/** Names of dynamic RPs*/
	List<QName> dynamicRPs = Arrays.asList(Utils.TASK_STATE_RP,Utils.TASK_OUTPUT_RP, Utils.TASK_LOG_RP, Utils.TASK_ERROR_RP);

	/**The endpoint of the remote task.*/
	private EndpointReferenceType epr;

	/** The state of the remote task. */
	private String state;
	/** The log of the remote task. */
	private String log;
	/** Any error of the remote task. */
	private String error;
	/** The time at which the remote task was started. */
	private Calendar startTime;
	/** The inputs with which the remote task was launched. */
	private Map<String,Object> inputs = new HashMap<String,Object>();
	/** The outputs produced by the remote task. */
	private Map<String,Object> outputs = new HashMap<String,Object>();
	/** The description of the remote task. */
	private TaskDescription type;
	
	/**
	 * Creates an instance from the endpoint of the remote task.
	 * @param epr the endpoint.
	 */
	public TaskProxy(EndpointReferenceType epr) throws Exception {
		this.epr=epr;
		this.synchronize();
	}
	
	/**
	 * Returns the state of the task.
	 * @return the state.
	 */
	public String getState() {return state;}
	
	/**
	 * Returns the log of the task.
	 * @return the log.
	 */
	public String getLog() {return log;}
	/**
	 * Returns the error of the task, if any.
	 * @return the error, or <code>null</code> if the task produced none.
	 */
	public String getError() {return error;}
	/**
	 * Returns the start time of the task.
	 * @return the start time.
	 */
	public Calendar getStartTime() {return startTime;}
	/**
	 * Returns the output produced by the task, if any.
	 * @return the output, or <code>null</code> if the task produced none yet.
	 */
	public Map<String,Object> getOutput() {return outputs;}
	/**
	 * Returns the input of the task, if any.
	 * @return the input, or <code>null</code> if the task had none.
	 */
	public Map<String,Object> getInput() {return inputs;}
	
	/**
	 * Returns the type of the task.
	 * @return the type.
	 */
	public TaskDescription getType() {return type;}
	
	/**
	 * Synchronises the proxy with the remote task.
	 * @throws Exception if the proxy could not be synchronised.
	 */
	public void synchronize() throws Exception {
		
		//scope is irrelevant for this operation
		TaskPortType task = new TaskServiceAddressingLocator().getTaskPortTypePort(epr);
		List<QName> RPs = dynamicRPs;
		if (startTime == null) {RPs = new ArrayList<QName>(RPs); RPs.addAll(staticRPs);}	
		GetMultipleResourcePropertiesResponse response = 
			task.getMultipleResourceProperties(new GetMultipleResourceProperties_Element(RPs.toArray(new QName[0])));
		MessageElement[] elements = response.get_any();
	
		for (MessageElement e : elements) 
			if (e.getQName().equals(Utils.TASK_STATE_RP)) state = e.getValue();
			else if (e.getQName().equals(Utils.TASK_OUTPUT_RP)) {
				Map<String,Object> outputs = Utils.intern(((AnyMap) e.getObjectValue(AnyMap.class)));
				if (outputs!=null) TaskProxy.this.outputs=outputs;
			}
			else if (e.getQName().equals(Utils.TASK_LOG_RP)) log = e.getValue();
			else if (e.getQName().equals(Utils.TASK_ERROR_RP)) error = e.getValue();
			else if (e.getQName().equals(Utils.TASK_STARTED_RP)) startTime = (Calendar) e.getObjectValue(Calendar.class);
			else if (e.getQName().equals(Utils.TASK_INPUT_RP)) { 
				Map<String,Object> inputs = Utils.intern(((AnyMap) e.getObjectValue(AnyMap.class)));
				if (inputs!=null) TaskProxy.this.inputs=inputs; 
			}
			else if (e.getQName().equals(Utils.TASK_TYPE_RP)) type= (TaskDescription) e.getObjectValue(TaskDescription.class);
	
	}
	/**{@inheritDoc}*/
	public String toString() {
		try {
			return "Type:"+getType()+"\nState:"+getState()+"\nInputs:"+getInput()+"\nLog:"+getLog()+"\nOutputs:"+getOutput()+"\nError:"+getError()+"\nStarted:"+getStartTime().getTime();
		}catch(Exception e) {throw new RuntimeException(e);}
	}

}
