/**
 * (c) 2014 FAO / UN (project: fi-security-client)
 */
package org.fao.fi.security.client.providers.token.impl;

import java.net.URL;

import javax.ws.rs.ClientErrorException;
import javax.ws.rs.NotFoundException;
import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.ClientRequestFilter;
import javax.ws.rs.client.Invocation.Builder;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import org.fao.fi.security.client.providers.token.spi.TokenProducerServiceClient;
import org.fao.fi.security.common.services.exceptions.token.TokenProcessingException;
import org.fao.fi.security.common.support.token.spi.TokenProcessor;
import org.fao.fi.security.common.utilities.LoggingClient;
import org.glassfish.jersey.client.ClientConfig;

/**
 * Place your class / interface description here.
 *
 * History:
 *
 * ------------- --------------- -----------------------
 * Date			 Author			 Comment
 * ------------- --------------- -----------------------
 * 2 May 2014   Fiorellato     Creation.
 *
 * @version 1.0
 * @since 2 May 2014
 */
abstract public class AbstractTokenProviderServiceClient extends LoggingClient implements TokenProducerServiceClient {
	private Client _client = null;
	
	private ClientConfig _config;
	private URL _endpoint;
	
	private TokenProcessor _tokenProcessor;
	
	public AbstractTokenProviderServiceClient(URL endpoint, TokenProcessor tokenProcessor) {
		this._config = new ClientConfig();
		
		this._endpoint = endpoint;
		this._tokenProcessor = tokenProcessor;
		
		this._client = ClientBuilder.newClient(this._config);
	}
	
	protected Builder buildRequest() {
		long end, start = System.currentTimeMillis();
		
		this._log.info("Building token request to {}", this._endpoint);
		
		try {
			return 
				this._client.
					target(this._endpoint.toString()).
					request().
					accept(MediaType.TEXT_PLAIN);
		} finally {
			end = System.currentTimeMillis();
			
			this._log.info("Token request to {} has been built in {} mSec.", this._endpoint, end - start);
		}
	}
	
	/* (non-Javadoc)
	 * @see org.fao.fi.security.client.providers.token.spi.TokenProducerServiceClient#requestToken()
	 */
	final public String requestToken() throws TokenProcessingException {
		long end, start = System.currentTimeMillis();
		
		this._log.info("Requesting and extracting token from {}", this._endpoint);
		
		try {
			return this.extractToken(this.request());
		} finally {
			end = System.currentTimeMillis();
			
			this._log.info("Token has been requested and extracted from {} in {} mSec.", this._endpoint, end - start);
		}
	}
	
	/* (non-Javadoc)
	 * @see org.fao.fi.security.client.providers.token.spi.TokenProducerServiceClient#request()
	 */
	@Override
	public Response request() {
		long end, start = System.currentTimeMillis();
		
		this._log.info("Requesting token from {}", this._endpoint);
		
		try {
			return this.buildRequest().get();
		} finally {
			end = System.currentTimeMillis();
			
			this._log.info("Token has been received from {} in {} mSec.", this._endpoint, end - start);
		}
	}
	
	/* (non-Javadoc)
	 * @see org.fao.fi.security.client.providers.token.spi.TokenProducerServiceClient#extractToken(javax.ws.rs.core.Response)
	 */
	@Override
	public String extractToken(Response response) throws TokenProcessingException {
		long end, start = System.currentTimeMillis();
		
		this._log.info("Extracting token from response received by {}", this._endpoint);
		
		try {
			long tempEnd, tempStart = System.currentTimeMillis();
			if(response.getStatus() == 200) {
				this._log.info("Reading response entity...");
				
				String token = response.readEntity(String.class);
				
				tempEnd = System.currentTimeMillis();
				
				this._log.info("Response entity has been read in {} mSec. ({} bytes in total)", tempEnd - tempStart, token.getBytes().length);
				
				this._log.info("Processing token before consumption via {}...", this._tokenProcessor);
				
				tempStart = System.currentTimeMillis();

				token = this._tokenProcessor.processBeforeConsumption(token);
				
				tempEnd = System.currentTimeMillis();
				
				this._log.info("Processing token before consumption via {} took {} mSec.", this._tokenProcessor, tempEnd - tempStart);
				
				return token;
			}
			
			if(response.getStatus() == 404)
				throw new NotFoundException(response);
			
			throw new ClientErrorException(response.getStatus());
		} finally {
			end = System.currentTimeMillis();
			
			this._log.info("Token has been extracted and processed from response received by {} in {} mSec.", this._endpoint, end - start);
		}
	}
	
	final public void registerFilter(ClientRequestFilter toRegister) {
		this._client.register(toRegister);
	}
}
