

package org.gcube.accounting.security.authz;

import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Vector;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.log4j.Logger;
import org.gcube.accounting.common.file.FileChangedListener;
import org.gcube.accounting.exception.NotFoundException;

public class FileAuthorizationManager implements AuthorizationManager,
		FileChangedListener {

	/**
	 * A logger for this class
	 */
	private static Logger logger = Logger
			.getLogger(FileAuthorizationManager.class);

	/**
	 * Convenience enum for file sections
	 */
	private enum Section {
		ROLE, PERMISSION
	}

	/**
	 * Special user name matching everybody
	 */
	private static final String ALL = "all";

	/**
	 * Special user name for anonymous users
	 */
	private static final String ANONYMOUS = "anonymous";

	/**
	 * The full set of permissions
	 */
	private Map<String, Collection<Permission>> permissions;

	/**
	 * The full set of roles
	 */
	private Map<String, Role> roles;

	/**
	 * The authorization definition file
	 */
	private File file;

	/**
	 * Basic constructor
	 * 
	 * @param file path
	 */
	public FileAuthorizationManager(String file) {
		this(new File(file));
	}

	/**
	 * Basic constructor
	 * 
	 * @param file
	 */
	public FileAuthorizationManager(File file) {
		this.file = file;
		// init status
		this.permissions = new HashMap<String, Collection<Permission>>();
		this.roles = new HashMap<String, Role>();
		// proceed with parsing the file
		this.parseAuthorizationFile();
	}

	/**
	 * Callback method for changes on the file
	 */
	public void fileChanged(File file) {
		logger.info("Authorization file has changed. Refreshing rules.");
		this.parseAuthorizationFile();
	}

	/**
	 * Parse a single line defining a role
	 * 
	 * @param line
	 * @return
	 * @throws Exception
	 */
	private Role parseRoleLine(String line) throws Exception {
		line = " " + line + " ";
		Pattern rolePattern = Pattern.compile(
				"^\\s*([0-9a-z_-]+)\\s*:\\s*(([0-9a-z_-]+\\s*,?\\s*)*)\\s*$",
				Pattern.CASE_INSENSITIVE);
		Matcher m = rolePattern.matcher(line);
		if (m.matches()) {
			Role r = new Role(m.group(1));
			String[] actions = m.group(2).split(",");
			for (String a : actions) {
				try {
					a = a.trim();
					String msg = String.format(
							"Adding action '%s' to role '%s'", a, r.getName());
					logger.info(msg);
					Action act = Action.getAction(a);
					r.addAction(act);
				} catch (NotFoundException e) {
					String msg = String.format(
							"Undefined action '%s' for role '%s'. Ignoring.",
							a, r.getName());
					logger.warn(msg);
				}
			}
			return r;
		} else {
			String msg = "Malformed role definition: " + line;
			logger.warn(msg);
			// TODO: use specific exception
			throw new Exception(msg);
		}
	}

	/**
	 * Parse a single line representing a permission.
	 * 
	 * @param line
	 * @return
	 * @throws Exception
	 */
	private Collection<Permission> parsePermissionLine(String line)
			throws Exception {
		line = " " + line + " ";
		// overall structure
		String[] parts = line.split(":");
		if (parts.length != 3) {
			String msg = "Malformed permission definition: " + line;
			// TODO: use specific exception
			throw new Exception(msg);
		}
		// users
		String[] users = parts[0].split(",");
		// roles
		String[] roles = parts[1].split(",");
		// target
		String target = parts[2];
		Collection<Permission> out = new Vector<Permission>();
		for (String u : users) {
			u = u.trim();
			for (String r : roles) {
				r = r.trim();
				Role role = this.roles.get(r);
				if (role != null) {
					target = target.trim();
					String msg = String.format(
							"Granting role '%s' to user '%s' over '%s'.", r, u,
							target);
					logger.info(msg);
					Permission p = new Permission(u, target, role);
					out.add(p);
				} else {
					String msg = String.format("Undefined role '%s'. Ignoring",
							r);
					logger.warn(msg);
				}
			}
		}
		return out;

	}

	/**
	 * Fully parses the authorization definition file. Populates local data
	 * structures.
	 */
	private void parseAuthorizationFile() {
		// init authorizations
		this.permissions.clear();
		this.roles.clear();
		// first look for roles
		this.parseAuthorizationFile(Section.ROLE);
		// then look for permissions
		this.parseAuthorizationFile(Section.PERMISSION);
	}

	/**
	 * Parse the given section of the authorization definition file. Populates
	 * both roles and permissions
	 * 
	 * @param section
	 */
	private void parseAuthorizationFile(Section section) {
		// pattern for role header
		Pattern rolePattern = Pattern.compile("\\s*\\[\\s*role\\s*\\]\\s*",
				Pattern.CASE_INSENSITIVE);
		// pattern for permission header
		Pattern permissionPattern = Pattern.compile(
				"\\s*\\[\\s*permission\\s*\\]\\s*", Pattern.CASE_INSENSITIVE);

		try {
			// Open the file
			FileInputStream fstream = new FileInputStream(this.file);
			DataInputStream in = new DataInputStream(fstream);
			BufferedReader br = new BufferedReader(new InputStreamReader(in));
			// current section during file scanning
			Section currentSection = null;
			// Read File Line By Line
			String line;
			while ((line = br.readLine()) != null) {
				// skip comments
				if (line.trim().startsWith("#"))
					continue;
				// skip empty lines
				if (line.trim().equals(""))
					continue;
				// look for a section header
				Matcher m1 = rolePattern.matcher(line);
				if (m1.matches()) {
					logger.debug("entering 'role' section");
					currentSection = Section.ROLE;
					continue;
				}
				Matcher m2 = permissionPattern.matcher(line);
				if (m2.matches()) {
					logger.debug("entering 'permission' section");
					currentSection = Section.PERMISSION;
					continue;
				}
				// if current mode is different from requested mode, skip lines
				if (section != currentSection) {
					continue;
				}
				// parse roles
				if (section == Section.ROLE) {
					try {
						this.addRole(this.parseRoleLine(line));
					}
					// TODO: use proper exception
					catch (Exception e) {
						logger.error(e.getMessage());
					}
				}
				// parse permissions
				if (section == Section.PERMISSION) {
					try {
						this.addPermissions(this.parsePermissionLine(line));
					}
					// TODO: use proper exception
					catch (Exception e) {
						logger.error(e.getMessage());
					}
				}
			}
			// Close the input stream
			in.close();
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		}
	}

	/**
	 * Add the given role to the set of known roles
	 * 
	 * @param r
	 * @throws Exception
	 */
	private void addRole(Role r) throws Exception {
		if (r == null)
			return;
		if (r.getName() == null)
			return;
		if (roles.containsKey(r.getName())) {
			String msg = String.format("Duplicate role '%s'", r.getName());
			logger.debug(msg);
			// TODO: use specific exception
			throw new Exception(msg);
		}
		logger.debug("Adding role " + r.getName());
		this.roles.put(r.getName(), r);
	}

	/**
	 * Add the given permissions to the set of known permissions
	 * 
	 * @param perms
	 */
	private void addPermissions(Collection<Permission> perms) {
		for (Permission p : perms)
			this.addPermission(p);
	}

	/**
	 * Retrieve the set of permissions for the given user. If not present,
	 * create and record an empty set.
	 * 
	 * @param userId
	 * @return
	 */
	private Collection<Permission> retrieveUserPermission(String userId) {
		Collection<Permission> userAuthz = permissions.get(userId);
		// create if not there
		if (userAuthz == null) {
			userAuthz = new Vector<Permission>();
			permissions.put(userId, userAuthz);
		}
		return userAuthz;
	}

	/**
	 * Add the given permission to the set of known permissions.
	 * 
	 * @param p
	 */
	private void addPermission(Permission p) {
		if (p == null)
			return;
		logger.debug(String.format("Adding permission '%s'", p.toString()));
		Collection<Permission> userAuthz = this.retrieveUserPermission(p
				.getUserId());
		userAuthz.add(p);
	}

	/**
	 * Compute the full set of permissions for the given user
	 * 
	 * @param userId
	 * @return
	 */
	private Collection<Permission> computeUserPermissions(String userId) {
		// anonymous
		if (userId == null) {
			return this.retrieveUserPermission(ANONYMOUS);
		}
		// authenticated
		else {
			Collection<Permission> out = new ArrayList<Permission>();
			// add user-specific permissions
			out.addAll(this.retrieveUserPermission(userId));
			// add common permissions
			out.addAll(this.retrieveUserPermission(ALL));
			return out;
		}
	}

	/**
	 * Check whether the given user is allowed to perform the given action
	 */
	public boolean isAllowed(String userId, Action action) {
		for (Permission p : this.computeUserPermissions(userId))
			if (p.isAllowed(action))
				return true;
		return false;
	}

	/**
	 * Check if the given user is allowed to perform the given action over the
	 * given object.
	 */
	public boolean isAllowed(String userId, Action action, String object) {
		for (Permission p : this.computeUserPermissions(userId))
			if (p.isAllowed(action, object))
				return true;
		return false;
	}

//	 public static void main(String[] args) {
//	 String fileName =
//	 "/home/ermanno/venus-c_new/temp.venusc.mab.usagetracker/conf/venusc-mab-usagetracker-authz";
//	 File file = new File(fileName);
//	 FileAuthorizationManager am = new FileAuthorizationManager(file);
//	 FileWatcher fw = new FileWatcher(file);
//	 fw.setListener(am);
//	 }

}
