package org.gcube.application.enm.service.conn;

import java.io.File;
import java.io.IOException;
import java.util.UUID;
import java.util.concurrent.ConcurrentMap;

import org.gcube.application.enm.common.xml.logs.ExperimentLogs;
import org.gcube.application.enm.common.xml.request.ExperimentRequest;
import org.gcube.application.enm.common.xml.results.ExperimentResults;
import org.gcube.application.enm.common.xml.resumes.ExperimentResumeType;
import org.gcube.application.enm.common.xml.resumes.ExperimentResumes;
import org.gcube.application.enm.common.xml.status.ExperimentStatus;
import org.gcube.application.enm.context.ServiceContext;
import org.gcube.application.enm.service.GenericJob;
import org.gcube.application.enm.service.PluginLoader;
import org.gcube.application.enm.service.util.FileUtils;
import org.gcube.application.enm.service.util.XmlHelper;
import org.gcube.common.core.utils.logging.GCUBELog;

/**
 * Persistence client.
 * 
 * @author Erik Torres <ertorser@upv.es>
 */
public class PersistenceClient {

	protected GCUBELog logger = new GCUBELog(PersistenceClient.class);	

	public static final String REQUEST_FILENAME = "request.xml";
	public static final String STATUS_FILENAME  = "status.xml";
	public static final String RESULTS_FILENAME = "results.xml";
	public static final String LOGS_FILENAME    = "logs.xml";

	public static final File PERSISTENCE_ROOT = ServiceContext.getContext()
			.getPersistenceRoot();
	public static String PERSISTENCE_ROOT_PATH;
	static {
		try {
			PERSISTENCE_ROOT_PATH = PERSISTENCE_ROOT.getCanonicalPath();
		} catch (Exception e) {
			PERSISTENCE_ROOT_PATH = null;			
		}
	}

	public PersistenceClient() {
		// Setup logging
		logger.trace("Constructor...");
	}

	public void write(final GenericJob job) throws IOException {
		write(job, new JobUpdate());
	}

	public void write(final GenericJob job, final JobUpdate updates) 
			throws IOException {
		validate(job);
		final File dir = getBasedir(job);
		final boolean isNewJobCreated = dir.mkdirs();
		if (isNewJobCreated)
			XmlHelper.typeToXmlFile(job.getRequest(), new File(dir, 
					REQUEST_FILENAME));
		if (updates.isStatusChange())
			XmlHelper.typeToXmlFile(job.getStatus(), new File(dir, 
					STATUS_FILENAME));
		if (updates.isResultsChange())
			XmlHelper.typeToXmlFile(job.getResults(), new File(dir, 
					RESULTS_FILENAME));
		if (updates.isLogsChange())
			XmlHelper.typeToXmlFile(job.getLogs(), 
					new File(dir, LOGS_FILENAME));
		logger.trace("Experiment '" + job.getUUID().toString() + "' updated");
	}

	public GenericJob read(final String credentials, final UUID experimentId)
			throws IOException {
		try {
			final File dir = getBasedir(credentials, experimentId);
			final ExperimentRequest request = XmlHelper.xmlFileToType(
					new File(dir, REQUEST_FILENAME));
			final ExperimentStatus status = XmlHelper.xmlFileToType(
					new File(dir, STATUS_FILENAME));
			final ExperimentResults results = XmlHelper.xmlFileToType(
					new File(dir, RESULTS_FILENAME));
			final ExperimentLogs logs = XmlHelper.xmlFileToType(
					new File(dir, LOGS_FILENAME));
			final String pluginName = 
					new InformationSystemClient().getJobPluginName(request);
			final GenericJob job = new PluginLoader().getNewJobInstance(
					experimentId, request, status, results, logs, pluginName);
			validate(job);
			return job;
		} catch (Exception e) {
			throw new IOException(e.getLocalizedMessage());
		}
	}

	public final void delete(final GenericJob job) throws IOException {
		FileUtils.deleteDirRecursive(getBasedir(job));
		logger.trace("The experiment '" + job.getUUID().toString() 
				+ "' was permanently removed");
	}	

	public String findCredentials(final UUID experimentId) {
		if (experimentId != null) {
			final String[] userDirs = FileUtils.listAllDirs(
					PERSISTENCE_ROOT_PATH);
			for (final String userDir : userDirs) {
				final String credentials = new File(userDir).getName();
				final String[] jobDirs = FileUtils.listAllDirs(userDir);
				for (final String jobDir : jobDirs) {
					final UUID itemId = UUID.fromString(
							new File(jobDir).getName());
					if (itemId.equals(experimentId))
						return credentials;
				}
			}
		}
		return null;
	}

	public void loadJobs(final ConcurrentMap<UUID, GenericJob> jobsMap) {
		loadJobs(jobsMap, true);
	}

	public void loadJobs(final ConcurrentMap<UUID, GenericJob> jobsMap, 
			final boolean pending) {
		logger.trace("Loading jobs from '" + PERSISTENCE_ROOT_PATH + "'");
		final String[] userDirs = FileUtils.listAllDirs(PERSISTENCE_ROOT_PATH);
		for (final String userDir : userDirs) {
			final String credentials = new File(userDir).getName();
			final String[] jobDirs = FileUtils.listAllDirs(userDir);
			for (final String jobDir : jobDirs) {
				final UUID experimentId = UUID.fromString(
						new File(jobDir).getName());
				try {
					boolean load = true;
					final GenericJob job = read(credentials, experimentId);
					if (pending) {
						switch (job.getStatus().getStatus()) {
						case PENDING:
						case EXECUTING:
							load = true;
							break;
						case CANCELLED:
						case FAILED:
						case FINISHED:
						default:
							load = false;
							break;
						}
					}
					if (load) {
						jobsMap.put(experimentId, job);
						logger.trace("Job loaded: '" + experimentId + "'");
					}
				} catch (Exception e) {
					logger.error("Failed to load experiment '" 
							+ experimentId.toString() + "': " 
							+ e.getLocalizedMessage());
				}
			}
		}
	}

	public ExperimentResumes getUserExperimentResumes(final String credentials) 
			throws IOException {
		final File userDir = getUserdir(credentials);
		final String[] jobDirs = FileUtils.listAllDirs(userDir
				.getCanonicalPath());
		final ExperimentResumes experimentResumes = new org.gcube.application
				.enm.common.xml.resumes.ObjectFactory().createExperimentResumes();		
		experimentResumes.setExperimentResumes(new org.gcube.application
				.enm.common.xml.resumes.ObjectFactory().createExperimentResumesType());
		experimentResumes.setCredentials(credentials);
		for (final String jobDir : jobDirs) {
			final UUID experimentId = UUID.fromString(
					new File(jobDir).getName());
			final GenericJob job = read(credentials, experimentId);
			experimentResumes.getExperimentResumes().getExperimentResume().add(
					resume(job));
		}
		return experimentResumes;
	}

	private ExperimentResumeType resume(final GenericJob job) {
		final ExperimentResumeType resume = new org.gcube.application.enm
				.common.xml.resumes.ObjectFactory().createExperimentResumeType();
		resume.setUuid(job.getUUID().toString());
		resume.setName(job.getRequest().getName());
		resume.setStatus(org.gcube.application.enm.common.xml.resumes
				.StatusType.fromValue(job.getStatus().getStatus().value()));
		resume.setCompletenessPercentage(job.getStatus().getCompletenessPercentage());
		resume.setStartDate(job.getStatus().getStartDate());
		resume.setEndDate(job.getStatus().getEndDate());
		return resume;
	}	

	private void validate(final GenericJob job) throws IOException {
		if (job == null)
			throw new IOException("Job cannot be null");
		if (job.getUUID() == null)
			throw new IOException("Experiment Id cannot be null");
		if (job.getRequest() == null)
			throw new IOException("Request cannot be null");
		if (job.getRequest().getCredentials() == null 
				|| job.getRequest().getCredentials().length() < 3)
			throw new IOException("Invalid user credentials");
		if (job.getRequest().getName() == null 
				|| job.getRequest().getName().length() < 3)
			throw new IOException("Invalid job name");
		if (job.getStatus() == null)
			throw new IOException("Status cannot be null");
		if (job.getResults() == null)
			throw new IOException("Results cannot be null");
		if (job.getLogs() == null)
			throw new IOException("Logs cannot be null");
	}

	private File getBasedir(final GenericJob job) throws IOException {
		return getBasedir(job.getRequest().getCredentials(), job.getUUID());
	}

	private File getBasedir(final String credentials, final UUID experimentId)
			throws IOException {
		return new File(getUserdir(credentials), experimentId.toString());
	}

	private File getUserdir(final String credentials) throws IOException {
		return new File(PERSISTENCE_ROOT.getCanonicalPath(), credentials);
	}	

}
