package org.gcube.application.geoportalcommon;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Arrays;

import org.gcube.application.geoportalcommon.shared.GNADataEntryConfigProfile;
import org.gcube.application.geoportalcommon.shared.GNADataViewerConfigProfile;
import org.gcube.application.geoportalcommon.shared.GeoNaItemRef;
import org.gcube.application.geoportalcommon.shared.PublicLink;
import org.gcube.portlets.user.urlshortener.UrlShortener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The Class GeoportalCommon.
 *
 * @author Francesco Mangiacrapa at ISTI-CNR francesco.mangiacrapa@isti.cnr.it
 * 
 *         Aug 5, 2021
 */
public class GeoportalCommon {

	/** The Constant LOG. */
	private static final Logger LOG = LoggerFactory.getLogger(GeoportalCommon.class);

	private GNADataViewerConfigProfile geonaDataProfile;

	/**
	 * Instantiates a new geoportal common.
	 */
	public GeoportalCommon() {
	}

	/**
	 * Instantiates a new geoportal common.
	 *
	 * @param geonaDataProfile the geona data profile
	 */
	public GeoportalCommon(GNADataViewerConfigProfile geonaDataProfile) {
		this.geonaDataProfile = geonaDataProfile;
	}

	/**
	 * Gets the public links for.
	 *
	 * @param item           the item
	 * @param createShortURL creates and returns the short URL also.
	 * @return the public links for
	 * @throws Exception the exception
	 */
	public GeoNaItemRef getPublicLinksFor(GeoNaItemRef item, boolean createShortURL) throws Exception {
		LOG.info("getPublicLinksFor called for: " + item);

		try {

			if (item == null)
				throw new Exception("Bad request, the item is null");

			if (item.getItemId() == null)
				throw new Exception("Bad request, the item id is null");

			if (item.getItemType() == null)
				throw new Exception("Bad request, the item type is null");

			if (this.geonaDataProfile == null)
				this.geonaDataProfile = readGNADataViewConfig(GeoportalCommonConstants.GEOPORTAL_DATA_VIEWER_APP);

			// Restricted Link
			String link = String.format("%s?%s=%s&%s=%s", geonaDataProfile.getRestrictedPortletURL(),
					GeoportalCommonConstants.GET_GEONA_ITEM_ID, item.getItemId(),
					GeoportalCommonConstants.GET_GEONA_ITEM_TYPE, item.getItemType());

			String shortUrl = link;
			try {
				if (createShortURL)
					shortUrl = getShortUrl(link);
			} catch (Exception e) {
				LOG.warn("Error on shortening the URL: ", e);
			}
			item.setRestrictedLink(new PublicLink(link, shortUrl));

			// Open Link
			link = String.format("%s?%s=%s&%s=%s", geonaDataProfile.getOpenPortletURL(),
					GeoportalCommonConstants.GET_GEONA_ITEM_ID, item.getItemId(),
					GeoportalCommonConstants.GET_GEONA_ITEM_TYPE, item.getItemType());

			shortUrl = link;
			try {
				shortUrl = getShortUrl(link);
			} catch (Exception e) {
				LOG.warn("Error on shortening the URL: ", e);
			}
			item.setOpenLink(new PublicLink(link, shortUrl));

			LOG.info("returning: " + item);
			return item;

		} catch (Exception e) {
			LOG.error("Error on getPublicLinksFor for: " + item, e);
			throw new Exception("Share link not available for this item. Try later or contact the support. Error: "
					+ e.getMessage());
		}
	}

	/**
	 * Gets the short url.
	 *
	 * @param longUrl the long url
	 * @return the short url
	 * @throws Exception the exception
	 */
	public String getShortUrl(String longUrl) throws Exception {
		LOG.info("getShortUrl called for " + longUrl);

		if (longUrl == null)
			return longUrl;

		try {

			UrlShortener shortener = new UrlShortener();

			if (shortener != null && shortener.isAvailable()) {
				String toShort = longUrl;
				String[] splitted = toShort.split("\\?");
				LOG.debug("Splitted long URL is: " + Arrays.asList(splitted));
				String link = toShort;
				if (splitted.length > 1) {
					LOG.debug("Query string detected, encoding it...");
					String encodedQuery = splitted[1];
					try {
						encodedQuery = URLEncoder.encode(splitted[1], "UTF-8");
					} catch (UnsupportedEncodingException e) {
						e.printStackTrace();

					}
					LOG.debug("encodedQuery is: " + encodedQuery);
					link = String.format("%s?%s", splitted[0], encodedQuery);
				}
				return shortener.shorten(link);
			}
			return longUrl;

		} catch (Exception e) {
			LOG.error("Error on shortening the longURL " + longUrl, e);
			return null;
		}
	}

	/**
	 * Gets the geona data profile.
	 *
	 * @return the geona data profile
	 */
	public GNADataViewerConfigProfile getGeonaDataProfile() {
		return geonaDataProfile;
	}

	/**
	 * Read GNA data view config.
	 *
	 * @param appID the app ID. If null uses the default
	 *              {@link GeoportalCommonConstants#GEOPORTAL_DATA_VIEWER_APP}
	 * @return the GNA data viewer config profile
	 * @throws Exception the exception
	 */
	public GNADataViewerConfigProfile readGNADataViewConfig(String appID) throws Exception {
		LOG.info("getGeoNaDataViewProfile called for AppID: " + appID);

		if (appID == null || appID.isEmpty()) {
			LOG.info("AppID is null, so using DEFAULT: " + appID);
			appID = GeoportalCommonConstants.GEOPORTAL_DATA_VIEWER_APP;
		}

		GNADataViewerConfigProfileReader gdvp = new GNADataViewerConfigProfileReader(appID);
		GNADataViewerConfigProfile profile = gdvp.readProfileFromInfrastrucure();
		LOG.info("Returning profile: " + profile);
		return profile;
	}

	/**
	 * Gets the GNA data config read from ApplicationProfile stored into IS.
	 *
	 * @return the GNA data config
	 * @throws Exception the exception
	 */
	public GNADataEntryConfigProfile readGNADataEntryConfig() throws Exception {

		GNADataEntryConfigProfileReader gnaConfigReader = new GNADataEntryConfigProfileReader();
		GNADataEntryConfigProfile config = gnaConfigReader.readProfileFromInfrastrucure();
		return config;
	}

}
