package org.gcube.common.gxrest.request;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Map;
import java.util.Objects;

import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Entity;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.Response;

import org.gcube.common.authorization.library.provider.SecurityTokenProvider;
import org.gcube.common.gxrest.response.inbound.GXInboundResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * A GX request based on JAX-RS. 
 * It requires a runtime implementation of JAX-RS on the classpath (e.g. Jersey) to work.
 * 
 * @author Manuele Simi (ISTI CNR)
 *
 */
public class GXWebTargetAdapterRequest implements GXHTTP {

	private final WebTarget adaptee;
	private static final Logger logger = LoggerFactory.getLogger(GXWebTargetAdapterRequest.class);
	private Entity<?> entity;
	/**
	 * Creates a new request.
	 * @param address the address of the web app to call
	 * @return the request
	 */
	public static GXWebTargetAdapterRequest newRequest(String address) {
		return new GXWebTargetAdapterRequest(address);
	}
	/**
	 * @param address
	 */
	public GXWebTargetAdapterRequest(String address) {
		Client client = ClientBuilder.newClient();
		this.adaptee = client.target(address);
		this.property(org.gcube.common.authorization.client.Constants.TOKEN_HEADER_ENTRY,
				SecurityTokenProvider.instance.get());
	}
	
	/**
	 * Overrides the default security token.
	 * @param token
	 * @return the request
	 */
	public GXWebTargetAdapterRequest setSecurityToken(String token) {
		this.property(org.gcube.common.authorization.client.Constants.TOKEN_HEADER_ENTRY,
				token);
		return this;
	}
	
	/**
	 * Sets a new property in the request.
	 * @param name
	 * @param value
	 * @return the request
	 */
	public GXWebTargetAdapterRequest property(String name, String value) {
		this.adaptee.property(name,value);
		return this;
	}
	
	/**
	 * Register an instance of a custom JAX-RS component (such as an extension provider or
     * a {@link javax.ws.rs.core.Feature feature} meta-provider) to be instantiated
     * and used in the scope of this request.
	 * @param component
	 */
	public void register(Object component) {
		this.adaptee.register(component);
	}

	/**
	 * Adds s positional path parameter to the request.
	 * @param path
	 * @return the request
	 * @throws UnsupportedEncodingException 
	 */
	public GXWebTargetAdapterRequest path(String path) throws UnsupportedEncodingException {
		this.adaptee.path(path);
		return this;
	}
	
	/**
	 * Sets the query parameters for the request.
	 * @param parameters the parameters that go in the URL after the address and the path params.
	 * @return the request
	 * @throws UnsupportedEncodingException
	 */
	public GXWebTargetAdapterRequest queryParams(Map<String, String> parameters) throws UnsupportedEncodingException {
		if (Objects.nonNull(parameters) && ! parameters.isEmpty()) {
			for (String key : parameters.keySet()) {;
				this.adaptee.queryParam(URLEncoder.encode(key, GXConnection.UTF8), 
						(URLEncoder.encode(parameters.get(key), GXConnection.UTF8)));
			}
		}
		return this;
	}

	
	/**
	 * Sets the body of the request.
	 * @param body
	 * @return the request
	 */
	public GXWebTargetAdapterRequest withEntity(Entity<?> entity) {
		this.entity = entity;;
		return this;
	}
	
	/* (non-Javadoc)
	 * @see org.gcube.common.gxrest.request.GXHTTP#put()
	 */
	@Override
	public GXInboundResponse put() throws Exception {
		Response response =  this.adaptee.request().put(this.entity);
		return new GXInboundResponse(response);
	}

	/* (non-Javadoc)
	 * @see org.gcube.common.gxrest.request.GXHTTP#delete()
	 */
	@Override
	public GXInboundResponse delete() throws Exception {
		Response response =  this.adaptee.request().delete();
		return new GXInboundResponse(response);
	}

	/* (non-Javadoc)
	 * @see org.gcube.common.gxrest.request.GXHTTP#post()
	 */
	@Override
	public GXInboundResponse post() throws Exception {
		Response response =  this.adaptee.request().post(this.entity, Response.class);
		return new GXInboundResponse(response);
	}

	/* (non-Javadoc)
	 * @see org.gcube.common.gxrest.request.GXHTTP#head()
	 */
	@Override
	public GXInboundResponse head() throws Exception {
		Response response =  this.adaptee.request().head();
		return new GXInboundResponse(response);
	}

	/* (non-Javadoc)
	 * @see org.gcube.common.gxrest.request.GXHTTP#get()
	 */
	@Override
	public GXInboundResponse get() throws Exception {
		Response response =  this.adaptee.request().get(Response.class);
		return new GXInboundResponse(response);
	}

}
