package org.gcube.common.homelibrary.jcr.repository;

import static org.gcube.resources.discovery.icclient.ICFactory.clientFor;
import static org.gcube.resources.discovery.icclient.ICFactory.queryFor;
import java.security.Principal;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.SimpleCredentials;
import javax.jcr.Value;
import javax.jcr.security.AccessControlList;
import javax.jcr.security.AccessControlManager;
import javax.jcr.security.AccessControlPolicy;
import javax.jcr.security.AccessControlPolicyIterator;
import javax.jcr.security.Privilege;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.jackrabbit.rmi.repository.URLRemoteRepository;
import org.gcube.common.encryption.StringEncrypter;
import org.gcube.common.homelibrary.home.User;
import org.gcube.common.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.common.homelibrary.jcr.workspace.util.Utils;
import org.gcube.common.resources.gcore.ServiceEndpoint;
import org.gcube.common.resources.gcore.ServiceEndpoint.AccessPoint;
import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.resources.discovery.client.api.DiscoveryClient;
import org.gcube.resources.discovery.client.queries.api.SimpleQuery;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;



public class JCRRepository {
	
	public static final String HL_NAMESPACE					= "hl:";
	public static final String JCR_NAMESPACE				= "jcr:";
	
	public static final String PATH_SEPARATOR 				= "/";
	public static final String ROOT_WEBDAV					= "/repository/default/";
	
	public static final String HOME_FOLDER 					= "Home";
	public static final String SHARED_FOLDER				= "Share";
	
	
	private static final String DOWNLOADS					= "Downloads";
	private static final String SMART_FOLDER 				= "Folders";
	private static final String GCUBE_FOLDER 				= "GCube";
	private static final String IN_BOX_FOLDER 				= "InBox";
	private static final String OUT_BOX_FOLDER				= "OutBox";
	private static final String BOOKMARK_FOLDER				= "Bookmarks";
	private static final String HIDDEN_FOLDER				= "HiddenFolder";
	
	
	
	private static final String NT_FOLDER 					= "nt:folder";
	private static final String NT_HOME						= "nthl:home";
	private static final String NT_ROOT_ITEM_SENT  			= "nthl:rootItemSentRequest";
	private static final String NT_ROOT_FOLDER_BULK_CREATOR = "nthl:rootFolderBulkCreator";
	
	private static final String SCOPES						= "hl:scopes";
	
	private static final String nameResource 				= "HomeLibraryRepository";
	
	
	
	private static Repository repository;
	
	private static String gcubeRootId;
	private static String sharedRootId;
	
	private String userHomeId;
	private String smartFoldersId;
	private String downloadsId;
	private String inBoxId;
	private String outBoxId;
	private String bookmarkFolderId;
	private String hiddenFolderId;
	
	private static String user;
	private static String pass;
	private static String url;
	private static String webdavUrl;
	
		
	private static Logger logger = LoggerFactory.getLogger(JCRRepository.class);
	
	/*private static void setLocalRespositoryParameters() {

		url = "http://localhost:8080/jackrabbit-webapp-patched-2.4.3";
		user = "workspacerep.imarine";
		pass = "gcube2010*onan";
		webdavUrl = "https://portal.i-marine.d4science.org/Home";
		url = "http://node11.d.d4science.research-infrastructures.eu:8080/jackrabbit-webapp-patched-2.4.3";

	}*/
	
	private static synchronized void initializeRepository() throws InternalErrorException {
		
		if(repository != null)
			return;

		String callerScope = ScopeProvider.instance.get();
		
		try {
			
			if (callerScope==null) throw new IllegalArgumentException("scope is null");
			
			String rootScope = Utils.getRootScope(callerScope);
			
			logger.debug("scope for repository creation is "+rootScope+" caller scope is "+callerScope);
					
			ScopeProvider.instance.set(rootScope);
			
			SimpleQuery query = queryFor(ServiceEndpoint.class);

			query.addCondition("$resource/Profile/Category/text() eq 'Database' and $resource/Profile/Name eq '"+ nameResource + "' ");

			DiscoveryClient<ServiceEndpoint> client = clientFor(ServiceEndpoint.class);

			List<ServiceEndpoint> resources = client.submit(query);

			if(resources.size() != 0) {	   
				try {
					ServiceEndpoint resource = resources.get(0);
					
					for (AccessPoint ap:resource.profile().accessPoints()) {
						
						if (ap.name().equals("JCR")) {
							url = ap.address();
							user = ap.username();
							pass = StringEncrypter.getEncrypter().decrypt(ap.password());
						}
						else if (ap.name().equals("WebDav")) {								
							webdavUrl = ap.address();				
						}
					}
				} catch (Exception e) {
					logger.error("error decrypting resource",e);
				}
			}
	
	
//			user = "workspacerep.imarine";
//			pass = "gcube2010*onan";
//			url = "http://node76.p.d4science.research-infrastructures.eu:8080/jackrabbit-webapp-patched-2.4.3";
//			webdavUrl = "https://portal.i-marine.d4science.org/Home";
			
			repository = new URLRemoteRepository(url + "/rmi");
			
			logger.debug("user is "+user+" password is null?"+(pass==null)+" and repository is null?"+(repository==null));
			
		} catch (Exception e) {
			throw new InternalErrorException(e);
		}finally{
			if (callerScope!=null)
				ScopeProvider.instance.set(callerScope);
		}
		
	}
	
	public static List<String> getHomeNames() throws RepositoryException{
		List<String> homes = new ArrayList<String>();
		Session session = null;
		try{
			try {
				session = getSession();
			} catch (InternalErrorException e) {
				logger.error("error gettin session",e);
				throw new RepositoryException(e);
			}
			Node home = (Node)session.getItem(PATH_SEPARATOR+HOME_FOLDER);
			NodeIterator it = home.getNodes();
			while (it.hasNext()){
				Node node = it.nextNode();
				if (node.getPrimaryNodeType().getName().equals(NT_HOME))
					homes.add(node.getName());
			}
		}finally{
			if (session != null)
				session.logout();
		}
		return homes;
	}
	
	private static void addUserToJCRUserManager(String userId, String userHome) {
		
		 GetMethod getMethod = null;
		 try {
             
			 HttpClient httpClient = new HttpClient();            
//			 System.out.println(url);
             getMethod =  new GetMethod(url + "/PortalUserManager?userId=" + userId + "&userHome=" +userHome);
             httpClient.executeMethod(getMethod);
             
             logger.debug("User set with status code " + getMethod.getResponseBodyAsString());
             
          } catch (Exception e) {
        	  logger.error("User set with error ", e);
          } finally {
        	  if(getMethod != null)
        		  getMethod.releaseConnection();
          }
   
	}
	
	public static Session getSession() throws InternalErrorException  {
			
		initializeRepository();
		synchronized (repository) {
			try {
				Session session = repository.login( 
						new SimpleCredentials(user, pass.toCharArray()));
				return session;
			} catch (Exception e) {
				throw new InternalErrorException(e);
			}
		}
	}
	
	public synchronized static void Initialize() throws InternalErrorException {
		
		logger.debug("Initialize repository");
		
		Session session = getSession();
		try {
			Node root = session.getRootNode();
			if (!exist(root, HOME_FOLDER)){
				root.addNode(HOME_FOLDER, NT_FOLDER);
			}

			if (!exist(root, GCUBE_FOLDER)){
				root.addNode(GCUBE_FOLDER, NT_FOLDER);
			}
			
			if (!exist(root, SHARED_FOLDER)) {
				root.addNode(SHARED_FOLDER, NT_FOLDER);
			}
				
			
			sharedRootId = root.getNode(SHARED_FOLDER).getIdentifier();
			gcubeRootId = root.getNode(GCUBE_FOLDER).getIdentifier();
			
			session.save();
			
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}
	
	public JCRRepository(final User user) throws InternalErrorException {
		
		Session session = getSession();
		try {
			Node root = session.getRootNode();
			Node home = root.getNode(HOME_FOLDER);
			
			
			Node userHome;
			if (!exist(home, user.getPortalLogin())) {
				userHome = home.addNode(user.getPortalLogin(), NT_HOME);
			} else {
				userHome = home.getNode(user.getPortalLogin());
			}
					
						
			session.save();
			userHomeId = userHome.getIdentifier();
			
			addUserToJCRUserManager(user.getPortalLogin(), userHome.getPath());
			userHome.getPath();
			
			
			Node smartFolders;
			if (!exist(userHome, SMART_FOLDER)) {
				smartFolders = userHome.addNode(SMART_FOLDER, NT_FOLDER);
			} else {
				smartFolders = userHome.getNode(SMART_FOLDER);
			}

			Node inBoxNode;
			if (!exist(userHome, IN_BOX_FOLDER)) {
				inBoxNode = userHome.addNode(IN_BOX_FOLDER, NT_ROOT_ITEM_SENT);
			} else {
				inBoxNode = userHome.getNode(IN_BOX_FOLDER);
			}
			
			Node outBoxNode;
			if (!exist(userHome, OUT_BOX_FOLDER)) {
				outBoxNode = userHome.addNode(OUT_BOX_FOLDER, NT_ROOT_ITEM_SENT);
			} else {
				outBoxNode = userHome.getNode(OUT_BOX_FOLDER);
			}
			
			//create a hidden folder if it does not exist
			Node bookmarkFolder;
			if (!exist(userHome,BOOKMARK_FOLDER)) {
				bookmarkFolder = userHome.addNode(BOOKMARK_FOLDER, NT_FOLDER);
			} else {
				bookmarkFolder = userHome.getNode(BOOKMARK_FOLDER);
			}
			
			//create a hidden folder if it does not exist
			Node hiddenFolder;
			if (!exist(userHome,HIDDEN_FOLDER)) {
				hiddenFolder = userHome.addNode(HIDDEN_FOLDER, NT_FOLDER);
			} else {
				hiddenFolder = userHome.getNode(HIDDEN_FOLDER);
			}
			
			Node downloads = null;
			if (!exist(userHome, DOWNLOADS)) {
				downloads = userHome.addNode(DOWNLOADS, NT_ROOT_FOLDER_BULK_CREATOR);
			} else {
				downloads = userHome.getNode(DOWNLOADS);
			}
				
			
			session.save();
			
			smartFoldersId = smartFolders.getIdentifier();
			inBoxId = inBoxNode.getIdentifier();
			outBoxId = outBoxNode.getIdentifier();
			downloadsId = downloads.getIdentifier(); 
			bookmarkFolderId = bookmarkFolder.getIdentifier();
			hiddenFolderId = hiddenFolder.getIdentifier();

			
		} catch (RepositoryException e) {
			logger.error("Repository not instantiated", e);
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	public static void removeUser(User user) throws Exception {
		
		Session session = getSession();
		Node node = session.getRootNode().getNode(HOME_FOLDER).getNode(user.getPortalLogin());
		node.remove();
		session.save();
	}
	
	
	public static void setACL(final String portalLogin , String userHome)
			throws InternalErrorException {
		
		
		Session session = null;
		try {
			
			Repository RMIrepository = new URLRemoteRepository(url + "/rmi");
			session = RMIrepository.login( 
					new SimpleCredentials(user, pass.toCharArray()));
			
			AccessControlManager accessControlManager = session.getAccessControlManager();
			AccessControlPolicyIterator acpIterator = accessControlManager.getApplicablePolicies(userHome);
			if (acpIterator.hasNext()) {
				logger.debug(" ------------ ACL Present ");
				AccessControlPolicy policy = acpIterator.nextAccessControlPolicy();
				AccessControlList list = (AccessControlList)policy;
				list.addAccessControlEntry(new Principal() {

					@Override
					public String getName() {
						return portalLogin;
					}
				}, new Privilege[]{accessControlManager.privilegeFromName(Privilege.JCR_READ)});

				accessControlManager.setPolicy(userHome, list);
			}
			
			session.save();
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} finally {
			if(session != null)
				session.logout();
		}
	}
	
	
	public void setScope(Session session, String scope) throws RepositoryException {
		
		Node userHome = getUserHome(session);
		
		Property scopes;
		try {
			scopes = userHome.getProperty(SCOPES);
		} catch (PathNotFoundException e) {
			String[] values = {scope};
			userHome.setProperty(SCOPES, values);
			return;
		}
		
		for (Value value : scopes.getValues()) {
			if(value.getString().equals(scope))
				return;
		}

		


		String[] newScopes = new String[scopes.getValues().length + 1];
		newScopes[0] = scope;
		for (int i = 1; i < newScopes.length; i++) {
			newScopes[i] = scopes.getValues()[i - 1].getString();
		}
		userHome.setProperty(SCOPES, newScopes);
	}
	
	
	private static boolean exist(Node parent, String childName) throws RepositoryException {
		
		try {
			parent.getNode(childName);
		} catch (PathNotFoundException e) {
			return false;
		} 
		return true;
	}
	
	/*
	public boolean existChild(Node father, String child){
		
		try {
			father.getNode(child);
		} catch (RepositoryException e) {
			return false;
		}
		return true;
	}*/
	
	public List<String> listScopes() throws RepositoryException, InternalErrorException {
		
		List<String> list = new LinkedList<String>();
		
		Session session = getSession();
		try {
			Node userHome = session.getNodeByIdentifier(userHomeId);
			Property scopes = userHome.getProperty(SCOPES);

			for (Value value  : scopes.getValues()) {
				list.add(value.getString());
			}
			return list;

		} catch (PathNotFoundException e) {
			return new LinkedList<String>();
		} finally {
			if (session != null)
				session.logout();
		}
	}
	
	public static Node getGCubeRoot(Session session) throws RepositoryException {
		return session.getNodeByIdentifier(gcubeRootId);
	}
	
	public static Node getSharedRoot(Session session) throws RepositoryException {
		return session.getNodeByIdentifier(sharedRootId);
	}
	
	/*
	public Node getApplicationsRoot(Session session) throws RepositoryException {
		return session.getNodeByIdentifier(applicationsRootId);
	}*/

	public Node getUserHome(Session session) throws RepositoryException {		
		return  session.getNodeByIdentifier(userHomeId); 
	}
	
	public Node getRootSmartFolders(Session session) throws RepositoryException {
		return  session.getNodeByIdentifier(smartFoldersId);
	}
	
	public Node getOwnInBoxFolder(Session session) throws RepositoryException {
		return  session.getNodeByIdentifier(inBoxId);
	}
	
	public Node getRootFolderBulkCreators(Session session) throws RepositoryException {
		return session.getNodeByIdentifier(downloadsId);
	}
	
	public Node getOutBoxFolder(Session session) throws RepositoryException {
		return session.getNodeByIdentifier(outBoxId);
	}
	
	public Node getBookmarksFolder(Session session) throws RepositoryException {
		return session.getNodeByIdentifier(bookmarkFolderId);
	}
	
	public Node getHiddenFolder(Session session) throws RepositoryException {
		return session.getNodeByIdentifier(hiddenFolderId);
	}
	
	public Node getInBoxFolder(Session session, String user) throws RepositoryException,
	InternalErrorException  {
		return session.getRootNode().getNode(HOME_FOLDER + PATH_SEPARATOR 
				+ user + PATH_SEPARATOR + IN_BOX_FOLDER);
	}
	
	public String getUserHomeUrl(String portalLogin) {
		return url + ROOT_WEBDAV + HOME_FOLDER + PATH_SEPARATOR + portalLogin;     
	}
	
	public String getWebDavUrl(String portalLogin) {
		return webdavUrl + PATH_SEPARATOR + portalLogin;      
	}
	
}
