package org.gcube.common.homelibrary.jcr.workspace.folder.items.gcube;

import java.io.ByteArrayInputStream;
import javax.jcr.Node;
import javax.jcr.RepositoryException;
import org.apache.commons.lang.Validate;
import org.gcube.common.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.common.homelibrary.home.workspace.folder.FolderItemType;
import org.gcube.common.homelibrary.home.workspace.folder.items.gcube.Metadata;
import org.gcube.common.homelibrary.jcr.workspace.JCRWorkspace;
import org.gcube.common.homelibrary.jcr.workspace.folder.items.JCRFile;
import org.gcube.common.homelibrary.jcr.workspace.folder.items.JCRWorkspaceFolderItem;
import org.gcube.common.homelibrary.util.Util;
import org.gcube.contentmanagement.blobstorage.transport.backend.RemoteBackendException;

public class JCRMetadata extends JCRWorkspaceFolderItem implements Metadata {

	private static final String SCHEMA 			= "hl:schema";
	private static final String LANGUAGE 		= "hl:language";
	private static final String COLLECTION_NAME = "hl:collectionName";
	private static final String OID 			= "hl:oid";
	
	private static final String NT_CONTENT      = "nthl:metadataItemContent";
	
	private final String schema;
	private final String language;
	private final String collectionName;
	
	private final String oid;
	
	private final JCRFile file;
	
	public JCRMetadata(JCRWorkspace workspace, Node node) throws RepositoryException, InternalErrorException {
		super(workspace, node);
		
		Node contentNode = node.getNode(CONTENT);
		this.schema = contentNode.getProperty(SCHEMA).getString();
		this.language = contentNode.getProperty(LANGUAGE).getString();
		this.collectionName = contentNode.getProperty(COLLECTION_NAME).getString();
		this.oid = contentNode.getProperty(OID).getString();
		this.file = new JCRFile(workspace, contentNode);
	}
	
	
	public JCRMetadata(JCRWorkspace workspace, Node node, String name,
			String description, String objectId, String schema, String language, 
			String collectionName, String data) throws RepositoryException, RemoteBackendException  {
		super(workspace, node, name, description);
		
		Validate.notNull(objectId, "Oid must be not null");
		Validate.notNull(schema, "Schema must be not null");
		Validate.notNull(language, "Language must be not null");
		Validate.notNull(collectionName, "Collection name must be not null");
		Validate.notNull(data, "Data must be not null");
		
		this.schema = schema;
		this.language = language;
		this.collectionName = collectionName;
		this.oid = objectId;
		
		
		node.setProperty(FOLDER_ITEM_TYPE,FolderItemType.DOCUMENT.toString());
		Node contentNode = node.addNode(CONTENT, NT_CONTENT);
		contentNode.setProperty(SCHEMA, schema);
		contentNode.setProperty(LANGUAGE, language);
		contentNode.setProperty(COLLECTION_NAME, collectionName);
		contentNode.setProperty(OID,objectId);
		this.file = new JCRFile(workspace, contentNode, new ByteArrayInputStream(data.getBytes()));		
	}

	
	@Override
	public String getSchema() {
		return schema;
	}

	@Override
	public String getLanguage() {
		return language;
	}

	@Override
	public String getData() {
		try {
			return Util.readStreamAsString(file.getData());
		} catch (Exception e) {
			return null;
		}
		
	}

	@Override
	public String getCollectionName() {
		return collectionName;
	}

	@Override
	public FolderItemType getFolderItemType() {
		return FolderItemType.METADATA;
	}

	@Override
	public String getURI() {
		return oid;
	}


	@Override
	public long getLength() {
		try {
			return file.getLength();
		} catch (Exception e) {
			return 0;
		}
	}


	@Override
	public void saveContent(Node node) throws RepositoryException {
		file.save(node);
		
	}

	@Override
	public String getMimeType() throws InternalErrorException {
		return null;
	}

}
