package org.gcube.common.homelibrary.jcr.workspace.folder.items.gcube.link;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.RepositoryException;
import org.apache.commons.lang.Validate;
import org.gcube.common.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.common.homelibrary.home.workspace.folder.FolderItemType;
import org.gcube.common.homelibrary.home.workspace.folder.items.gcube.DocumentMetadata;
import org.gcube.common.homelibrary.home.workspace.folder.items.gcube.link.DocumentLink;
import org.gcube.common.homelibrary.jcr.workspace.JCRWorkspace;
import org.gcube.common.homelibrary.jcr.workspace.folder.items.JCRWorkspaceFolderItem;
import org.gcube.common.homelibrary.jcr.workspace.folder.items.gcube.JCRDocumentMetadata;

public class JCRDocumentLink extends JCRWorkspaceFolderItem implements DocumentLink {

	private static final String METADATA   		= "hl:metadata"; 
	private static final String ANNOTATIONS 	= "hl:annotations";
	private static final String COLLECTION_NAME = "hl:collectionName";
	private static final String OID				= "hl:oid";
	
	private static final String NT_CONTENT 		= "nthl:documentLinkItemContent";
	
	private final String oid;
	private final Map<String, DocumentMetadata> metadata;
	private final Map<String, String> annotations;
	private final String collectionName;
	
	public JCRDocumentLink(JCRWorkspace workspace, Node node) throws RepositoryException, InternalErrorException {
		super(workspace, node);
		
		Node contentNode = node.getNode(CONTENT);
		this.collectionName = contentNode.getProperty(COLLECTION_NAME).getString();
		this.oid = contentNode.getProperty(OID).getString();
		this.metadata = deserializeMetadata(contentNode);
		this.annotations = deserializeAnnotations(contentNode);
	}
	
	public JCRDocumentLink(JCRWorkspace workspace, Node node, String name,
			String description, String oid, String mimeType, Map<String, String> metadata,
			Map<String, String> annotations, String collectionName) throws RepositoryException {
		super(workspace, node, name, description);
		
		Validate.notNull(oid, "Oid must be not null");
		Validate.notNull(mimeType, "MimeType must be not null");
		Validate.notNull(metadata, "Metadata must be not null");
		Validate.notNull(annotations, "Annotations must be not null");
		Validate.notNull(collectionName, "CollectionName must be not null");
		
		
		
		
		node.setProperty(FOLDER_ITEM_TYPE, FolderItemType.DOCUMENT_LINK.toString());
		
		Node contentNode;
		contentNode = node.addNode(CONTENT,NT_CONTENT);
		this.oid = oid;
		this.collectionName = collectionName;
		this.metadata = serializeMetadata(contentNode, metadata);
		this.annotations = serializeAnnotations(contentNode, annotations);

		contentNode.setProperty(OID, oid);
		contentNode.setProperty(COLLECTION_NAME, collectionName);
	}
	
	private Map<String,DocumentMetadata> serializeMetadata( Node contentNode, Map<String,String> metadata) throws RepositoryException {
		
		Node metadataNode = contentNode.getNode(METADATA);
		HashMap<String,DocumentMetadata> map = new HashMap<String,DocumentMetadata>();
		for (Entry<String, String> entry : metadata.entrySet()) {
			metadataNode.setProperty(entry.getKey(), entry.getValue());
			map.put(entry.getKey(), new JCRDocumentMetadata(entry.getKey(),
					entry.getValue()));
		}
		return map;
	}
	
	private Map<String,DocumentMetadata> deserializeMetadata(Node contentNode) throws RepositoryException {
		
		Node metadataNode = contentNode.getNode(METADATA);
		Map<String,DocumentMetadata> map = new HashMap<String,DocumentMetadata>();
		
		PropertyIterator iterator =  metadataNode.getProperties();
		for (Property property = iterator.nextProperty(); iterator.hasNext(); property = iterator.nextProperty()) {
			if(!property.getName().startsWith("jcr:")) {
				DocumentMetadata metadata = new JCRDocumentMetadata(property.getName(), property.getValue().getString());
				map.put(property.getName(),metadata);
			}
		}
		return map;
	}
		
	private Map<String, String> serializeAnnotations(Node contentNode, Map<String, String> annotations) throws RepositoryException {
		
		Node annotationsNode = contentNode.getNode(ANNOTATIONS);
		for (Entry<String,String> entry : annotations.entrySet()) 
			annotationsNode.setProperty(entry.getKey(), entry.getValue());
		
		return annotations;
	}
	
	private Map<String, String> deserializeAnnotations(Node contentNode) throws RepositoryException {
		
		Map<String,String> map = new HashMap<String,String>();
		Node annotationsNode = contentNode.getNode(ANNOTATIONS);
		PropertyIterator iterator =  annotationsNode.getProperties();
		for (Property property = iterator.nextProperty(); iterator.hasNext(); property = iterator.nextProperty()) {
			if(!property.getName().startsWith("jcr:")) 
				map.put(property.getName(), property.getValue().getString());
		}
		return map;
	}

	@Override
	public String getOID() {
		return oid;
	}

	@Override
	public Map<String, DocumentMetadata> getMetadata() {
		return metadata;
	}

	@Override
	public Map<String, String> getAnnotation() {
		return annotations;
	}

	@Override
	public String getCollectionName() {
		return collectionName;
	}

	@Override
	public String getMimeType() {
		return null;
	}

	@Override
	public FolderItemType getFolderItemType() {
		return FolderItemType.DOCUMENT_LINK;
	}

	@Override
	public long getLength() throws InternalErrorException {
		return 0;
	}

	@Override
	public void saveContent(Node node) throws RepositoryException {
		
		
	}

	
}
