package org.gcube.common.software.analyser;

import java.io.File;
import java.net.URL;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import org.gcube.com.fasterxml.jackson.databind.JsonNode;
import org.gcube.com.fasterxml.jackson.databind.ObjectMapper;
import org.gcube.com.fasterxml.jackson.databind.node.ArrayNode;
import org.gcube.com.fasterxml.jackson.databind.node.ObjectNode;
import org.gcube.common.software.export.SoftwareArtifactProcessor;
import org.gcube.common.software.model.ExporterConfig;
import org.gcube.common.software.model.GlobalConfig;
import org.gcube.common.software.model.SoftwareArtifactConfig;
import org.gcube.common.software.model.SoftwareArtifactFile;
import org.gcube.common.software.model.Variables;
import org.gcube.common.software.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class Analyser {
	
	private static final Logger logger = LoggerFactory.getLogger(Analyser.class);

	protected ObjectMapper objectMapper;
	
	protected File outputDirectory;
	protected ObjectNode globalConfiguration;
	protected ArrayNode versionConfigurations;
	
	public Analyser() throws Exception {
		this.objectMapper = Utils.getObjectMapper();
	}
	
	public void setOutputDirectory(File outputDirectory) {
		this.outputDirectory = outputDirectory;
	}

	public ObjectNode getGlobalConfiguration() {
		return globalConfiguration;
	}

	public void setGlobalConfiguration(ObjectNode originalGlobal) {
		this.globalConfiguration = originalGlobal.deepCopy();
	}
	
	public ArrayNode getVersionConfigurations() {
		return versionConfigurations;
	}

	public void setVersionConfigurations(ArrayNode originalVersions) {
		this.versionConfigurations = originalVersions.deepCopy();
	}
	
	protected SoftwareArtifactConfig actualizeSoftwareVersionConfig(JsonNode version) throws Exception {
		((ObjectNode) version).remove(GlobalConfig.EXPORTERS_PROPERTY_NAME);
		Variables variables = objectMapper.treeToValue(version, Variables.class);
		Set<String> missingVariables = variables.parse();
		int size = missingVariables.size();
		if(size>0) {
			throw new Exception("The following variables has been used but not defined or cannot be actualised" + 
					missingVariables.toArray(new String[size]).toString());
		}
		JsonNode swVersion = objectMapper.convertValue(variables.getProperties(), JsonNode.class);
		SoftwareArtifactConfig softwareVersionConfig = objectMapper.treeToValue(swVersion, SoftwareArtifactConfig.class);
		
		List<SoftwareArtifactFile> svfs = softwareVersionConfig.getFiles();
		for(SoftwareArtifactFile svf : svfs) {
			URL url = svf.getURL();
			String urlString = variables.replaceAllVariables(url.toString());
			svf.setURL(new URL(urlString));
			String desiredName = svf.getDesiredName();
			desiredName = variables.replaceAllVariables(desiredName);
			svf.setDesiredName(desiredName);
		}
		
		return softwareVersionConfig;
	}
	
	protected GlobalConfig getGlobalConfig(JsonNode node) throws Exception {
		Variables variables = objectMapper.treeToValue(node, Variables.class);
		variables.parse();
		JsonNode sc = objectMapper.convertValue(variables.getProperties(), JsonNode.class);
		GlobalConfig globalConfig = objectMapper.treeToValue(sc, GlobalConfig.class);
		globalConfig.setOriginalJson(globalConfiguration);
		return globalConfig;
	}
	
	protected ExporterConfig actualizeExporterConfig(ExporterConfig exporterConfig, SoftwareArtifactConfig softwareVersionConfig) throws Exception {
		ObjectNode versionNode =  objectMapper.valueToTree(softwareVersionConfig);
		Variables versionVariables = objectMapper.treeToValue(versionNode, Variables.class);
		
		ObjectNode node =  objectMapper.valueToTree(exporterConfig);
		Variables variables = objectMapper.treeToValue(node, Variables.class);
		
		variables.parseWith(versionVariables);
		JsonNode ec = objectMapper.convertValue(variables.getProperties(), JsonNode.class);
		return objectMapper.treeToValue(ec, ExporterConfig.class);
	}
	
	protected void checkExporters(Set<String> availableExporterNames, Set<String> requestedExporterNames) throws Exception {
		if(!availableExporterNames.containsAll(requestedExporterNames)) {
			requestedExporterNames.removeAll(availableExporterNames);
			throw new Exception("The following requested exporters does not exists " + requestedExporterNames);
		}
	}
	
	
	public List<File> analyse() throws Exception {
	
		GlobalConfig globalConfig = getGlobalConfig(globalConfiguration);
		
		Map<String, Class<? extends SoftwareArtifactProcessor>> availableExporters = SoftwareArtifactProcessor.getAvailableExporters();
		Map<String,ExporterConfig> requestedExporters = globalConfig.getExporters();
		checkExporters(availableExporters.keySet(), requestedExporters.keySet());
		
		if(outputDirectory==null) {
			outputDirectory = new File(globalConfig.getFileName());
		}
		
		if(!outputDirectory.exists()) {
			Files.createDirectories(outputDirectory.toPath());
		}
		
		SoftwareArtifactConfig previous = null;
		int i = 0; 
		
		List<File> outputFiles = new ArrayList<>();
		
		for(i=0; i<versionConfigurations.size(); i++) {
			ObjectNode versionConfig = (ObjectNode) versionConfigurations.get(i).deepCopy();
			JsonNode mergedVersionConfig = Utils.merge(globalConfiguration, versionConfig);
			
			SoftwareArtifactConfig softwareVersionConfig = actualizeSoftwareVersionConfig(mergedVersionConfig);
			softwareVersionConfig.setOriginalJson(versionConfig);
			softwareVersionConfig.setPrevious(previous);
			
			logger.trace("Going to process {}", softwareVersionConfig.getTitle());			
		
			
			for(String className : requestedExporters.keySet()) {
				logger.debug("Going to export with {}", className);
				Class<? extends SoftwareArtifactProcessor> exporterClass = availableExporters.get(className);
				
				ExporterConfig exporterConfig = requestedExporters.get(className);
				exporterConfig = actualizeExporterConfig(exporterConfig, softwareVersionConfig);
						
				SoftwareArtifactProcessor sve = exporterClass.newInstance();
				sve.setOutputDirectory(outputDirectory);
				sve.setGlobalConfig(globalConfig);
				sve.setSoftwareVersionConfig(softwareVersionConfig); 
				sve.setExporterConfig(exporterConfig);
				sve.setFirst(i==0);
				
				boolean last = i==(versionConfigurations.size()-1); 
				sve.setLast(last);
				sve.export();
				
				if(last) {
					outputFiles.add(sve.getOutputFile());
				}
			} 
			
			Thread.sleep(TimeUnit.SECONDS.toMillis(2));
			
			previous = softwareVersionConfig;
		}
		
		return outputFiles;
		
	}
	
}
