/*
 * Decompiled with CFR 0.152.
 */
package com.mongodb;

import com.mongodb.BasicDBObject;
import com.mongodb.ChangeEvent;
import com.mongodb.ChangeListener;
import com.mongodb.CommandResult;
import com.mongodb.DBObject;
import com.mongodb.DBPort;
import com.mongodb.Loggers;
import com.mongodb.Mongo;
import com.mongodb.MongoOptions;
import com.mongodb.PooledConnectionProvider;
import com.mongodb.ServerAddress;
import com.mongodb.ServerConnectionState;
import com.mongodb.ServerDescription;
import com.mongodb.ServerSettings;
import com.mongodb.ServerType;
import com.mongodb.ServerVersion;
import com.mongodb.SocketSettings;
import com.mongodb.Tags;
import java.io.IOException;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.bson.util.annotations.ThreadSafe;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
@ThreadSafe
class ServerMonitor {
    private static final Logger LOGGER = Loggers.getLogger("cluster");
    private ServerAddress serverAddress;
    private final ChangeListener<ServerDescription> serverStateListener;
    private final SocketSettings socketSettings;
    private final ServerSettings settings;
    private final Mongo mongo;
    private final PooledConnectionProvider connectionProvider;
    private int count;
    private long elapsedNanosSum;
    private volatile boolean isClosed;
    private final Thread monitorThread;
    private final Lock lock = new ReentrantLock();
    private final Condition condition = this.lock.newCondition();

    ServerMonitor(ServerAddress serverAddress, ChangeListener<ServerDescription> serverStateListener, SocketSettings socketSettings, ServerSettings settings, String clusterId, Mongo mongo, PooledConnectionProvider connectionProvider) {
        this.serverAddress = serverAddress;
        this.serverStateListener = serverStateListener;
        this.socketSettings = socketSettings;
        this.settings = settings;
        this.mongo = mongo;
        this.connectionProvider = connectionProvider;
        this.monitorThread = new Thread((Runnable)new ServerMonitorRunnable(), "cluster-" + clusterId + "-" + serverAddress);
        this.monitorThread.setDaemon(true);
    }

    void start() {
        this.monitorThread.start();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void connect() {
        this.lock.lock();
        try {
            this.condition.signal();
        }
        finally {
            this.lock.unlock();
        }
    }

    public void close() {
        this.isClosed = true;
        this.monitorThread.interrupt();
    }

    private MongoOptions getOptions() {
        MongoOptions options = new MongoOptions();
        options.setConnectTimeout(this.socketSettings.getConnectTimeout(TimeUnit.MILLISECONDS));
        options.setSocketTimeout(this.socketSettings.getReadTimeout(TimeUnit.MILLISECONDS));
        options.setSocketFactory(this.socketSettings.getSocketFactory());
        return options;
    }

    static boolean descriptionHasChanged(ServerDescription previousServerDescription, ServerDescription currentServerDescription) {
        return !previousServerDescription.equals(currentServerDescription);
    }

    static boolean stateHasChanged(ServerDescription previousServerDescription, ServerDescription currentServerDescription) {
        return ServerMonitor.descriptionHasChanged(previousServerDescription, currentServerDescription) || previousServerDescription.getAverageLatencyNanos() != currentServerDescription.getAverageLatencyNanos();
    }

    static boolean exceptionHasChanged(Throwable previousException, Throwable currentException) {
        if (currentException == null) {
            return previousException != null;
        }
        if (previousException == null) {
            return true;
        }
        if (!currentException.getClass().equals(previousException.getClass())) {
            return true;
        }
        if (currentException.getMessage() == null) {
            return previousException.getMessage() != null;
        }
        return !currentException.getMessage().equals(previousException.getMessage());
    }

    private ServerDescription lookupServerDescription(DBPort connection) throws IOException {
        LOGGER.fine(String.format("Checking status of %s", this.serverAddress));
        long startNanoTime = System.nanoTime();
        CommandResult isMasterResult = connection.runCommand(this.mongo.getDB("admin"), new BasicDBObject("ismaster", (Object)1));
        ++this.count;
        this.elapsedNanosSum += System.nanoTime() - startNanoTime;
        CommandResult buildInfoResult = connection.runCommand(this.mongo.getDB("admin"), new BasicDBObject("buildinfo", (Object)1));
        return this.createDescription(isMasterResult, buildInfoResult, this.elapsedNanosSum / (long)this.count);
    }

    private ServerDescription createDescription(CommandResult commandResult, CommandResult buildInfoResult, long averageLatencyNanos) {
        return ServerDescription.builder().state(ServerConnectionState.Connected).version(ServerMonitor.getVersion(buildInfoResult)).address(commandResult.getServerUsed()).type(ServerMonitor.getServerType(commandResult)).hosts(this.listToSet((List)commandResult.get("hosts"))).passives(this.listToSet((List)commandResult.get("passives"))).arbiters(this.listToSet((List)commandResult.get("arbiters"))).primary(commandResult.getString("primary")).maxDocumentSize(commandResult.getInt("maxBsonObjectSize", ServerDescription.getDefaultMaxDocumentSize())).maxMessageSize(commandResult.getInt("maxMessageSizeBytes", ServerDescription.getDefaultMaxMessageSize())).maxWriteBatchSize(commandResult.getInt("maxWriteBatchSize", ServerDescription.getDefaultMaxWriteBatchSize())).tags(ServerMonitor.getTagsFromDocument((DBObject)commandResult.get("tags"))).setName(commandResult.getString("setName")).minWireVersion(commandResult.getInt("minWireVersion", ServerDescription.getDefaultMinWireVersion())).maxWireVersion(commandResult.getInt("maxWireVersion", ServerDescription.getDefaultMaxWireVersion())).averageLatency(averageLatencyNanos, TimeUnit.NANOSECONDS).ok(commandResult.ok()).build();
    }

    private static ServerVersion getVersion(CommandResult buildInfoResult) {
        return new ServerVersion(((List)buildInfoResult.get("versionArray")).subList(0, 3));
    }

    private Set<String> listToSet(List<String> list) {
        if (list == null || list.isEmpty()) {
            return Collections.emptySet();
        }
        return new HashSet<String>(list);
    }

    private static ServerType getServerType(BasicDBObject isMasterResult) {
        if (ServerMonitor.isReplicaSetMember(isMasterResult)) {
            if (isMasterResult.getBoolean("ismaster", false)) {
                return ServerType.ReplicaSetPrimary;
            }
            if (isMasterResult.getBoolean("secondary", false)) {
                return ServerType.ReplicaSetSecondary;
            }
            if (isMasterResult.getBoolean("arbiterOnly", false)) {
                return ServerType.ReplicaSetArbiter;
            }
            if (isMasterResult.containsKey("setName") && isMasterResult.containsField("hosts")) {
                return ServerType.ReplicaSetOther;
            }
            return ServerType.ReplicaSetGhost;
        }
        if (isMasterResult.containsKey("msg") && isMasterResult.get("msg").equals("isdbgrid")) {
            return ServerType.ShardRouter;
        }
        return ServerType.StandAlone;
    }

    private static boolean isReplicaSetMember(BasicDBObject isMasterResult) {
        return isMasterResult.containsKey("setName") || isMasterResult.getBoolean("isreplicaset", false);
    }

    private static Tags getTagsFromDocument(DBObject tagsDocuments) {
        if (tagsDocuments == null) {
            return new Tags();
        }
        Tags tags = new Tags();
        for (String key : tagsDocuments.keySet()) {
            tags.put(key, tagsDocuments.get(key).toString());
        }
        return tags;
    }

    private ServerDescription getConnectingServerDescription(Throwable throwable) {
        return ServerDescription.builder().type(ServerType.Unknown).state(ServerConnectionState.Connecting).address(this.serverAddress).exception(throwable).build();
    }

    class ServerMonitorRunnable
    implements Runnable {
        ServerMonitorRunnable() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void run() {
            DBPort connection = null;
            try {
                ServerDescription currentServerDescription = ServerMonitor.this.getConnectingServerDescription(null);
                Throwable currentException = null;
                while (!ServerMonitor.this.isClosed) {
                    ServerDescription previousServerDescription = currentServerDescription;
                    Throwable previousException = currentException;
                    try {
                        if (connection == null) {
                            connection = new DBPort(ServerMonitor.this.serverAddress, null, ServerMonitor.this.getOptions(), 0);
                        }
                        try {
                            currentServerDescription = ServerMonitor.this.lookupServerDescription(connection);
                        }
                        catch (IOException e) {
                            ServerMonitor.this.count = 0;
                            ServerMonitor.this.elapsedNanosSum = 0L;
                            if (connection != null) {
                                connection.close();
                                connection = null;
                                ServerMonitor.this.connectionProvider.invalidate();
                            }
                            connection = new DBPort(ServerMonitor.this.serverAddress, null, ServerMonitor.this.getOptions(), 0);
                            try {
                                currentServerDescription = ServerMonitor.this.lookupServerDescription(connection);
                            }
                            catch (IOException e1) {
                                connection.close();
                                connection = null;
                                throw e1;
                            }
                        }
                    }
                    catch (Throwable t) {
                        currentException = t;
                        currentServerDescription = ServerMonitor.this.getConnectingServerDescription(t);
                    }
                    if (!ServerMonitor.this.isClosed) {
                        try {
                            this.logStateChange(previousServerDescription, previousException, currentServerDescription, currentException);
                            this.sendStateChangedEvent(previousServerDescription, currentServerDescription);
                        }
                        catch (Throwable t) {
                            LOGGER.log(Level.WARNING, "Exception in monitor thread during notification of server state change", t);
                        }
                    }
                    this.waitForNext();
                }
            }
            finally {
                if (connection != null) {
                    connection.close();
                }
            }
        }

        private void sendStateChangedEvent(ServerDescription previousServerDescription, ServerDescription currentServerDescription) {
            if (ServerMonitor.stateHasChanged(previousServerDescription, currentServerDescription)) {
                ServerMonitor.this.serverStateListener.stateChanged(new ChangeEvent<ServerDescription>(previousServerDescription, currentServerDescription));
            }
        }

        private void logStateChange(ServerDescription previousServerDescription, Throwable previousException, ServerDescription currentServerDescription, Throwable currentException) {
            if (ServerMonitor.descriptionHasChanged(previousServerDescription, currentServerDescription) || ServerMonitor.exceptionHasChanged(previousException, currentException)) {
                if (currentException != null) {
                    LOGGER.log(Level.INFO, String.format("Exception in monitor thread while connecting to server %s", ServerMonitor.this.serverAddress), currentException);
                } else {
                    LOGGER.info(String.format("Monitor thread successfully connected to server with description %s", currentServerDescription));
                }
            }
        }

        private void waitForNext() {
            try {
                long millisToSleep;
                long minimumNanosToWait;
                long timeWaiting;
                long timeRemaining = this.waitForSignalOrTimeout();
                if (timeRemaining > 0L && (timeWaiting = ServerMonitor.this.settings.getHeartbeatFrequency(TimeUnit.NANOSECONDS) - timeRemaining) < (minimumNanosToWait = ServerMonitor.this.settings.getHeartbeatConnectRetryFrequency(TimeUnit.NANOSECONDS)) && (millisToSleep = TimeUnit.MILLISECONDS.convert(minimumNanosToWait - timeWaiting, TimeUnit.NANOSECONDS)) > 0L) {
                    Thread.sleep(millisToSleep);
                }
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private long waitForSignalOrTimeout() throws InterruptedException {
            ServerMonitor.this.lock.lock();
            try {
                long l = ServerMonitor.this.condition.awaitNanos(ServerMonitor.this.settings.getHeartbeatFrequency(TimeUnit.NANOSECONDS));
                return l;
            }
            finally {
                ServerMonitor.this.lock.unlock();
            }
        }
    }
}

