/**
 * 
 */
package org.gcube.data.tml.uri;

import static java.util.Collections.*;
import static org.gcube.data.trees.patterns.Patterns.*;
import gr.uoa.di.madgik.grs.record.GenericRecord;
import gr.uoa.di.madgik.grs.record.field.Field;
import gr.uoa.di.madgik.grs.record.field.StringField;

import java.io.IOException;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLConnection;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;

import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.scope.GCUBEScopeManager;
import org.gcube.data.tm.stubs.Path;
import org.gcube.data.tml.uri.tm.Handler;
import org.gcube.data.trees.data.Node;
import org.gcube.data.trees.data.Tree;
import org.gcube.data.trees.patterns.Pattern;
import org.globus.wsrf.encoding.ObjectSerializer;

/**
 * Utility methods for data URI creation and manipulation.
 * @author Fabio Simeoni
 *
 */
public class URIs {

	static {
		Handler.activateProtocol();
	}
	
	/**Scheme of cms URIs.*/
	public static final String PROTOCOL="tm";
	
	/**
	 * Indicates whether a URI is a valid data URI.
	 * @param uri the URI.
	 * @throws URISyntaxException if the URI fails validation.
	 */
	public static void validate(URI uri) throws URISyntaxException {
		if (!PROTOCOL.equals(uri.getScheme()) || 
				uri.getAuthority()==null || 
					uri.getPath()==null ||
						uri.getPath().length()<2)
			throw new IllegalArgumentException(new URISyntaxException(uri.toString(),"uri is not a well-formed data URI"));
	}

	/**
	 * Constructs a data URI from a source identifier and one or more node identifiers.
	 * @param sourceId the source identifier.
	 * @param identifiers the node identifiers.
	 * @return the URI.
	 * @throws IllegalArgumentException if the input is <code>null</code> or empty.
	 */
	public static URI make(String sourceId, String ... identifiers) throws IllegalArgumentException {
		
		if (sourceId==null || identifiers==null || identifiers.length==0)
			throw new IllegalArgumentException("null or empty input");
		
		StringBuilder path = new StringBuilder();
		for (Object id : identifiers) 
			path.append("/"+encode(id.toString()));

		URI uri = null;
		try {
			uri = new URI(PROTOCOL,encode(sourceId),path.toString(),null);
		}
		catch(URISyntaxException e) {
			throw new RuntimeException("error in generation uri with "+PROTOCOL+","+encode(sourceId)+","+path,e);
		}
		return uri;
	}
	
	/**
	 * Returns the source identifier in a data URI.
	 * @param uri the URI.
	 * @return the identifier.
	 * @throws URISyntaxException if the URI is not a data URI.
	 */
	public static String sourceId(URI uri) throws URISyntaxException {
		validate(uri);
		return decode(uri.getAuthority());
	}
	
	/**
	 * Returns the root identifier in a data URI.
	 * @param uri the URI.
	 * @return the identifier.
	 * @throws URISyntaxException if the URI is not a data URI.
	 */
	public static String rootId(URI uri) throws URISyntaxException {
		validate(uri);
		String p = uri.getPath().substring(1);
		if (p.endsWith("/")) 
			p = p.substring(0,p.length()-1);
		int index = p.indexOf("/");
		return decode(p.substring(0,index>0?index:p.length()));
	}
	
	/**
	 * Returns the identifier of the node identified by a data URI.
	 * @param uri the URI.
	 * @return the identifier.
	 * @throws URISyntaxException if the URI is not a data URI.
	 */
	public static String nodeID(URI uri) throws URISyntaxException {
		validate(uri);
		return decode(uri.getPath().substring(uri.getPath().lastIndexOf("/")+1));
	}
	
	/**
	 * Returns the identifiers in a data URI.
	 * @param uri the URI.
	 * @return the identifiers.
	 * @throws URISyntaxException if the URI is not a data URI.
	 */
	public static String[] nodeIDs(URI uri) throws URISyntaxException {
		validate(uri);
		List<String> ids = new ArrayList<String>();
		for (String s : uri.getPath().substring(1).split("/")) //will be validated here
			ids.add(decode(s));
		return ids.toArray(new String[0]);
	}
	
	/**
	 * Returns a data URI for the parent of the node identified by another data URI.
	 * @param uri the input URI.
	 * @return the parent URI.
	 * @throws URISyntaxException if the input URI is not a content URL.
	 */
	public static URI parentURI(URI uri) throws URISyntaxException {
		validate(uri);
		String u = uri.getPath();
		return	make(uri.getAuthority(),u.substring(1,u.lastIndexOf("/")).split("/"));
	}
	
	/**
	 * Returns a data URI for the tree of the node identified by another data URI.
	 * @param uri the input URI.
	 * @return the tree URI.
	 * @throws URISyntaxException if the input URI is not a data URI.
	 */
	public static URI treeURI(URI uri) throws URISyntaxException {
		validate(uri);
		return make(uri.getAuthority(),rootId(uri));
	}
	
	//decoding helper
	private static String decode(String s) {
		
		String decoded = null;
		try {
			decoded = URLDecoder.decode(s, "UTF-8");
		}
		catch(UnsupportedEncodingException e) {}//surely supported
		
		return decoded;
	}
	
	//encoding helper
	private static String encode(String s) {
		
		String encoded = null;
		try {
			encoded = URLEncoder.encode(s, "UTF-8");
		}
		catch(UnsupportedEncodingException e) {}//surely supported
		
		return encoded;
	}
	
	/**
	 * Returns a {@link Pattern} for the existence of the path of identifiers in a data URI.
	 * @param uri the URI.
	 * @return the predicate.
	 * @throws URISyntaxException if the URI is not a data URI.
	 */
	public static final Pattern pattern(URI uri) throws URISyntaxException {
		return hasPath(nodeIDs(uri));
	}
	
	/**
	 * Returns the serialisation of a data URI as an element of ResultSet, as expected by the service.
	 * @param uri the URI.
	 * @return the element.
	 * @throws URISyntaxException if the URI is not a data URI.
	 * @throws Exception  if the URI could not be serialised as an element.
	 */
	public static final GenericRecord toRSElement(URI uri) throws URISyntaxException, Exception {
		StringWriter writer =new StringWriter();
		ObjectSerializer.serialize(writer, new Path(nodeIDs(uri)), Path.getTypeDesc().getXmlType());
		GenericRecord record = new GenericRecord();
		record.setFields(new Field[]{new StringField(writer.toString())});
		return record;
	}
	
	/**
	 * Returns a URL connection in a given scope.
	 * @param uri a data URI.
	 * @param scope the scope.
	 * @return the connection.
	 * @throws IOException if the connections could not be established.
	 * @throws URISyntaxException if the URI is not a data URI or if the protocol handler for the <code>cms</code> scheme is not active.
	 * @deprecated since 2.3.1. Use {@link URLConnection} normally in current scope.
	 */
	public static URLConnection connection(URI uri, String scope) throws IOException, URISyntaxException {
		validate(uri);
		URLConnection connection = uri.toURL().openConnection();
		GCUBEScopeManager.DEFAULT.setScope(GCUBEScope.getScope(scope));
		return connection;
	}
	
	/**
	 * Returns the data URI of a {@link Node}. 
	 * <p>
	 * <p>
	 * <code>tm://</code><em>id0</em><code>/</code><em>id1</em><code>/</code>...<code>/</code><em>idN</em>
	 * <p>
	 * where 
	 * <ul>
	 * <li><em>id0</em> is the identifier of the data source that contains the tree in which the node occurs;</li>
	 * <li><em>id1</em> is the identifier of the root of the tree;</li>
	 * <li><em>idN</em> is the identifier of the node;</li>
	 * <li><code>/</code><em>id2</em>...<em>idN-1</em> are the identifiers of the nodes on the path that 
	 * connects the root of the tree to the node.</li>
	 * </ul>
	 * 
	 * @param node the node
	 * @return the URI
	 * @throws IllegalStateException if the node or one of its ancestors does not have an identifier, or
	 * if the root of the tree does not specify a source identifier
	 */
	public static URI uri(Node node) throws IllegalStateException {
		
		if (node.id()==null) 
			throw new IllegalStateException("node has no identifier");
		
		//recursion here less efficient
		Node child = node;
		Node last=null;
		List<String> path = new ArrayList<String>();
		while (child!=null) {
			
			if (child.id()==null) 
				throw new IllegalStateException("no identifier for node "+child);
			
			path.add(child.id());
			
			last= child;
			child = child.parent();
		}
		
		String sourceId = ((Tree) last).sourceId();
		if (sourceId==null) 
			throw new IllegalStateException("root has no source identifier");
		
		reverse(path);
		
		return make(sourceId, path.toArray(new String[0]));
	}
	
}
