/**
 * 
 */
package org.gcube.data.tml.utils;

import static org.gcube.data.streams.dsl.Streams.*;

import java.io.StringReader;
import java.io.StringWriter;
import java.net.URI;

import org.gcube.common.core.faults.GCUBEFault;
import org.gcube.data.streams.Stream;
import org.gcube.data.streams.dsl.publish.PublishRsWithClause;
import org.gcube.data.streams.generators.Generator;
import org.gcube.data.tm.stubs.AddOutcome;
import org.gcube.data.tm.stubs.AddOutcomeFailure;
import org.gcube.data.tm.stubs.AddOutcomeSuccess;
import org.gcube.data.tm.stubs.InvalidTreeFault;
import org.gcube.data.tm.stubs.Path;
import org.gcube.data.tm.stubs.UnknownTreeFault;
import org.gcube.data.tm.stubs.UpdateFailure;
import org.gcube.data.tml.exceptions.InvalidTreeException;
import org.gcube.data.tml.exceptions.UnknownTreeException;
import org.gcube.data.tml.outcomes.AddTreeOutcome;
import org.gcube.data.tml.outcomes.UpdateTreeFailure;
import org.gcube.data.trees.data.Node;
import org.gcube.data.trees.data.Tree;
import org.globus.wsrf.encoding.ObjectDeserializer;
import org.globus.wsrf.encoding.ObjectSerializer;
import org.xml.sax.InputSource;

/**
 * Result set conversion facilities.
 * 
 * @author Fabio Simeoni
 *
 */
public class TMStreams {

	
	/**
	 * Pipes an {@link Stream} of {@link Node}s through a logger.
	 * @param stream the stream
	 * @return the logged stream
	 */
	public static <N extends Node> Stream<N> log(Stream<N> stream) {
		TMStreamLogger<N> logger = new TMStreamLogger<N>();
		return monitor(pipe(stream).through(logger)).with(logger);
	}

	/**
	 * Converts a result set of {@link Path}s into a {@link Stream}.
	 * @param locator the result set locator
	 * @return the stream
	 */
	public static Stream<Path> pathsIn(URI locator) {
		return pipe(stringsIn(locator)).through(new PathParser());
	}
	
	/**
	 * Converts a result set of {@link AddOutcome}s into a {@link Stream}
	 * of added trees that throws faults in correspondence of remote addition failures.
	 * @param locator the result set locator
	 * @return the stream
	 */
	public static Stream<Tree> addedTreesIn(URI locator) {
		return pipe(stringsIn(locator)).through(new AddOutcomeIdParser());
	}
	
	/**
	 * Converts a result set of {@link UpdateFailure}s into a {@link Stream}
	 * of {@link Throwable}s.
	 * @param locator the result set locator
	 * @return the stream
	 */
	public static Stream<UpdateFailure> updateFailuresIn(URI locator) {
		return pipe(stringsIn(locator)).through(new UpdateFailureParser());
	}
	
	/** 
	 * Publishes a {@link Stream} of {@link Path}s.
	 * @param stream the stream
	 * @return an ongoing publication sentence ready for further configuration
	 */
	public static PublishRsWithClause<Path> publishPathsIn(Stream<Path> stream) {
		return publish(stream).using(new PathSerialiser());
	}
	 
	 
	/**
	 * Publishes a {@link Stream} of {@link UpdateTreeFailure}s.
	 * @param stream the stream
	 * @return an ongoing publication sentence ready for further configuration
	 */
	public static PublishRsWithClause<UpdateTreeFailure> publishUpdateFailures(Stream<UpdateTreeFailure> stream) {
		return publish(stream).using(new UpdateTreeFailureSerialiser());
	}
	
	/**
	 * Publishes a {@link Stream} of {@link AddTreeOutcome}s.
	 * @param stream the stream
	 * @return an ongoing publication sentence ready for further configuration
	 */
	public static PublishRsWithClause<AddTreeOutcome> publishAddOutcomes(Stream<AddTreeOutcome> stream) {
		return publish(stream).using(new AddTreesOutcomeSerialiser());
	}
	
	
	
	
	private static class PathParser implements Generator<String, Path> {

		public Path yield(String element)  {
			try {
				return (Path) ObjectDeserializer.deserialize(new InputSource(
					new StringReader(element)), Path.class);
			}
			catch(Exception e) {
				throw new RuntimeException(e);
			}
		}

	}
	
	private static class AddOutcomeIdParser implements Generator<String,Tree> {
		
		public Tree yield(String element) {
			
			try {
				AddOutcome outcome = 
					(AddOutcome) ObjectDeserializer.deserialize(new InputSource(new StringReader(element)), AddOutcome.class);
				if (outcome.getSuccess()!=null)
					return Utils.toTree(outcome.getSuccess().getOutput());
				else
					//axis is broken and will not return a specific subclass
					throw outcome.getFailure().getFault().toException();
			} catch (Exception e) {
				throw new RuntimeException(e);
			}
		}
	}

	private static class UpdateFailureParser implements Generator<String,UpdateFailure> {
		
		public UpdateFailure yield(String element) {
			try {
				return (UpdateFailure) ObjectDeserializer.deserialize(new InputSource(new StringReader(element)), UpdateFailure.class);
			}
			catch(Exception e) {
				throw new RuntimeException(e);
			}
		}

	}
	
	private static class PathSerialiser implements Generator<Path,String> {
		
		public String yield(Path path) {
			
			try {
				
				StringWriter w = new StringWriter();
				
				ObjectSerializer.serialize(w, path, Path.getTypeDesc().getXmlType());
			
				return w.toString();
			}
			catch (Exception e) {//we take this serialisation failure to be unrecoverable
				throw new RuntimeException(e);
			}
		}
	};
	
	private static class UpdateTreeFailureSerialiser implements Generator<UpdateTreeFailure,String> {
		
		public String yield(UpdateTreeFailure outcome) {
			
			try {
				
				GCUBEFault fault = null;
				Exception failure = outcome.failure();
				
				if (failure instanceof UnknownTreeException)
					fault = new UnknownTreeFault();
				else if (failure instanceof InvalidTreeException)
					fault = new InvalidTreeFault();
				else
					fault = new GCUBEFault();
					
				fault = Utils.newFault(fault, failure);
				
				UpdateFailure failureStub = new UpdateFailure(fault,outcome.identifier()); 
				
				StringWriter w = new StringWriter();
				
				ObjectSerializer.serialize(w, failureStub, UpdateFailure.getTypeDesc().getXmlType());
			
				return w.toString();
			}
			catch (Exception e) {//we take this serialisation failure to be unrecoverable
				throw new RuntimeException(e);
			}
		}
	};
	
	
	private static class AddTreesOutcomeSerialiser implements Generator<AddTreeOutcome,String> {
		
		public String yield(AddTreeOutcome outcome) {
			
			try {
				
				AddOutcome stubOutcome= new AddOutcome();
				
				if (outcome.isFailure()) {
					Exception failure= outcome.failure();
					GCUBEFault fault = failure instanceof InvalidTreeException?
									new InvalidTreeFault():
										new GCUBEFault();
					fault = Utils.newFault(fault, failure);
					stubOutcome.setFailure(new AddOutcomeFailure(fault));
				}
				else
					stubOutcome.setSuccess(new AddOutcomeSuccess(Utils.toHolder((outcome.tree()))));
				
				StringWriter w = new StringWriter();
				
				ObjectSerializer.serialize(w, stubOutcome, AddOutcome.getTypeDesc().getXmlType());
				return w.toString();
			}
			catch (Exception e) { //we take this serialisation failure to be unrecoverable
				throw new RuntimeException(e);
			}
			
		}
	};
}
