package org.gcube.data.tml.clients;

import static org.gcube.data.tml.clients.APIUtils.*;
import gr.uoa.di.madgik.grs.reader.ForwardReader;
import gr.uoa.di.madgik.grs.record.GenericRecord;
import gr.uoa.di.madgik.grs.record.field.StringField;

import java.io.StringReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.gcube.common.clients.Client;
import org.gcube.common.clients.DiscoveryException;
import org.gcube.common.clients.gcore.GCoreDiscoveryClient;
import org.gcube.common.clients.gcore.GCorePassThroughClient;
import org.gcube.common.clients.gcore.StatefulQuery;
import org.gcube.data.tm.stubs.BindOutcome;
import org.gcube.data.tm.stubs.InvalidRequestFault;
import org.gcube.data.tm.stubs.SourceBinding;
import org.gcube.data.tm.stubs.SourceBindings;
import org.gcube.data.tm.stubs.TBinderPortType;
import org.gcube.data.tml.clients.APIUtils.TBinderCall;
import org.gcube.data.tml.clients.providers.TBinderProvider;
import org.gcube.data.tml.exceptions.InvalidRequestException;
import org.globus.wsrf.encoding.ObjectDeserializer;
import org.xml.sax.InputSource;

/**
 * A {@link TBinderClient} implementation.
 
 * @author Fabio Simeoni
 *
 */
public class BinderClient implements TBinderClient  {
	
	final Client<TBinderPortType> client;
	
	/**
	 * Creates an instance with a given {@link Client}.
	 * @param client the client
	 */
	public BinderClient(Client<TBinderPortType> client) {
		this.client=client;
	}
	
	/**
	 * Creates an instance for binding to Tree Binder services discovered with a given query.
	 * @param query the query.
	 */
	public BinderClient(StatefulQuery query) {
		this(new GCoreDiscoveryClient<TBinderPortType>(TBinderProvider.INSTANCE,query));
	}
	
	/**
	 * Creates an instance for binding to a given Tree Binder service.
	 * @param address the service address
	 */
	public BinderClient(URL address) {
		this(new GCorePassThroughClient<TBinderPortType>(TBinderProvider.INSTANCE,endpoint(address,TBinderProvider.INSTANCE.name())));
	}

	/**{@inheritDoc}*/ 
	@Override
	public List<SourceBinding> bind(final BindingParameters parameters) throws IllegalArgumentException,DiscoveryException, InvalidRequestException, Exception {

		if (parameters==null)
			throw new IllegalArgumentException("parameters are null");
		
		TBinderCall<SourceBindings> call = new TBinderCall<SourceBindings>() {
			@Override
			public SourceBindings call(TBinderPortType service) throws Exception {
				try {
					return service.bind(parameters.parameters);
				}
				catch(InvalidRequestFault f) {//transforms specific fault into @unrecoverable exception, preserving remote cause
					throw new InvalidRequestException(f.getFaultMessage(),f.toException().getCause());
				}
				//other faults are let through and will be treated as retry-equivalent semantics
			}
		};
				
		SourceBindings output = client.make(call);
		
		List<SourceBinding> bindings = new ArrayList<SourceBinding>();
		
		if (output!=null && output.getBindings()!=null)
			for (SourceBinding binding : output.getBindings()) 
				bindings.add(binding);
		
		return bindings;
		
	}
	
	/**{@inheritDoc}*/ 
	@Override
	public void bindAsync(final BindingParameters parameters, final BinderConsumer consumer) throws IllegalArgumentException,DiscoveryException,Exception {
		
		if (parameters==null || consumer==null)
			throw new IllegalArgumentException("parameters or consumers are null");
		
		TBinderCall<Void> call = new TBinderCall<Void>() {
			@Override
			public Void call(TBinderPortType service) throws Exception {

				String locatorString = service.bindAsync(parameters.parameters);
				URI locator = null;
				try {
					//invoke service
					locator = new URI(locatorString);
				}
				catch(URISyntaxException e) {
					throw new Exception("null response",e);
				}
	
				Iterator<GenericRecord> iter=new ForwardReader<GenericRecord>(locator).iterator();
				BindOutcome response = null ;
			    
				while(iter.hasNext()){
			    	GenericRecord elem= iter.next(); 
			        if(elem!=null) {
			        	String payload = ((StringField) elem.getField(0)).getPayload();
			        	response = (BindOutcome) ObjectDeserializer.deserialize(new InputSource(new StringReader(payload)), BindOutcome.class);	        			       
	
			        }
			    }
			    
			    if (response==null)  
			    	throw new Exception("asynchronous response is invalid (empty channel)");
			     
				if (response.getSuccess()!=null) {
					List<SourceBinding> bindings = new ArrayList<SourceBinding>();
					for (SourceBinding binding : response.getSuccess().getBindings()) 
						bindings.add(binding);
					consumer.onCompletion(bindings);
				}
				else if (response.getFailure()!=null) { 
					//unfortunately, on deserialization as GCUBEFault Axis loses the specific fault. 
					consumer.onFailure(response.getFailure().getFault());
				}
				else 
					throw new Exception("asyncrhonous response is invalid (no evidence of either failure or success)");
					
					return null;
			}
		};
		
		
		client.make(call);
		
	}
}
