package org.gcube.data.tml.clients;

import static org.gcube.data.tml.utils.TMStreams.*;
import static org.gcube.data.trees.io.Bindings.*;
import static org.gcube.data.trees.streams.TreeStreams.*;

import java.net.URI;

import org.apache.axis.message.addressing.EndpointReferenceType;
import org.gcube.common.clients.Client;
import org.gcube.common.clients.DiscoveryException;
import org.gcube.common.clients.gcore.GCoreDiscoveryClient;
import org.gcube.common.clients.gcore.GCorePassThroughClient;
import org.gcube.common.clients.gcore.StatefulQuery;
import org.gcube.common.core.faults.GCUBEFault;
import org.gcube.common.core.faults.GCUBEUnrecoverableException;
import org.gcube.data.streams.Stream;
import org.gcube.data.tm.stubs.InvalidTreeFault;
import org.gcube.data.tm.stubs.TWriterPortType;
import org.gcube.data.tm.stubs.UnknownTreeFault;
import org.gcube.data.tm.stubs.UnsupportedOperationFault;
import org.gcube.data.tm.stubs.UnsupportedRequestFault;
import org.gcube.data.tm.stubs.UpdateFailure;
import org.gcube.data.tml.clients.APIUtils.TWriterCall;
import org.gcube.data.tml.clients.providers.TWriterProvider;
import org.gcube.data.tml.clients.queries.QueryBuilder;
import org.gcube.data.tml.exceptions.InvalidTreeException;
import org.gcube.data.tml.exceptions.UnknownTreeException;
import org.gcube.data.tml.exceptions.UnsupportedOperationException;
import org.gcube.data.tml.exceptions.UnsupportedRequestException;
import org.gcube.data.tml.utils.Utils;
import org.gcube.data.trees.data.Tree;
import org.oasis.wsrf.faults.BaseFaultType;
import org.oasis.wsrf.lifetime.Destroy;
import org.w3c.dom.Element;


/**
 * A {@link TWriterClient} implementation.
 
 * @author Fabio Simeoni
 */
public class WriterClient implements TWriterClient  {

	//inner service client that executes calls
	private final Client<TWriterPortType> client;

	
	public WriterClient(Client<TWriterPortType> client) {
		this.client=client;
	}
	
	/**
	 * Creates an instance for writing in a data source discovered with a given query.
	 * @param query the query.
	 */
	public WriterClient(StatefulQuery query) {
		this(new GCoreDiscoveryClient<TWriterPortType>(TWriterProvider.INSTANCE,query));
	}
	
	/**
	 * Creates an instance for writing in a given data source.
	 * @param sourceId the source identifier
	 */
	public WriterClient(String sourceId) {
		this(QueryBuilder.findWriteSource().withId(sourceId).build());
	}
	
	/**
	 * Creates an instance to write through a service at a given endpoint.
	 * @param endpoint the service endpoint
	 */
	public WriterClient(EndpointReferenceType endpoint) {
		this(new GCorePassThroughClient<TWriterPortType>(TWriterProvider.INSTANCE,endpoint));
	}

	
	@Override
	public Tree add(final Tree doc) throws DiscoveryException,InvalidTreeException,Exception {
		return add(toElement(doc));
	}
	
	@Override
	public Tree add(final Element doc) throws DiscoveryException,InvalidTreeException,Exception {
		
		TWriterCall<Tree> call = new TWriterCall<Tree>() {
			public Tree call(TWriterPortType service) throws Exception {
				try {
					return Utils.toTree(service.add(Utils.toHolder(doc))); 
				}
				catch(UnsupportedOperationFault f) {
					throw new UnsupportedOperationException("operation is not supported by target plugin");
				}
				catch(UnsupportedRequestFault f) {
					throw new UnsupportedRequestException("request is not supported by target plugin");
				}
				catch(InvalidTreeFault f) {
					throw new InvalidTreeException("document is invalid",f.toException().getCause());
				}
				catch(GCUBEFault f) {
					throw f.toException();
				}
			}
		};
		
		return client.make(call);
		
	}

	
	@Override
	public Stream<Tree> add(Stream<Tree> trees) throws DiscoveryException,Exception {
		
		URI treeRs = publishTreesIn(trees).withDefaults();
		return add(treeRs);
	}
	
	@Override
	public Stream<Tree> add(final URI treeRs) throws DiscoveryException,Exception {
		
		TWriterCall<String> call = new TWriterCall<String>() {
			public String call(TWriterPortType service) throws Exception {
				try{
					return service.addRS(treeRs.toString()); 
				}
				catch(UnsupportedOperationFault f) {
					throw new UnsupportedOperationException("operation is not supported by target plugin");
				}
				catch(UnsupportedRequestFault f) {
					throw new UnsupportedRequestException("request is not supported by target plugin");
				}
				catch(GCUBEFault f) {
					throw f.toException();
				}
			}
		};
		
		String outcomeRs = client.make(call); 
		return addedTreesIn(java.net.URI.create(outcomeRs));
		
	}

	
	@Override
	public void update(final Tree doc) throws Exception, DiscoveryException, InvalidTreeException, UnknownTreeException {
		
		TWriterCall<Void> call = new TWriterCall<Void>() {
			public Void call(TWriterPortType service) throws Exception {
				try{
					service.update(Utils.toHolder(toElement(doc)));
					return null;
				}
				catch(UnsupportedOperationFault f) {
					throw new UnsupportedOperationException("operation is not supported by target plugin");
				}
				catch(UnsupportedRequestFault f) {
					throw new UnsupportedRequestException("request is not supported by target plugin");
				}
				catch(InvalidTreeFault f) {
					throw new InvalidTreeException("delta tree is malformed",f.toException().getCause());
				}
				catch(UnknownTreeFault f) {
					throw new UnknownTreeException("delta tree does not identify a known tree",f.toException().getCause());
				}
				catch(GCUBEFault f) {
					throw f.toException();
				}
			}
		};
		
		client.make(call);
		
	}
	

	@Override
	public Stream<UpdateFailure> update(Stream<Tree> deltas) throws Exception, DiscoveryException {
		URI deltaRs = publishTreesIn(deltas).withDefaults();
		return update(deltaRs);
		
	}
	
	@Override
	public Stream<UpdateFailure> update(final URI locator) throws Exception, DiscoveryException {
		
		TWriterCall<String> call = new TWriterCall<String>() {
			public String call(TWriterPortType service) throws Exception {
				try{
					return service.updateRS(locator.toString()); 
				}
				catch(UnsupportedOperationFault f) {
					throw new UnsupportedOperationException("operation is not supported by target plugin");
				}
				catch(UnsupportedRequestFault f) {
					throw new UnsupportedRequestException("request is not supported by target plugin");
				}
				catch(GCUBEFault f) {
					throw f.toException();
				}
			}
		};
		
		String failuresRs = client.make(call);
		return updateFailuresIn(java.net.URI.create(failuresRs));
		
	}
	
	/**{@inheritDoc}*/
	synchronized void delete() throws Exception {
		
		TWriterCall<Void> call = new TWriterCall<Void>() {
			public Void call(TWriterPortType service) throws Exception {
				
				try{
					service.destroy(new Destroy());
					return null;
				}
				catch (BaseFaultType f) {
					throw new GCUBEUnrecoverableException(f);
				}
			}
		};
		
		client.make(call);
		
		
	}
}
