package org.gcube.data.tml.clients.queries;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;

import org.gcube.common.clients.gcore.StatefulQuery;
import org.gcube.data.tml.Constants;
import org.gcube.data.tml.clients.providers.AbstractProvider;

/**
 * Builds queries for Tree Manager services that give read or write access to data sources.
 * 
 * @author Fabio Simeoni
 *
 */
public class SourceQueryBuilder {

	static private final String RPCondition = 
			"$result/descendant::*[local-name()='%1$s'] eq '%2$s'";
	
	static private final String QNameRPCondition = 
			"some $type in $result/descendant::*[local-name()='Type'] " +
			"satisfies (resolve-QName($type,$type) eq QName('%1$s','%2$s'))";
	
	static private final String sourcePropertyCondition = 
			"some $prop in $result/descendant::*[local-name()='"+Constants.PROPERTY_RPNAME+"'] satisfies (%1$s)";
	
	static private final String propertyCondition = 
			"$prop/child::*[local-name()='%1$s'] eq '%2$s'";
	
	
	
	private AbstractProvider<?> provider;
	private String sourceId;
	private String name;
	private String plugin;
	private Map<String,String> properties = new HashMap<String, String>();
	private List<QName> types = new ArrayList<QName>();
	
	protected SourceQueryBuilder(AbstractProvider<?> provider) {
		this.provider=provider;
	}
	
	/**
	 * Sets a source identifier on the query.
	 * @param id the identifier
	 * @return this builder
	 */
	public SourceQueryBuilder withId(String id) {
		this.sourceId=id;
		return this;
	}
	
	/**
	 * Sets a source name on the query.
	 * @param name the name
	 * @return this builder
	 */
	public SourceQueryBuilder withName(String name) {
		this.name=name;
		return this;
	}
	
	/**
	 * Sets a source type on the query.
	 * @param name the type
	 * @return this builder
	 */
	public SourceQueryBuilder withType(QName type) {
		this.types.add(type);
		return this;
	}
	
	/**
	 * Sets a plugin on the query.
	 * @param name the plugin name
	 * @return this builder
	 */
	public SourceQueryBuilder withPlugin(String name) {
		this.plugin=name;
		return this;
	}
	
	/**
	 * Sets a source property on the query.
	 * @param name the name of the property
	 * @param value the value of the property
	 * @return
	 */
	public SourceQueryBuilder withProperty(String name, String value) {
		this.properties.put(name,value);
		return this;
	}
	
	
	/**
	 * Returns the query.
	 * @return the query.
	 */
	public StatefulQuery build() {
	
		StatefulQuery query = new StatefulQuery(provider);
		
		String condition = toString();
		if (!condition.isEmpty())
			query.query().addGenericCondition(condition);
		
		return query;
	}
	
	@Override
	public String toString() {
		
		List<String> conditions = new ArrayList<String>();
		
		if (sourceId!=null)
			conditions.add(String.format(RPCondition,Constants.SOURCEID_RPNAME,sourceId));
		
		if (name!=null)
			conditions.add(String.format(RPCondition,Constants.SOURCENAME_RPNAME,name));
		
		if (!types.isEmpty())
			for (QName type : types)
				conditions.add(String.format(QNameRPCondition,Constants.SOURCETYPE_RPNAME,type.getNamespaceURI(),type.getLocalPart()));
		
		if (plugin!=null)
			conditions.add(String.format(RPCondition,Constants.PLUGIN_RPNAME,plugin));
		
		if (!properties.isEmpty()) {
			List<String> propConditions = new ArrayList<String>(); 
			for (Map.Entry<String,String> prop : properties.entrySet())
				propConditions.add(String.format(propertyCondition,prop.getKey(),prop.getValue()));
			conditions.add(String.format(sourcePropertyCondition,QueryBuilder.concatenate(propConditions)));
		}
		
		return conditions.isEmpty()?"":QueryBuilder.concatenate(conditions);
	}

}
