package org.gcube.data.tm.state;

import static org.gcube.data.tml.utils.TMStreams.*;

import java.net.URI;

import org.gcube.common.core.faults.GCUBEException;
import org.gcube.common.core.faults.GCUBEUnrecoverableException;
import org.gcube.data.streams.Stream;
import org.gcube.data.streams.publishers.ThreadProvider;
import org.gcube.data.tm.context.ServiceContext;
import org.gcube.data.tm.stubs.Path;
import org.gcube.data.tmf.api.SourceReader;
import org.gcube.data.tml.exceptions.InvalidTreeException;
import org.gcube.data.tml.exceptions.UnknownPathException;
import org.gcube.data.tml.exceptions.UnknownTreeException;
import org.gcube.data.tml.exceptions.UnsupportedOperationException;
import org.gcube.data.tml.exceptions.UnsupportedRequestException;
import org.gcube.data.trees.data.Node;
import org.gcube.data.trees.data.Tree;
import org.gcube.data.trees.patterns.Pattern;
import org.globus.wsrf.ResourceException;

/**
 * A stateful resource of the T-Reader service.
 * 
 * @author Fabio Simeoni
 * @author Lucio Lelii (lucio.lelii@isti.cnr.it)
 */
public class TReaderResource extends AccessorResource {

	// little helper
	private SourceReader reader() throws GCUBEException {
		try {
			return getLocalResource().source().reader();
		} catch (ResourceException e) {
			throw new GCUBEUnrecoverableException(e);
		}
	}

	/**
	 * Returns a {@link Tree} in tha bound source that has a given identifier
	 * after it has been pruned with a given {@link Pattern}.
	 * 
	 * @param id
	 *            the identifier
	 * @param pattern
	 *            the {@link Pattern}
	 * @return the pruned {@link Tree}
	 * @throws UnsupportedOperationException
	 *             if the target plugin does not support this operation
	 * @throws UnsupportedRequestException
	 *             if the target plugin does not support this particular request
	 * @throws UnknownTreeException
	 *             if the bound source does not contain a tree with the given
	 *             identifier
	 * @throws InvalidTreeException
	 *             if the trees with the the given identifier does not match the
	 *             predicate
	 * @throws Exception
	 *             if the operation fails for an unexpected error
	 */
	public Tree get(String id, Pattern pattern)
			throws UnsupportedOperationException, UnsupportedRequestException,
			UnknownTreeException, InvalidTreeException, Exception {
		return reader().get(id, pattern);
	}

	public static final ThreadProvider SERVICETHREADPROVIDER = new ThreadProvider() {
		/** {@inheritDoc} */
		@Override
		public Thread newThread(Runnable r) {
			return ServiceContext.getContext().newServiceThread(r);
		}
	};

	/**
	 * Returns {@link Tree}s in the bound source that have given identifiers,
	 * after they have been pruned by a given {@link Pattern}.
	 * 
	 * @param idStream
	 *            the identifiers
	 * @param pattern
	 *            the pattern
	 * @return the pruned trees, relatively ordered as their identifiers
	 * @throws UnsupportedOperationException
	 *             if the target plugin does not support this operation
	 * @throws UnsupportedRequestException
	 *             if the target plugin does not support this particular request
	 * @throws Exception
	 *             if the operation fails for an unexpected error
	 */
	public Stream<Tree> get(Stream<String> idStream,
			Pattern pattern) throws UnsupportedOperationException,
			UnsupportedRequestException, Exception {
		return log(reader().get(idStream, pattern));
	}

	/**
	 * Returns all the {@link Tree}s in the bound source, after pruning them
	 * with a given {@link Pattern}.
	 * 
	 * @param pattern
	 *            the pattern
	 * @return the pruned trees
	 * @throws UnsupportedOperationException
	 *             if the target plugin does not support this operation
	 * @throws UnsupportedRequestException
	 *             if the target plugin does not support this particular request
	 * @throws Exception
	 *             if the operation fails for an unexpected error
	 */
	public Stream<Tree> get(Pattern pattern)
			throws UnsupportedOperationException, UnsupportedRequestException,
			Exception {
		return log(reader().get(pattern));
	}

	/**
	 * Returns a document node from the path of identifiers which connects it to
	 * the document root.
	 * 
	 * @param ids
	 *            the identifiers
	 * @return the node
	 * @throws UnsupportedOperationException
	 *             if the target plugin does not support this operation
	 * @throws UnsupportedRequestException
	 *             if the target plugin does not support this particular request
	 * @throws UnknownPathException
	 *             if the path does not lead to a node
	 * @throws Exception
	 *             if the operation fails for an unexpected error
	 */
	public Node getNode(String... ids) throws UnsupportedOperationException,
			UnsupportedRequestException, UnknownPathException, Exception {
		return reader().getNode(ids);
	}

	/**
	 * Returns document nodes identified by document {@link Path}s.
	 * 
	 * @param pathsRs
	 *            the locator of a result set of paths
	 * @return the locator to a result set of nodes, relatively ordered as the
	 *         corresponding paths
	 * @throws UnsupportedOperationException
	 *             if the target plugin does not support this operation
	 * @throws UnsupportedRequestException
	 *             if the target plugin does not support this particular request
	 * @throws Exception
	 *             if the operation fails for an unexpected error
	 */
	public Stream<Node> getNodes(URI pathsRs) throws UnsupportedOperationException,
			UnsupportedRequestException, Exception {

			Stream<Path> paths = pathsIn(pathsRs);

			return log(reader().getNodes(paths));
	}

}
