package org.gcube.data.analysis.statisticalmanager.dataspace;

import static org.gcube.data.streams.dsl.Streams.convert;
import static org.gcube.data.streams.dsl.Streams.pipe;
import gr.uoa.di.madgik.grs.record.GenericRecord;
import gr.uoa.di.madgik.grs.record.field.StringField;

import java.io.File;
import java.net.URI;
import java.rmi.RemoteException;
import java.util.List;

import org.gcube.common.core.faults.GCUBEFault;
import org.gcube.common.core.faults.GCUBEUnrecoverableException;
import org.gcube.common.core.porttypes.GCUBEPortType;
import org.gcube.common.core.types.VOID;
import org.gcube.data.analysis.statisticalmanager.SMOperationStatus;
import org.gcube.data.analysis.statisticalmanager.SMResourceType;
import org.gcube.data.analysis.statisticalmanager.ServiceContext;
import org.gcube.data.analysis.statisticalmanager.dataspace.exporter.CSVExporter;
import org.gcube.data.analysis.statisticalmanager.dataspace.importer.CSVImporter;
import org.gcube.data.analysis.statisticalmanager.dataspace.importer.OccurrenceStreamConverter;
import org.gcube.data.analysis.statisticalmanager.dataspace.importer.RSWrapper;
import org.gcube.data.analysis.statisticalmanager.persistence.DataBaseManager;
import org.gcube.data.analysis.statisticalmanager.persistence.HibernateManager;
import org.gcube.data.analysis.statisticalmanager.persistence.SMPersistenceManager;
import org.gcube.data.analysis.statisticalmanager.stubs.DataSpacePortType;
import org.gcube.data.analysis.statisticalmanager.stubs.SMCreateTableFromCSVRequest;
import org.gcube.data.analysis.statisticalmanager.stubs.SMCreateTableFromDataStreamRequest;
import org.gcube.data.analysis.statisticalmanager.stubs.SMCreatedTablesRequest;
import org.gcube.data.analysis.statisticalmanager.stubs.SMImporters;
import org.gcube.data.analysis.statisticalmanager.stubs.SMImportersRequest;
import org.gcube.data.analysis.statisticalmanager.stubs.SMTables;
import org.gcube.data.analysis.statisticalmanager.stubs.SMTablesMetadata;
import org.gcube.data.spd.plugin.fwk.model.OccurrencePoint;
import org.gcube.data.spd.plugin.fwk.model.binding.Bindings;
import org.gcube.data.streams.Stream;
import org.gcube.data.streams.generators.Generator;
import org.gcube.dataanalysis.ecoengine.datatypes.enumtypes.TableTemplates;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMImport;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMTable;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMTableMetadata;
import org.hibernate.Query;
import org.hibernate.Session;

public class DataSpace extends GCUBEPortType implements DataSpacePortType {

	@Override
	protected ServiceContext getServiceContext() {
		return ServiceContext.getContext();
	}

	@Override
	public String getDBParameters(String string) throws RemoteException,
			GCUBEFault {
		
		String url = DataBaseManager.getUrlDB() + 
		"?user="+ DataBaseManager.getUsername() +
		"&password=" + DataBaseManager.getPassword(); 

		return url;

	}

	@Override
	public SMImporters getImporters(SMImportersRequest request)
			throws RemoteException, GCUBEFault {

		Session session = HibernateManager.getSessionFactory().openSession();
		try {
			Query query = session
			.createQuery("select importer from SMImport  importer "
					+ "where importer.portalLogin like :name "
					+ "and importer.objectType like :template");
							
			query.setParameter("name",
					(request.getUser() != null) ? request.getUser() : "%");
			query.setParameter("template",
					(request.getObjectType() != null) ? request.getObjectType() : "%");

			@SuppressWarnings("unchecked")
			List<Object> objects = query.list();

			SMImport[] importers = objects
					.toArray(new SMImport[objects.size()]);
			return new SMImporters(importers);

		} finally {
			session.close();
		}
	}

	@Override
	public long createTableFromDataStream(
			final SMCreateTableFromDataStreamRequest request)
			throws RemoteException, GCUBEFault {

		try {
			final long importerId = SMPersistenceManager.addImporter(request);
			
			Stream<GenericRecord> genericRecords = convert(URI.create(request.getRsLocator())).of(GenericRecord.class).withDefaults();
			Generator<GenericRecord,OccurrencePoint> generator = new Generator<GenericRecord, OccurrencePoint>() {
				   @Override
				public OccurrencePoint yield(GenericRecord element) {
				     try {
						return Bindings.fromXml(((StringField)element.
									getField("result")).getPayload());
					} catch (Exception e) {
						return null;
					}
				   }
				};
			final Stream<OccurrencePoint> stream = pipe(genericRecords).through(generator);	
			
			
			Thread th = new Thread() {
				@Override
				public void run() {
					String resourceId = null;
					try {
						logger.debug("Init import from stream ");
						OccurrenceStreamConverter converter = new OccurrenceStreamConverter(
								DataBaseManager.getDataSource(),
								stream);
						converter.run();

						resourceId = converter.getTableName();

						SMTable table = new SMTable(request.getTableType());
						table.setResourceType(SMResourceType.TABULAR.ordinal());
						table.setResourceId(resourceId);
						table.setName(request.getTableName());
						table.setDescription(request.getDescription());
						
						SMPersistenceManager.addCreatedResource(
								importerId, table);

					} catch (Exception e) {
						logger.debug("Error import stream ",e);
						SMPersistenceManager.setOperationStatus(
								importerId, SMOperationStatus.FAILED);
					}
				}
			};
			th.start();

			return importerId;
		} catch (Exception e) {
			throw new GCUBEUnrecoverableException(e).toFault();
		}
	}

	@Override
	public long createTableFromCSV(final SMCreateTableFromCSVRequest request)
			throws RemoteException, GCUBEFault {

		try {
			final long importerId = SMPersistenceManager
					.addImporter(request);

			logger.debug("retrieve file ");
			logger.debug("Locator :" + request.getRsLocator());
			
			final File file = RSWrapper.getStreamFromLocator(new URI(
					request.getRsLocator()));
			
			logger.debug("File created " + file );
			
			Thread th = new Thread() {
				@Override
				public void run() {
					try {
						logger.debug("Init import ");
						CSVImporter converter = new CSVImporter(
								file, request.isHasHeader(),
								request.getTableName(), request.getTableType(),
								request.getDelimiter(),request.getCommentChar());
						String resourceId = converter.toTabularData();
						logger.debug("Import completed with resource id " + resourceId);

						SMTable table = new SMTable(request.getTableType());
						table.setResourceType(SMResourceType.TABULAR.ordinal());
						table.setResourceId(resourceId);
						table.setName(request.getTableName());
						table.setDescription(request.getDescription());
						
						SMPersistenceManager.addCreatedResource(
								importerId, table);

					} catch (Exception e) {
						logger.error("Import failed ",e);
						SMPersistenceManager.setOperationStatus(
								importerId, SMOperationStatus.FAILED);
					}
				}
			};
			th.start();

			return importerId;
		} catch (Exception e) {
			throw new GCUBEUnrecoverableException(e).toFault();
		}

	}

	@Override
	public SMTables getTables(SMCreatedTablesRequest request)
			throws RemoteException, GCUBEFault {

		Session session = HibernateManager.getSessionFactory().openSession();
		try {
			Query query = session
					.createQuery("select table from SMOperation  o "
							+ "join o.abstractResource.resource, SMTable table "
							+ "where o.abstractResource.abstractResourceId = table.resourceId "
							+ "and (o.portalLogin like :name or o.portalLogin = null) "
							+ "and table.template like :template");

			query.setParameter("name",
					(request.getUser() != null) ? request.getUser() : "%");
			String template = request.getTemplate();
			query.setParameter("template",
					((template != null) && !template.equals(TableTemplates.GENERIC)) ? template : "%");
			@SuppressWarnings("unchecked")
			List<Object> objects = query.list();

			SMTable[] tables = objects.toArray(new SMTable[objects.size()]);
			return new SMTables(tables);

		} finally {
			session.close();
		}

	}

	@Override
	public SMImport getImporter(String importerId) throws RemoteException,
			GCUBEFault {
		
		Session session = HibernateManager.getSessionFactory().openSession();
		try {
			Query query = session
					.createQuery("select importer from SMImport  importer "
							+ "where importer.operationId = :operationId");

			query.setParameter("operationId",Long.valueOf(importerId));

			@SuppressWarnings("unchecked")
			List<Object> objects = query.list();
			 
			return (SMImport)objects.get(0);

		} finally {
			session.close();
		}
	}

	@Override
	public VOID removeImporter(String operationId)
			throws RemoteException, GCUBEFault {
		
		SMPersistenceManager.removeOperation(Long.valueOf(operationId));
		return new VOID();
	}

	@Override
	public SMTablesMetadata getTablesMetadata(SMCreatedTablesRequest request)
			throws RemoteException, GCUBEFault {
		
		List<SMTableMetadata> tables = SMPersistenceManager.getTables(request.getUser(), request.getTemplate());
		return new SMTablesMetadata(tables.toArray(new SMTableMetadata[tables.size()]));
	}

	@Override
	public VOID checkImport(String importId) throws RemoteException, GCUBEFault {
		SMPersistenceManager.checkImport(Long.parseLong(importId));
		return new VOID();
	}

	@Override
	public SMImporters getUncheckedImports(SMImportersRequest request)
			throws RemoteException, GCUBEFault {
		SMImporters imports = SMPersistenceManager.getUncheckedImports(request.getUser());
		return imports;
	}

	@Override
	public VOID removeTable(String tableId) throws RemoteException, GCUBEFault {
		SMPersistenceManager.removeTable(tableId);
		return new VOID();
	}

	@Override
	public String exportTable(String tableId) throws RemoteException,
			GCUBEFault {
		
		try {
			File file = File.createTempFile("export", "service");
			CSVExporter exporter = new CSVExporter(tableId, file);
			exporter.exporterToFile();
			
			RSWrapper rs = new RSWrapper(ServiceContext.getContext().getScope());
			rs.add(file);
			
			return rs.getLocator().toString();
		} catch (Exception e) {
			logger.debug("Error in export ",e);
		}
		
		return null;
	}
}
