package org.gcube.data.analysis.tabulardata.operation.factories.types;

import java.util.List;
import org.gcube.data.analysis.tabulardata.operation.ImmutableOperationDescriptor;
import org.gcube.data.analysis.tabulardata.operation.OperationDescriptor;
import org.gcube.data.analysis.tabulardata.operation.OperationId;
import org.gcube.data.analysis.tabulardata.operation.OperationScope;
import org.gcube.data.analysis.tabulardata.operation.OperationType;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.parameters.LeafParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.Parameter;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerFactory;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.InvalidInvocationException;

abstract class BaseWorkerFactory implements WorkerFactory {

	protected abstract String getOperationName();

	protected abstract String getOperationDescription();

	protected OperationId getOperationId(){
		return new OperationId(this.getOperationName().hashCode());
	}

	protected abstract OperationScope getOperationScope();

	protected abstract OperationType getOperationType();
	
		
	@Override
	public OperationDescriptor getOperationDescriptor() {
		return new ImmutableOperationDescriptor(getOperationId(), getOperationName(), getOperationDescription(),
				getOperationScope(), getOperationType(), getParameters());
	}

	protected abstract List<Parameter> getParameters();

	/**
	 * Check the presence of a parameter and the type in an invocation.
	 * 
	 * @param id the id of the parameter to check
	 * @param type the type of the
	 * @param invocation the invocation where the parameter list is contained
	 * @throws InvalidInvocationException 
	 */
	@SuppressWarnings({ "unused", "unchecked" })
	private <T> void checkParameterValue(String id, Class<T> type, OperationInvocation invocation)
			throws InvalidInvocationException {
		Object obj = invocation.getParameterInstances().get(id);
		if (obj == null)
			throw new InvalidInvocationException(invocation, String.format(
					"Parameter %s is missing", id));
		T typedParameter = null;
		try {
			typedParameter = (T) obj;
		} catch (Exception e) {
			throw new InvalidInvocationException(invocation, e.getMessage());
		}
	}
	/**
	 * Check if the parameter definition matches the invocation
	 * @param parameter the parameter to check
	 * @param invocation the operation invocation
	 * @throws InvalidInvocationException
	 */
	private void checkParameter(LeafParameter<?> parameter, OperationInvocation invocation) throws InvalidInvocationException{
		checkParameterValue(parameter.getIdentifier(), parameter.getParameterType(), invocation);
	}
	
	/**
	 * Check for column id or table id presence according to the table scope and check presence and types of each required {@link LeafParameter}
	 * @param invocation the invocation to check
	 * @throws InvalidInvocationException an error reporting what is wrong
	 */
	protected void performBaseChecks(OperationInvocation invocation) throws InvalidInvocationException{
		switch(getOperationScope()){
		case COLUMN:
			checkColumnIdPresence(invocation);
		case TABLE:
			checkTableIdPresence(invocation);
		default:
			for (Parameter parameter : getParameters()) {
				if (parameter instanceof LeafParameter && parameter.getCardinality().getMinimum()>0)
				checkParameter((LeafParameter<?>)parameter, invocation);
			}
		}
	}
	
//	@SuppressWarnings("unchecked")
//	private <T> T getParameterValue(String id, Class<T> type, OperationInvocation invocation)
//			throws InvalidInvocationException {
//		checkParameterValue(id, type, invocation);
//		return (T) invocation.getParameterInstances().get(id);
//	}


	private void checkTableIdPresence(OperationInvocation invocation) throws InvalidInvocationException{
		if (invocation.getTargetTableId() == null ) throw new InvalidInvocationException(invocation, "Target table is missing");
	}

	private void checkColumnIdPresence(OperationInvocation invocation) throws InvalidInvocationException {
		if (invocation.getTargetColumnId() == null ) throw new InvalidInvocationException(invocation, "Target column is missing");
	}

}
