package org.gcube.data.analysis.tabulardata.operation.factories.types;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.gcube.data.analysis.tabulardata.expression.Expression;
import org.gcube.data.analysis.tabulardata.operation.ImmutableOperationDescriptor;
import org.gcube.data.analysis.tabulardata.operation.OperationDescriptor;
import org.gcube.data.analysis.tabulardata.operation.OperationId;
import org.gcube.data.analysis.tabulardata.operation.OperationScope;
import org.gcube.data.analysis.tabulardata.operation.OperationType;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.parameters.CompositeParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.LeafParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.Parameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.ExpressionParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.MultivaluedStringParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.RegexpStringParameter;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerFactory;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.InvalidInvocationException;

public abstract class BaseWorkerFactory implements WorkerFactory {

	protected abstract String getOperationName();

	protected abstract String getOperationDescription();

	protected OperationId getOperationId() {
		return new OperationId(this.getClass());
	}

	public List<WorkerFactory> getPrecoditionValidations(){
		return Collections.emptyList();
	}
	
	protected abstract OperationScope getOperationScope();

	protected abstract OperationType getOperationType();

	public OperationDescriptor getOperationDescriptor() {
		return new ImmutableOperationDescriptor(getOperationId(), getOperationName(), getOperationDescription(),
				getOperationScope(), getOperationType(), getParameters());
	}

	protected abstract List<Parameter> getParameters();

	/**
	 * Check the presence of a parameter and the type in an invocation.
	 * 
	 * @param id the id of the parameter to check
	 * @param type the type of the
	 * @param invocation the invocation where the parameter list is contained
	 * @throws InvalidInvocationException 
	 */
	@SuppressWarnings({ "unused", "unchecked" })
	private <T> void checkParameterValue(String id, Class<T> type, OperationInvocation invocation)
			throws InvalidInvocationException {
		Object obj = invocation.getParameterInstances().get(id);
		if (obj == null)
			throw new InvalidInvocationException(invocation, String.format(
					"Parameter %s is missing", id));
		T typedParameter = null;
		try {
			typedParameter = (T) obj;
		} catch (Exception e) {
			throw new InvalidInvocationException(invocation, e.getMessage());
		}
	}
	/**
	 * Check if the parameter definition matches the invocation
	 * @param parameter the parameter to check
	 * @param invocation the operation invocation
	 * @throws InvalidInvocationException
	 */
	private void checkParameter(LeafParameter<?> parameter, OperationInvocation invocation) throws InvalidInvocationException{
		checkParameterValue(parameter.getIdentifier(), parameter.getParameterType(), invocation);
	}
	
	/**
	 * Check Mandatory presence of parameter according to their cardinality. Checks also parameter instance classes and values
	 * 
	 * @param toCheckParams
	 * @param paramInstances
	 * @param invocation
	 * @throws InvalidInvocationException
	 */
	private void checkParameters(List<Parameter> toCheckParams,Map<String,Object> paramInstances,OperationInvocation invocation)throws InvalidInvocationException{
		for (Parameter parameter : toCheckParams){
			
			// check mandatory presence
			if(parameter.getCardinality().getMinimum()>0 && (!paramInstances.containsKey(parameter.getIdentifier())||paramInstances.get(parameter.getIdentifier())==null)) 
				throw new InvalidInvocationException(invocation,String.format("Parameter %s is missing or null",parameter.getIdentifier())); 
			
			// check mandatory Iterables (cardinality 2..N)
			if(parameter.getCardinality().getMinimum()>1 && !(paramInstances.get(parameter.getIdentifier()) instanceof Iterable))
				throw new InvalidInvocationException(invocation, String.format("Parameter %s must be multiple", parameter.getIdentifier()));			
			
			// check mandatory Single instance
			if(parameter.getCardinality().getMaximum()==1 && (paramInstances.get(parameter.getIdentifier()) instanceof Iterable))
				throw new InvalidInvocationException(invocation, String.format("Parameter %s cannot be multiple", parameter.getIdentifier()));
			
			//Check value
			Object value=paramInstances.get(parameter.getIdentifier());
			if(value!=null){
				if(value instanceof Iterable){
					for(Object obj:(Iterable)value){
						checkParameterInstance(obj,parameter,invocation);
						if(parameter instanceof CompositeParameter) 
							checkParameters(((CompositeParameter)parameter).getParameters(),(Map<String,Object>) obj,invocation);			
					}
				}else {
					checkParameterInstance(value,parameter,invocation);
					if(parameter instanceof CompositeParameter) 
						checkParameters(((CompositeParameter)parameter).getParameters(),(Map<String,Object>) value,invocation);			
				}
				
					
			}
		}
	}
	
	
	private void checkParameterInstance(Object obj, Parameter parameter, OperationInvocation invocation) throws InvalidInvocationException{
		if(parameter instanceof LeafParameter){
			LeafParameter<?> leaf=(LeafParameter<?>)parameter;
			//validate class
			if (!leaf.getParameterType().isAssignableFrom(obj.getClass())) 
					throw new InvalidInvocationException(invocation, String.format("Invalid %s parameter instance class. Found %s, expected %s ", parameter.getIdentifier(), obj.getClass(), ((LeafParameter<?>)parameter).getParameterType()));
			
			//validate value
			try{
			if(leaf instanceof ExpressionParameter) 
				if(!((ExpressionParameter)leaf).validate((Expression) obj)) {
					((Expression)obj).validate();
				}
			
			if(leaf instanceof MultivaluedStringParameter){
				MultivaluedStringParameter multi=(MultivaluedStringParameter)leaf;
				String param=(String)obj;
				if(!multi.validate(param)) {					
					throw new Exception(String.format("Passed argument %s is not among valid ones %s ",param,multi.getAdmittedValues()));
				}
			}
			
			if(leaf instanceof RegexpStringParameter) {
				RegexpStringParameter reg=((RegexpStringParameter)leaf);
				String param=(String)obj;
				if(!reg.validate(param)) {
					throw new Exception(String.format("Passed argument %s doesn't match regexp constraint %s",param,reg.getRegexp()));
				}
			}
			
			
			}catch(Exception e){
				throw new InvalidInvocationException(invocation,String.format("Parameter %s is invalid. Failure cause : %s ",parameter.getIdentifier(),e.getMessage()));
			}
			
		}else if(parameter instanceof CompositeParameter){
			try{
				Map<String,Object> map=(Map<String,Object>) obj;
			}catch(Exception e){				
				throw new InvalidInvocationException(invocation, String.format("Parameter %s must implement Map<String,Object>", parameter.getIdentifier()));
			}
		}
	}
	
	
	/**
	 * Check for column id or table id presence according to the table scope and check presence and types of each required {@link LeafParameter}
	 * @param invocation the invocation to check
	 * @throws InvalidInvocationException an error reporting what is wrong
	 */
	protected void performBaseChecks(OperationInvocation invocation) throws InvalidInvocationException{
		if(invocation==null) throw new InvalidInvocationException(invocation,"Operation invocation cannot be null");
		if(invocation.getParameterInstances()==null) throw new InvalidInvocationException(invocation, "Paramater map cannot be null");
		switch(getOperationScope()){
		case COLUMN:
			checkColumnIdPresence(invocation);
		case TABLE:
			checkTableIdPresence(invocation);
		default:
			
			checkParameters(getParameters(), invocation.getParameterInstances(), invocation);
			
//			for (Parameter parameter : getParameters()) {
//				if (parameter instanceof LeafParameter && parameter.getCardinality().getMinimum()>0)
//				checkParameter((LeafParameter<?>)parameter, invocation);
//			}
		}
	}
	
//	@SuppressWarnings("unchecked")
//	private <T> T getParameterValue(String id, Class<T> type, OperationInvocation invocation)
//			throws InvalidInvocationException {
//		checkParameterValue(id, type, invocation);
//		return (T) invocation.getParameterInstances().get(id);
//	}


	private void checkTableIdPresence(OperationInvocation invocation) throws InvalidInvocationException{
		if (invocation.getTargetTableId() == null ) throw new InvalidInvocationException(invocation, "Target table is missing");
	}

	private void checkColumnIdPresence(OperationInvocation invocation) throws InvalidInvocationException {
		if (invocation.getTargetColumnId() == null ) throw new InvalidInvocationException(invocation, "Target column is missing");
	}

}
