package org.gcube.data.analysis.tabulardata.operation.factories.types;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.exceptions.NoSuchTableException;
import org.gcube.data.analysis.tabulardata.expression.Expression;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnLocalId;
import org.gcube.data.analysis.tabulardata.model.column.ColumnReference;
import org.gcube.data.analysis.tabulardata.model.column.ColumnType;
import org.gcube.data.analysis.tabulardata.model.datatype.DataType;
import org.gcube.data.analysis.tabulardata.model.exceptions.NoSuchColumnException;
import org.gcube.data.analysis.tabulardata.model.metadata.common.ImmutableLocalizedText;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.table.TableId;
import org.gcube.data.analysis.tabulardata.operation.ImmutableOperationDescriptor;
import org.gcube.data.analysis.tabulardata.operation.OperationDescriptor;
import org.gcube.data.analysis.tabulardata.operation.OperationId;
import org.gcube.data.analysis.tabulardata.operation.OperationScope;
import org.gcube.data.analysis.tabulardata.operation.OperationType;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.parameters.CompositeParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.LeafParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.Parameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.ColumnTypeParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.DataTypeParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.ExpressionParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.LocalizedTextChoiceParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.MultivaluedStringParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.RegexpStringParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.TargetColumnParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.TargetTableParameter;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerFactory;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.InvalidInvocationException;

public abstract class BaseWorkerFactory implements WorkerFactory {

	protected abstract String getOperationName();

	protected abstract String getOperationDescription();

	protected OperationId getOperationId() {
		return new OperationId(this.getClass());
	}

	public List<WorkerFactory> getPrecoditionValidations(){
		return Collections.emptyList();
	}
	
	protected abstract OperationScope getOperationScope();

	protected abstract OperationType getOperationType();

	public OperationDescriptor getOperationDescriptor() {
		return new ImmutableOperationDescriptor(getOperationId(), getOperationName(), getOperationDescription(),
				getOperationScope(), getOperationType(), getParameters());
	}

	protected abstract List<Parameter> getParameters();

	
	/**
	 * Check Mandatory presence of parameter according to their cardinality. Checks also parameter instance classes and values
	 * 
	 * @param toCheckParams
	 * @param paramInstances
	 * @param invocation
	 * @throws InvalidInvocationException
	 */
	private void checkParameters(List<Parameter> toCheckParams,Map<String,Object> paramInstances,OperationInvocation invocation,CubeManager cubeManager)throws InvalidInvocationException{
		for (Parameter parameter : toCheckParams){
			
			if(paramInstances.containsKey(parameter.getIdentifier())&&paramInstances.get(parameter.getIdentifier())==null)
					throw new InvalidInvocationException(invocation,String.format("Parameter %s is null",parameter.getIdentifier())); 
			// check mandatory presence
			if(parameter.getCardinality().getMinimum()>0 && (!paramInstances.containsKey(parameter.getIdentifier()))) 
				throw new InvalidInvocationException(invocation,String.format("Parameter %s is missing",parameter.getIdentifier())); 
			
			// check mandatory Iterables (cardinality 2..N)
			if(parameter.getCardinality().getMinimum()>1 && !(paramInstances.get(parameter.getIdentifier()) instanceof Iterable))
				throw new InvalidInvocationException(invocation, String.format("Parameter %s must be multiple", parameter.getIdentifier()));			
			
			// check mandatory Single instance
			if(parameter.getCardinality().getMaximum()==1 && (paramInstances.get(parameter.getIdentifier()) instanceof Iterable))
				throw new InvalidInvocationException(invocation, String.format("Parameter %s cannot be multiple", parameter.getIdentifier()));
			
			//Check value
			Object value=paramInstances.get(parameter.getIdentifier());
			if(value!=null){
				if(value instanceof Iterable){
					for(Object obj:(Iterable)value){
						checkParameterInstance(obj,parameter,invocation,cubeManager);
						if(parameter instanceof CompositeParameter) 
							checkParameters(((CompositeParameter)parameter).getParameters(),(Map<String,Object>) obj,invocation,cubeManager);			
					}
				}else {
					checkParameterInstance(value,parameter,invocation,cubeManager);
					if(parameter instanceof CompositeParameter) 
						checkParameters(((CompositeParameter)parameter).getParameters(),(Map<String,Object>) value,invocation,cubeManager);			
				}
				
					
			}
		}
	}
	
	
	private void checkParameterInstance(Object obj, Parameter parameter, OperationInvocation invocation,CubeManager cubeManager) throws InvalidInvocationException{
		if(parameter instanceof LeafParameter){
			LeafParameter<?> leaf=(LeafParameter<?>)parameter;
			//validate class
			if (!leaf.getParameterType().isAssignableFrom(obj.getClass())) 
					throw new InvalidInvocationException(invocation, String.format("Invalid %s parameter instance class. Found %s, expected %s ", parameter.getIdentifier(), obj.getClass(), ((LeafParameter<?>)parameter).getParameterType()));
			
			//validate value
			try{
			if(leaf instanceof ExpressionParameter) 
				if(!((ExpressionParameter)leaf).validate((Expression) obj)) {
					((Expression)obj).validate();
				}
			
			if(leaf instanceof MultivaluedStringParameter){
				MultivaluedStringParameter multi=(MultivaluedStringParameter)leaf;
				String param=(String)obj;
				if(!multi.validate(param)) {					
					throw new Exception(String.format("Passed argument %s is not among valid ones %s ",param,multi.getAdmittedValues()));
				}
			}
			
			if(leaf instanceof RegexpStringParameter) {
				RegexpStringParameter reg=((RegexpStringParameter)leaf);
				String param=(String)obj;
				if(!reg.validate(param)) {
					throw new Exception(String.format("Passed argument %s doesn't match regexp constraint %s",param,reg.getRegexp()));
				}
			}
			
			if(leaf instanceof TargetColumnParameter){
				ColumnReference ref=(ColumnReference) obj;
				TargetColumnParameter param=(TargetColumnParameter) leaf;
				Table table=getTable(invocation, ref.getTableId(), cubeManager);			
				if(!param.getAllowedTableTypes().contains(table.getTableType())) throw new Exception(String.format("Invalid table type %s, allowed types are : %s.",table.getTableType().getName(),param.getAllowedTableTypes()));
				Column col=getColumn(invocation, ref.getColumnId(), table, cubeManager);
				if(!param.getAllowedColumnTypes().contains(col.getColumnType())) throw new Exception(String.format("Invalid column type %s, allowed types are : %s.",col.getColumnType().getName(),param.getAllowedColumnTypes()));				
			}
			
			if(leaf instanceof TargetTableParameter){
				TableId tableId=(TableId) obj;
				TargetTableParameter param=(TargetTableParameter) leaf;
				Table table=getTable(invocation, tableId, cubeManager);			
				if(!param.getAllowedTableTypes().contains(table.getTableType())) throw new Exception(String.format("Invalid table type %s, allowed types are : %s.",table.getTableType().getName(),param.getAllowedTableTypes()));
			}
			
			if(leaf instanceof LocalizedTextChoiceParameter){
				ImmutableLocalizedText text=(ImmutableLocalizedText) obj;
				LocalizedTextChoiceParameter param=(LocalizedTextChoiceParameter) leaf;
				if(!param.getLabelChoices().contains(text))throw new Exception(String.format("Passed argument %s is not among valid ones %s ",text,param.getLabelChoices()));
			}
			
			
			if(leaf instanceof ColumnTypeParameter){
				ColumnType type=(ColumnType)obj;
				ColumnTypeParameter param=(ColumnTypeParameter) leaf;
				if(!param.getAllowedColumnTypes().contains(type))throw new Exception(String.format("Passed argument %s is not among valid ones %s ",type,param.getAllowedColumnTypes()));
			}
			
			if(leaf instanceof DataTypeParameter){
				DataType type=(DataType) obj;
				DataTypeParameter param=(DataTypeParameter) leaf;
				if(!param.getAllowedDataTypes().contains(type)) throw new Exception(String.format("Passed argument %s is not among valid ones %s ",type,param.getAllowedDataTypes()));
			}
			
			}catch(Exception e){
				throw new InvalidInvocationException(invocation,String.format("Parameter %s is invalid. Failure cause : %s ",parameter.getIdentifier(),e.getMessage()));
			}
			
		}else if(parameter instanceof CompositeParameter){
			try{
				Map<String,Object> map=(Map<String,Object>) obj;
			}catch(Exception e){				
				throw new InvalidInvocationException(invocation, String.format("Parameter %s must implement Map<String,Object>", parameter.getIdentifier()));
			}
		}
	}
	
	
	/**
	 * Check for column id or table id presence according to the table scope and check presence and types of each required {@link LeafParameter}
	 * @param invocation the invocation to check
	 * @throws InvalidInvocationException an error reporting what is wrong
	 */
	protected void performBaseChecks(OperationInvocation invocation,CubeManager cubeManager) throws InvalidInvocationException{
		if(invocation==null) throw new InvalidInvocationException(invocation,"Operation invocation cannot be null");
		if(invocation.getParameterInstances()==null) throw new InvalidInvocationException(invocation, "Paramater map cannot be null");		
		switch(getOperationScope()){
		case COLUMN:
			Table table=getTable(invocation, invocation.getTargetTableId(), cubeManager);
			getColumn(invocation,invocation.getTargetColumnId(),table,cubeManager);
			break;
		case TABLE:
			getTable(invocation, invocation.getTargetTableId(), cubeManager);
			break;
		}
		checkParameters(getParameters(), invocation.getParameterInstances(), invocation, cubeManager);
	}
	
	
	private Column getColumn(OperationInvocation invocation,ColumnLocalId colId,Table table,CubeManager cubeManager) throws InvalidInvocationException{
		if(colId==null) throw new InvalidInvocationException(invocation, "Column id cannot be null");		
		try{
			return table.getColumnById(colId);
		}catch(NoSuchColumnException e){
			throw new InvalidInvocationException(invocation, e.getMessage());
		}
	}
	
	private Table getTable(OperationInvocation invocation,TableId tableId,CubeManager cubeManager)throws InvalidInvocationException{
		if(tableId==null) throw new InvalidInvocationException(invocation, "Table id cannot be null");
		try{
			return cubeManager.getTable(tableId);
		}catch(NoSuchTableException e){
			throw new InvalidInvocationException(invocation, e.getMessage());
		}
	}
	
	
}
