package org.gcube.data.analysis.tabulardata.operation.csv.exporter;

import static org.gcube.data.analysis.tabulardata.operation.csv.Constants.ENCODING;
import static org.gcube.data.analysis.tabulardata.operation.csv.Constants.SEPARATOR;

import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.util.Date;
import java.util.Locale;
import java.util.Map;

import org.gcube.contentmanagement.blobstorage.service.IClient;
import org.gcube.contentmanager.storageclient.wrapper.AccessType;
import org.gcube.contentmanager.storageclient.wrapper.MemoryType;
import org.gcube.contentmanager.storageclient.wrapper.StorageClient;
import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.TableMetaCreator;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.metadata.common.NamesMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.table.ExportMetadata;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.csv.Constants;
import org.gcube.data.analysis.tabulardata.operation.worker.BaseWorker;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.OperationException;
import org.postgresql.PGConnection;
import org.postgresql.copy.CopyManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CSVExport extends BaseWorker {
	
	private static Logger logger = LoggerFactory.getLogger(CSVExport.class);
	
	CubeManager cubeManager;
	
	DatabaseConnectionProvider connectionProvider;
	
	private String encoding;
	private Character separator;
	
	public CSVExport(OperationInvocation invocation, CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider) {
		super(invocation);
		retrieveParameters();
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
	}

	public void run() {
		inProgress(0f);
		Table table = cubeManager.getTable(invocation.getTargetTableId());
		inProgress(0.1f);
		File exportFile;
		try{
			exportFile = File.createTempFile("export", ".csv");
		}catch (Exception e) {
			logger.error("error creating file", e);
			fail(new OperationException("error creating file", e));
			return;
		}
		inProgress(0.3f);
		try{
			OutputStreamWriter outputStreamWriter = new OutputStreamWriter(new FileOutputStream(exportFile));
			copy(outputStreamWriter, table, invocation.getParameterInstances());
		}catch (Exception e) {
			logger.error("error copying table to file", e);
			exportFile.delete();
			fail(new OperationException("error copying table to file", e));
			return;
		}
		inProgress(0.7f);
		String url;
		try{
			url = storeFile(exportFile);
		}catch (Exception e) {
			logger.error("error storing file", e);
			exportFile.delete();
			fail(new OperationException("error storing file", e));
			return;
		}
		inProgress(0.9f);
		Table returnTable = createMetaTable(table, url);
		succeed(returnTable);
	}

	private String storeFile(File exportFile){
		IClient client = new StorageClient(Constants.SERVICE_CLASS, Constants.SERVICE_NAME, "CSVExport", AccessType.SHARED, MemoryType.VOLATILE).getClient();
		String remotePath = "/CSVexport/"+exportFile.getName();
		client.put(true).LFile(exportFile.getAbsolutePath()).RFile(remotePath);
		return  client.getUrl().RFile(remotePath);
	}

	private Table createMetaTable(Table table, String url) {
		TableMetaCreator tmc = cubeManager.modifyTableMeta(table.getId());
		tmc.setTableMetadata(new ExportMetadata("CSV", url, new Date()));
		return tmc.create();
	}
	
	private long copy(OutputStreamWriter outputStreamWriter, Table table, Map<String, Object> parameters) throws Exception {
		PGConnection conn = connectionProvider.getPostgreSQLConnection();
		CopyManager cpManager = conn.getCopyAPI();
		StringBuilder columns = new StringBuilder("SELECT ");
		for (Column c : table.getColumns()) {
			if(parameters.containsKey(c.getName()) && (Boolean)parameters.get(c.getName())){
				String label = c.getName();
				try{
					NamesMetadata metadata = ((NamesMetadata)c.getMetadata(NamesMetadata.class));
					if (metadata.hasTextWithLocale(Locale.ENGLISH))
						label = metadata.getTextWithLocale(Locale.ENGLISH).getValue();
				}catch (Exception e) {}
				columns.append(c.getName()).append(" as ").append(String.format("\"%s\"", label));
				columns.append(",");
			}
		}
		columns.deleteCharAt(columns.length() - 1);
		columns.append(" from "+table.getName());
		
		String sqlCmd = String.format("COPY ( %s ) TO STDOUT ( FORMAT CSV ,DELIMITER '%c', HEADER %b, ENCODING '%s');",
				columns.toString(), separator, true, encoding);
		logger.info("executing copy for csv import with query {}",sqlCmd);
		
		return cpManager.copyOut(sqlCmd, outputStreamWriter);
	}
	
	private void retrieveParameters() {
		Map<String, Object> parameters = invocation.getParameterInstances();
		separator = (Character) parameters.get(SEPARATOR);
		encoding = (String) parameters.get(ENCODING);
	}
}
